// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_PREVIEWS_CONTENT_PREVIEWS_DECIDER_IMPL_H_
#define COMPONENTS_PREVIEWS_CONTENT_PREVIEWS_DECIDER_IMPL_H_

#include <stdint.h>

#include <memory>
#include <string>
#include <vector>

#include "base/callback.h"
#include "base/macros.h"
#include "base/memory/ref_counted.h"
#include "base/memory/weak_ptr.h"
#include "base/sequence_checker.h"
#include "base/single_thread_task_runner.h"
#include "base/time/time.h"
#include "components/blocklist/opt_out_blocklist/opt_out_blocklist_data.h"
#include "components/blocklist/opt_out_blocklist/opt_out_blocklist_delegate.h"
#include "components/previews/content/previews_decider.h"
#include "components/previews/content/previews_optimization_guide.h"
#include "components/previews/core/previews_block_list.h"
#include "components/previews/core/previews_experiments.h"
#include "components/previews/core/previews_logger.h"
#include "net/nqe/effective_connection_type.h"
#include "url/gurl.h"

namespace base {
class Clock;
}

namespace blocklist {
class OptOutStore;
}

namespace content {
class NavigationHandle;
}

namespace previews {
class PreviewsUIService;

typedef base::RepeatingCallback<bool(PreviewsType)> PreviewsIsEnabledCallback;

// A class to manage the IO portion of inter-thread communication between
// previews/ objects. Created on the UI thread, but used only on the IO thread
// after initialization.
class PreviewsDeciderImpl : public PreviewsDecider,
                            public blocklist::OptOutBlocklistDelegate {
 public:
  explicit PreviewsDeciderImpl(base::Clock* clock);
  ~PreviewsDeciderImpl() override;

  // blocklist::OptOutBlocklistDelegate:
  void OnNewBlocklistedHost(const std::string& host, base::Time time) override;
  void OnUserBlocklistedStatusChange(bool blocklisted) override;
  void OnBlocklistCleared(base::Time time) override;

  // Initializes the blocklist and and stores the passed in members.
  // |previews_ui_service| owns |this|, and shares the same lifetime.
  virtual void Initialize(
      PreviewsUIService* previews_ui_service,
      std::unique_ptr<blocklist::OptOutStore> previews_opt_out_store,
      std::unique_ptr<PreviewsOptimizationGuide> previews_opt_guide,
      const PreviewsIsEnabledCallback& is_enabled_callback,
      blocklist::BlocklistData::AllowedTypesAndVersions allowed_previews);

  // Adds log message of the navigation asynchronously.
  void LogPreviewNavigation(const GURL& url,
                            bool opt_out,
                            PreviewsType type,
                            base::Time time,
                            uint64_t page_id) const;

  // Adds a log message for the preview decision made (|reason|) asynchronously.
  // |passed_reasons| is a collection of reason codes that correspond to
  // eligibility checks that were satisfied prior to determining |reason| and
  // so the opposite of these |passed_reasons| codes was true. The method
  // takes ownership of |passed_reasons|. |page_id| is generated by
  // PreviewsDeciderImpl, and used to group decisions into groups on the page,
  // messages that don't need to be grouped can pass in 0 as page_id.
  void LogPreviewDecisionMade(
      PreviewsEligibilityReason reason,
      const GURL& url,
      base::Time time,
      PreviewsType type,
      std::vector<PreviewsEligibilityReason>&& passed_reasons,
      PreviewsUserData* user_data) const;

  // Adds a navigation to |url| to the block list with result |opt_out|.
  void AddPreviewNavigation(const GURL& url,
                            bool opt_out,
                            PreviewsType type,
                            uint64_t page_id);

  // Clears the history of the block list between |begin_time| and |end_time|,
  // both inclusive. Additional, clears the appropriate data from the hint
  // cache. TODO(mcrouse): Rename to denote clearing all necessary data,
  // including the Fetched hints and the blocklist.
  void ClearBlockList(base::Time begin_time, base::Time end_time);

  // Change the status of whether to ignore the decisions made by
  // PreviewsBlockList to |ignored|. Virtualized in testing.
  virtual void SetIgnorePreviewsBlocklistDecision(bool ignored);

  // The previews block list that decides whether a navigation can use previews.
  PreviewsBlockList* block_list() const { return previews_block_list_.get(); }

  // PreviewsDecider implementation:
  bool ShouldAllowPreviewAtNavigationStart(
      PreviewsUserData* previews_data,
      content::NavigationHandle* navigation_handle,
      bool is_reload,
      PreviewsType type) const override;
  bool ShouldCommitPreview(PreviewsUserData* previews_data,
                           content::NavigationHandle* navigation_handle,
                           PreviewsType type) const override;

  // Generates a page ID that is guaranteed to be unique from any other page ID
  // generated in this browser session. Also, guaranteed to be non-zero.
  uint64_t GeneratePageId();

  void SetEffectiveConnectionType(
      net::EffectiveConnectionType effective_connection_type);

  PreviewsOptimizationGuide* previews_opt_guide() const {
    return previews_opt_guide_.get();
  }

  // When a preview is reloaded, this is called. No Previews are allowed for
  // params::SingleOptOutDuration after that reload is reported.
  void AddPreviewReload();

 protected:
  // Sets a blocklist for testing.
  void SetPreviewsBlocklistForTesting(
      std::unique_ptr<PreviewsBlockList> previews_back_list);

 private:
  // Returns whether the preview |type| should be considered for |url|.
  // This is an initial check on the preview |type| being enabled and the
  // |url| not being a local URL.
  bool ShouldConsiderPreview(PreviewsType type,
                             const GURL& url,
                             PreviewsUserData* previews_data) const;

  // Determines the eligibility of the preview |type| for |url|.
  PreviewsEligibilityReason DeterminePreviewEligibility(
      PreviewsUserData* previews_data,
      content::NavigationHandle* navigation_handle,
      bool is_reload,
      PreviewsType type,
      std::vector<PreviewsEligibilityReason>* passed_reasons) const;

  // Returns previews eligibility with respect to the local blocklist.
  PreviewsEligibilityReason CheckLocalBlocklist(
      const GURL& url,
      PreviewsType type,
      std::vector<PreviewsEligibilityReason>* passed_reasons) const;

  // Whether the preview |type| should be allowed to be considered for |url|
  // subject to any server provided optimization hints. This is meant for
  // checking the initial navigation URL. Returns ALLOWED if no reason found
  // to deny the preview for consideration.
  PreviewsEligibilityReason ShouldAllowPreviewPerOptimizationHints(
      PreviewsUserData* previews_data,
      content::NavigationHandle* navigation_handle,
      PreviewsType type,
      std::vector<PreviewsEligibilityReason>* passed_reasons) const;

  // Whether |url| is allowed for |type| according to server provided
  // optimization hints, if available. This is meant for checking the committed
  // navigation URL against any specific hint details.
  PreviewsEligibilityReason ShouldCommitPreviewPerOptimizationHints(
      PreviewsUserData* previews_data,
      content::NavigationHandle* navigation_handle,
      PreviewsType type,
      std::vector<PreviewsEligibilityReason>* passed_reasons) const;

  // The UI service object owns |this| and exists as long as |this| does.
  PreviewsUIService* previews_ui_service_;

  std::unique_ptr<PreviewsBlockList> previews_block_list_;

  // Holds optimization guidance from the server.
  std::unique_ptr<PreviewsOptimizationGuide> previews_opt_guide_;

  // Whether the decisions made by PreviewsBlockList should be ignored or not.
  // This can be changed by chrome://interventions-internals to test/debug the
  // behavior of Previews decisions.
  // This is related to a test flag and should only be true when the user has
  // set it in flags. See previews::IsPreviewsBlocklistIgnoredViaFlag.
  bool blocklist_ignored_;

  // The estimate of how slow a user's connection is. Used for triggering
  // Previews.
  net::EffectiveConnectionType effective_connection_type_ =
      net::EffectiveConnectionType::EFFECTIVE_CONNECTION_TYPE_UNKNOWN;

  base::Clock* clock_;

  base::Optional<base::Time> recent_preview_reload_time_;

  // Whether the preview is enabled. Valid after Initialize() is called.
  PreviewsIsEnabledCallback is_enabled_callback_;

  uint64_t page_id_;

  SEQUENCE_CHECKER(sequence_checker_);

  base::WeakPtrFactory<PreviewsDeciderImpl> weak_factory_{this};

  DISALLOW_COPY_AND_ASSIGN(PreviewsDeciderImpl);
};

}  // namespace previews

#endif  // COMPONENTS_PREVIEWS_CONTENT_PREVIEWS_DECIDER_IMPL_H_
