/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
package jalview.util;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class FileUtils
{
  /*
   * Given string glob pattern (see
   * https://docs.oracle.com/javase/7/docs/api/java/nio/file/FileSystem.html#getPathMatcher(java.lang.String)
   * ) return a List of Files that match the pattern.
   * Note this is a java style glob, not necessarily a bash-style glob, though there are sufficient similarities. 
   */
  public static List<File> getFilesFromGlob(String pattern)
  {
    return getFilesFromGlob(pattern, true);
  }

  public static List<File> getFilesFromGlob(String pattern,
          boolean allowSingleFilenameThatDoesNotExist)
  {
    pattern = substituteHomeDir(pattern);
    String relativePattern = pattern.startsWith(File.separator) ? null
            : pattern;
    List<File> files = new ArrayList<>();
    /*
     * For efficiency of the Files.walkFileTree(), let's find the longest path that doesn't need globbing.
     * We look for the first glob character * { ? and then look for the last File.separator before that.
     * Then we can reset the path to look at and shorten the globbing pattern.
     * Relative paths can be used in pattern, which work from the pwd (though these are converted into
     * full paths in the match). 
     */
    int firstGlobChar = -1;
    boolean foundGlobChar = false;
    for (char c : new char[] { '*', '{', '?' })
    {
      if (pattern.indexOf(c) > -1
              && (pattern.indexOf(c) < firstGlobChar || !foundGlobChar))
      {
        firstGlobChar = pattern.indexOf(c);
        foundGlobChar = true;
      }
    }
    int lastFS = pattern.lastIndexOf(File.separatorChar, firstGlobChar);
    if (foundGlobChar)
    {
      String pS = pattern.substring(0, lastFS + 1);
      String rest = pattern.substring(lastFS + 1);
      if ("".equals(pS))
      {
        pS = ".";
      }
      Path parentDir = Paths.get(pS);
      if (parentDir.toFile().exists())
      {
        try
        {
          String glob = "glob:" + parentDir.toString() + File.separator
                  + rest;
          // don't use Platform.isWin() as this class is used in getdown
          if (System.getProperty("os.name").indexOf("Win") >= 0)
          {
            // escape "\\" on Windows
            // This ultimately replaces "\\" == '\' with "\\\\" = '\\' to escape
            // backslashes
            glob = glob.replaceAll("\\\\", "\\\\\\\\");
          }
          PathMatcher pm = FileSystems.getDefault().getPathMatcher(glob);
          int maxDepth = rest.contains("**") ? 1028
                  : (int) (rest.chars()
                          .filter(ch -> ch == File.separatorChar).count())
                          + 1;

          Files.walkFileTree(parentDir,
                  EnumSet.of(FileVisitOption.FOLLOW_LINKS), maxDepth,
                  new SimpleFileVisitor<Path>()
                  {
                    @Override
                    public FileVisitResult visitFile(Path path,
                            BasicFileAttributes attrs) throws IOException
                    {
                      if (pm.matches(path))
                      {
                        files.add(path.toFile());
                      }
                      return FileVisitResult.CONTINUE;
                    }

                    @Override
                    public FileVisitResult visitFileFailed(Path file,
                            IOException exc) throws IOException
                    {
                      return FileVisitResult.CONTINUE;
                    }
                  });
        } catch (IOException e)
        {
          e.printStackTrace();
        }
      }
    }
    else
    {
      // no wildcards
      File f = new File(pattern);
      if (allowSingleFilenameThatDoesNotExist || f.exists())
      {
        files.add(f);
      }
    }
    Collections.sort(files);

    return files;
  }

  public static List<String> getFilenamesFromGlob(String pattern)
  {
    // convert list of Files to list of File.getPath() Strings
    return getFilesFromGlob(pattern).stream().map(f -> f.getPath())
            .collect(Collectors.toList());
  }

  public static String substituteHomeDir(String path)
  {
    return path.startsWith("~" + File.separator)
            ? System.getProperty("user.home") + path.substring(1)
            : path;
  }

  /*
   * This method returns the basename of File file
   */
  public static String getBasename(File file)
  {
    return getBasenameOrExtension(file, false);
  }

  /*
   * This method returns the extension of File file.
   */
  public static String getExtension(File file)
  {
    return getBasenameOrExtension(file, true);
  }

  public static String getBasenameOrExtension(File file, boolean extension)
  {
    if (file == null)
      return null;

    String value = null;
    String filename = file.getName();
    int lastDot = filename.lastIndexOf('.');
    if (lastDot > 0) // don't truncate if starts with '.'
    {
      value = extension ? filename.substring(lastDot + 1)
              : filename.substring(0, lastDot);
    }
    else
    {
      value = extension ? "" : filename;
    }
    return value;
  }

  /*
   * This method returns the dirname of the first --append or --open value. 
   * Used primarily for substitutions in output filenames.
   */
  public static String getDirname(File file)
  {
    if (file == null)
      return null;

    String dirname = null;
    File p = file.getParentFile();
    if (p == null)
    {
      p = new File(".");
    }
    File d = new File(substituteHomeDir(p.getPath()));
    dirname = d.getPath();
    return dirname;
  }

  public static String convertWildcardsToPath(String value, String wildcard,
          String dirname, String basename)
  {
    if (value == null)
    {
      return null;
    }
    StringBuilder path = new StringBuilder();
    int lastFileSeparatorIndex = value.lastIndexOf(File.separatorChar);
    int wildcardBeforeIndex = value.indexOf(wildcard);
    if (lastFileSeparatorIndex > wildcard.length() - 1
            && wildcardBeforeIndex < lastFileSeparatorIndex)
    {
      path.append(value.substring(0, wildcardBeforeIndex));
      path.append(dirname);
      path.append(value.substring(wildcardBeforeIndex + wildcard.length(),
              lastFileSeparatorIndex + 1));
    }
    else
    {
      path.append(value.substring(0, lastFileSeparatorIndex + 1));
    }
    int wildcardAfterIndex = value.indexOf(wildcard,
            lastFileSeparatorIndex);
    if (wildcardAfterIndex > lastFileSeparatorIndex)
    {
      path.append(value.substring(lastFileSeparatorIndex + 1,
              wildcardAfterIndex));
      path.append(basename);
      path.append(value.substring(wildcardAfterIndex + wildcard.length()));
    }
    else
    {
      path.append(value.substring(lastFileSeparatorIndex + 1));
    }
    return path.toString();
  }

  public static File getParentDir(File file)
  {
    if (file == null)
    {
      return null;
    }
    File parentDir = file.getAbsoluteFile().getParentFile();
    return parentDir;
  }

  public static boolean checkParentDir(File file, boolean mkdirs)
  {
    if (file == null)
    {
      return false;
    }
    File parentDir = getParentDir(file);
    if (parentDir.exists())
    {
      // already exists, nothing to do so nothing to worry about!
      return true;
    }

    if (!mkdirs)
    {
      return false;
    }

    Path path = file.toPath();
    for (int i = 0; i < path.getNameCount(); i++)
    {
      Path p = path.getName(i);
      if ("..".equals(p.toString()))
      {
        LaunchUtils.syserr(true, false,
                "Cautiously not running mkdirs on " + file.toString()
                        + " because the path to be made contains '..'");
        return false;
      }
    }

    return mkdirs(parentDir);
  }

  /**
   * get a guessed file extension from a String only
   * 
   * @param String
   *          filename
   * @return String extension
   */
  public static String getExtension(String filename)
  {
    return getBaseOrExtension(filename, true);
  }

  /**
   * getBase returns everything in a path/URI up to (and including) an extension
   * dot. Note this is not the same as getBasename() since getBasename() only
   * gives the filename base, not the path too. If no extension dot is found
   * (i.e. a dot in character position 2 or more of the filename (after the last
   * slash) then the whole path is considered the base.
   * 
   * @param filename
   * @return String base
   */
  public static String getBase(String filename)
  {
    return getBaseOrExtension(filename, false);
  }

  public static String getBaseOrExtension(String filename0,
          boolean extension)
  {
    if (filename0 == null)
    {
      return null;
    }
    String filename = filename0;
    boolean isUrl = false;
    if (HttpUtils.startsWithHttpOrHttps(filename))
    {
      try
      {
        URL url = new URL(filename);
        filename = url.getPath();
        isUrl = true;
      } catch (MalformedURLException e)
      {
        // continue to treat as a filename
      }
    }
    int dot = filename.lastIndexOf('.');
    int slash = filename.lastIndexOf('/');
    if (!File.separator.equals("/") && !isUrl)
    {
      slash = filename.lastIndexOf(File.separator);
    }
    // only the dot of the filename (not dots in path) and not if it's a .hidden
    // file
    boolean hasExtension = dot > slash + 1;
    if (extension)
    {
      return hasExtension ? filename.substring(dot + 1) : null;
    }
    else
    {
      dot = filename0.lastIndexOf('.');
      return hasExtension ? filename0.substring(0, dot + 1) : filename0;
    }
  }

  public static Path getCanonicalPath(Path path)
  {
    return path.normalize();
  }

  public static Path getCanonicalPath(File file)
  {
    return getCanonicalPath(file.toPath());
  }

  public static Path getCanonicalPath(String pathString)
  {
    return getCanonicalPath(Paths.get(pathString));
  }

  public static File getCanonicalFile(File file)
  {
    return getCanonicalPath(file).toFile();
  }

  public static File getCanonicalFile(String pathString)
  {
    return getCanonicalPath(Paths.get(pathString)).toFile();
  }

  public static boolean mkdirs(File file)
  {
    try
    {
      Files.createDirectories(getCanonicalPath(file));
      return file.exists();
    } catch (IOException e)
    {
      LaunchUtils.syserr(true, false, "Failed to make directory " + file
              + "\n" + e.getStackTrace());
    }
    return false;
  }

  /**
   * Look for files that use a template, with two "%s" formatting entries, to
   * look for files with the template substituted with combinations of root and
   * version. If versionWhitelist is not null then these paths will be added. If
   * versionBlacklist is not null then globbed versions will be looked for and
   * these versions excluded. If both are given then both will be included. If
   * separator is not null then it will be added before the version number, and
   * additionally a path without the separator and version will be looked for or
   * added if the whitelist or blacklist are supplied respectively.
   * 
   * @param templates
   * @param roots
   * @param versionWhitelist
   * @param versionBlacklist
   * @param separator
   * @return
   */
  public static List<File> getMatchingVersionedFiles(String[] templates,
          String[] roots, String[] versionWhitelist,
          String[] versionBlacklist, String versionSeparator,
          boolean exists)
  {
    Set<File> matchingFiles = new HashSet<>();
    if (templates == null)
    {
      ErrorLog.errPrintln(
              "getMatchingVersionedFiles called with a null template array");
      List<File> files = new ArrayList<File>();
      files.addAll(matchingFiles);
      return files;
    }

    for (String template : templates)
    {
      ErrorLog.errPrintln("Using template '" + template + "'");
      for (String root : roots)
      {
        ErrorLog.errPrintln("Using root '" + root + "'");

        // Blacklist. Use a file glob for version and see what's there
        if (versionBlacklist != null)
        {
          String globMatch = String.format(template, root, "*");
          ErrorLog.errPrintln("Using glob '" + globMatch + "'");
          List<File> foundFiles = FileUtils.getFilesFromGlob(globMatch,
                  false);
          for (File found : foundFiles)
          {
            ErrorLog.errPrintln("Checking " + found.getPath() + " is okay");
            boolean add = true;
            for (String notVersion : versionBlacklist)
            {
              StringBuilder vSB = new StringBuilder();
              if (versionSeparator != null)
              {
                vSB.append(versionSeparator);
              }
              vSB.append(notVersion);
              String versionString = vSB.toString();
              if (String.format(template, root, versionString)
                      .equals(found.getPath()))
              {
                add = false;
                ErrorLog.errPrintln(
                        "Not adding " + found.getPath() + ": version '"
                                + notVersion + "' is in the blacklist");
                break;
              }
            }
            if (add)
            {
              ErrorLog.errPrintln("Adding " + found.getPath() + " to list");
              matchingFiles.add(found);
            }
          }

          if (versionSeparator != null)
          {
            // try without a version number too
            String nonVersioned = String.format(template, root, "");
            matchingFiles.addAll(
                    FileUtils.getFilesFromGlob(nonVersioned, false));
          }
        }

        // Whitelist. Just add a path for every whitelisted version (or check it
        // exists).
        if (versionWhitelist != null)
        {
          ErrorLog.errPrintln("Adding " + versionWhitelist.length
                  + " whitelist versions");
          for (String addVersion : versionWhitelist)
          {
            StringBuilder vSB = new StringBuilder();
            if (versionSeparator != null)
            {
              vSB.append(versionSeparator);
            }
            vSB.append(addVersion);
            String versionString = vSB.toString();
            String versionPath = String.format(template, root,
                    versionString);
            ErrorLog.errPrintln(
                    "Adding whitelist path '" + versionPath + "'");
            File file = new File(versionPath);
            if (file.exists() || !exists)
            {
              matchingFiles.add(file);
            }
          }

          if (versionSeparator != null)
          {
            // try without a version number too
            String nonVersioned = String.format(template, root, "");
            File file = new File(nonVersioned);
            if (file.exists() || !exists)
            {
              matchingFiles.add(file);
            }
          }
        }
      }
    }

    List<File> files = new ArrayList<File>();
    files.addAll(matchingFiles);
    return files;
  }
}
