/*
* Copyright (c) 2016, 2021, Intel Corporation. All rights reserved.
* Intel Math Library (LIBM) Source Code
*
* DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
*
* This code is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License version 2 only, as
* published by the Free Software Foundation.
*
* This code is distributed in the hope that it will be useful, but WITHOUT
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
* version 2 for more details (a copy is included in the LICENSE file that
* accompanied this code).
*
* You should have received a copy of the GNU General Public License version
* 2 along with this work; if not, write to the Free Software Foundation,
* Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
*
* Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
* or visit www.oracle.com if you need additional information or have any
* questions.
*
*/

#include "precompiled.hpp"
#include "asm/assembler.hpp"
#include "asm/assembler.inline.hpp"
#include "macroAssembler_x86.hpp"
#include "runtime/stubRoutines.hpp"
#include "stubRoutines_x86.hpp"
#include "utilities/globalDefinitions.hpp"

/******************************************************************************/
//                     ALGORITHM DESCRIPTION - SIN()
//                     ---------------------
//
//     1. RANGE REDUCTION
//
//     We perform an initial range reduction from X to r with
//
//          X =~= N * pi/32 + r
//
//     so that |r| <= pi/64 + epsilon. We restrict inputs to those
//     where |N| <= 932560. Beyond this, the range reduction is
//     insufficiently accurate. For extremely small inputs,
//     denormalization can occur internally, impacting performance.
//     This means that the main path is actually only taken for
//     2^-252 <= |X| < 90112.
//
//     To avoid branches, we perform the range reduction to full
//     accuracy each time.
//
//          X - N * (P_1 + P_2 + P_3)
//
//     where P_1 and P_2 are 32-bit numbers (so multiplication by N
//     is exact) and P_3 is a 53-bit number. Together, these
//     approximate pi well enough for all cases in the restricted
//     range.
//
//     The main reduction sequence is:
//
//             y = 32/pi * x
//             N = integer(y)
//     (computed by adding and subtracting off SHIFTER)
//
//             m_1 = N * P_1
//             m_2 = N * P_2
//             r_1 = x - m_1
//             r = r_1 - m_2
//     (this r can be used for most of the calculation)
//
//             c_1 = r_1 - r
//             m_3 = N * P_3
//             c_2 = c_1 - m_2
//             c = c_2 - m_3
//
//     2. MAIN ALGORITHM
//
//     The algorithm uses a table lookup based on B = M * pi / 32
//     where M = N mod 64. The stored values are:
//       sigma             closest power of 2 to cos(B)
//       C_hl              53-bit cos(B) - sigma
//       S_hi + S_lo       2 * 53-bit sin(B)
//
//     The computation is organized as follows:
//
//          sin(B + r + c) = [sin(B) + sigma * r] +
//                           r * (cos(B) - sigma) +
//                           sin(B) * [cos(r + c) - 1] +
//                           cos(B) * [sin(r + c) - r]
//
//     which is approximately:
//
//          [S_hi + sigma * r] +
//          C_hl * r +
//          S_lo + S_hi * [(cos(r) - 1) - r * c] +
//          (C_hl + sigma) * [(sin(r) - r) + c]
//
//     and this is what is actually computed. We separate this sum
//     into four parts:
//
//          hi + med + pols + corr
//
//     where
//
//          hi       = S_hi + sigma r
//          med      = C_hl * r
//          pols     = S_hi * (cos(r) - 1) + (C_hl + sigma) * (sin(r) - r)
//          corr     = S_lo + c * ((C_hl + sigma) - S_hi * r)
//
//     3. POLYNOMIAL
//
//     The polynomial S_hi * (cos(r) - 1) + (C_hl + sigma) *
//     (sin(r) - r) can be rearranged freely, since it is quite
//     small, so we exploit parallelism to the fullest.
//
//          psc4       =   SC_4 * r_1
//          msc4       =   psc4 * r
//          r2         =   r * r
//          msc2       =   SC_2 * r2
//          r4         =   r2 * r2
//          psc3       =   SC_3 + msc4
//          psc1       =   SC_1 + msc2
//          msc3       =   r4 * psc3
//          sincospols =   psc1 + msc3
//          pols       =   sincospols *
//                         <S_hi * r^2 | (C_hl + sigma) * r^3>
//
//     4. CORRECTION TERM
//
//     This is where the "c" component of the range reduction is
//     taken into account; recall that just "r" is used for most of
//     the calculation.
//
//          -c   = m_3 - c_2
//          -d   = S_hi * r - (C_hl + sigma)
//          corr = -c * -d + S_lo
//
//     5. COMPENSATED SUMMATIONS
//
//     The two successive compensated summations add up the high
//     and medium parts, leaving just the low parts to add up at
//     the end.
//
//          rs        =  sigma * r
//          res_int   =  S_hi + rs
//          k_0       =  S_hi - res_int
//          k_2       =  k_0 + rs
//          med       =  C_hl * r
//          res_hi    =  res_int + med
//          k_1       =  res_int - res_hi
//          k_3       =  k_1 + med
//
//     6. FINAL SUMMATION
//
//     We now add up all the small parts:
//
//          res_lo = pols(hi) + pols(lo) + corr + k_1 + k_3
//
//     Now the overall result is just:
//
//          res_hi + res_lo
//
//     7. SMALL ARGUMENTS
//
//     If |x| < SNN (SNN meaning the smallest normal number), we
//     simply perform 0.1111111 cdots 1111 * x. For SNN <= |x|, we
//     do 2^-55 * (2^55 * x - x).
//
// Special cases:
//  sin(NaN) = quiet NaN, and raise invalid exception
//  sin(INF) = NaN and raise invalid exception
//  sin(+/-0) = +/-0
//
/******************************************************************************/

// The 32 bit code is at most SSE2 compliant
ATTRIBUTE_ALIGNED(8) static const juint _zero_none[] =
{
    0x00000000UL, 0x00000000UL, 0x00000000UL, 0xbff00000UL
};

ATTRIBUTE_ALIGNED(4) static const juint __4onpi_d[] =
{
    0x6dc9c883UL, 0x3ff45f30UL
};

ATTRIBUTE_ALIGNED(4) static const juint _TWO_32H[] =
{
    0x00000000UL, 0x41f80000UL
};

ATTRIBUTE_ALIGNED(4) static const juint _pi04_3d[] =
{
    0x54442d00UL, 0x3fe921fbUL, 0x98cc5180UL, 0x3ce84698UL, 0xcbb5bf6cUL,
    0xb9dfc8f8UL
};

ATTRIBUTE_ALIGNED(4) static const juint _pi04_5d[] =
{
    0x54400000UL, 0x3fe921fbUL, 0x1a600000UL, 0x3dc0b461UL, 0x2e000000UL,
    0x3b93198aUL, 0x25200000UL, 0x396b839aUL, 0x533e63a0UL, 0x37027044UL
};

ATTRIBUTE_ALIGNED(4) static const juint _SCALE[] =
{
    0x00000000UL, 0x32600000UL
};

ATTRIBUTE_ALIGNED(4) static const juint _zeros[] =
{
    0x00000000UL, 0x00000000UL, 0x00000000UL, 0x80000000UL
};

ATTRIBUTE_ALIGNED(4) static const juint _pi04_2d[] =
{
    0x54400000UL, 0x3fe921fbUL, 0x1a626331UL, 0x3dc0b461UL
};

ATTRIBUTE_ALIGNED(4) static const juint _TWO_12H[] =
{
    0x00000000UL, 0x40b80000UL
};

ATTRIBUTE_ALIGNED(2) static const jushort __4onpi_31l[] =
{
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x836e, 0xa2f9,
    0x40d8, 0x0000, 0x0000, 0x0000, 0x2a50, 0x9c88, 0x40b7, 0x0000, 0x0000, 0x0000,
    0xabe8, 0xfe13, 0x4099, 0x0000, 0x0000, 0x0000, 0x6ee0, 0xfa9a, 0x4079, 0x0000,
    0x0000, 0x0000, 0x9580, 0xdb62, 0x4058, 0x0000, 0x0000, 0x0000, 0x1c82, 0xc9e2,
    0x403d, 0x0000, 0x0000, 0x0000, 0xb1c0, 0xff28, 0x4019, 0x0000, 0x0000, 0x0000,
    0xef14, 0xaf7a, 0x3ffe, 0x0000, 0x0000, 0x0000, 0x48dc, 0xc36e, 0x3fdf, 0x0000,
    0x0000, 0x0000, 0x3740, 0xe909, 0x3fbe, 0x0000, 0x0000, 0x0000, 0x924a, 0xb801,
    0x3fa2, 0x0000, 0x0000, 0x0000, 0x3a32, 0xdd41, 0x3f83, 0x0000, 0x0000, 0x0000,
    0x8778, 0x873f, 0x3f62, 0x0000, 0x0000, 0x0000, 0x1298, 0xb1cb, 0x3f44, 0x0000,
    0x0000, 0x0000, 0xa208, 0x9cfb, 0x3f26, 0x0000, 0x0000, 0x0000, 0xbaec, 0xd7d4,
    0x3f06, 0x0000, 0x0000, 0x0000, 0xd338, 0x8909, 0x3ee7, 0x0000, 0x0000, 0x0000,
    0x68b8, 0xe04d, 0x3ec7, 0x0000, 0x0000, 0x0000, 0x4e64, 0xdf90, 0x3eaa, 0x0000,
    0x0000, 0x0000, 0xc1a8, 0xeb1c, 0x3e89, 0x0000, 0x0000, 0x0000, 0x2720, 0xce7d,
    0x3e6a, 0x0000, 0x0000, 0x0000, 0x77b8, 0x8bf1, 0x3e4b, 0x0000, 0x0000, 0x0000,
    0xec7e, 0xe4a0, 0x3e2e, 0x0000, 0x0000, 0x0000, 0xffbc, 0xf12f, 0x3e0f, 0x0000,
    0x0000, 0x0000, 0xfdc0, 0xb301, 0x3deb, 0x0000, 0x0000, 0x0000, 0xc5ac, 0x9788,
    0x3dd1, 0x0000, 0x0000, 0x0000, 0x47da, 0x829b, 0x3db2, 0x0000, 0x0000, 0x0000,
    0xd9e4, 0xa6cf, 0x3d93, 0x0000, 0x0000, 0x0000, 0x36e8, 0xf961, 0x3d73, 0x0000,
    0x0000, 0x0000, 0xf668, 0xf463, 0x3d54, 0x0000, 0x0000, 0x0000, 0x5168, 0xf2ff,
    0x3d35, 0x0000, 0x0000, 0x0000, 0x758e, 0xea4f, 0x3d17, 0x0000, 0x0000, 0x0000,
    0xf17a, 0xebe5, 0x3cf8, 0x0000, 0x0000, 0x0000, 0x9cfa, 0x9e83, 0x3cd9, 0x0000,
    0x0000, 0x0000, 0xa4ba, 0xe294, 0x3cba, 0x0000, 0x0000, 0x0000, 0xd7ec, 0x9afe,
    0x3c9a, 0x0000, 0x0000, 0x0000, 0xae80, 0x8fc6, 0x3c79, 0x0000, 0x0000, 0x0000,
    0x3304, 0x8560, 0x3c5c, 0x0000, 0x0000, 0x0000, 0x6d70, 0xdf8f, 0x3c3b, 0x0000,
    0x0000, 0x0000, 0x3ef0, 0xafc3, 0x3c1e, 0x0000, 0x0000, 0x0000, 0xd0d8, 0x826b,
    0x3bfe, 0x0000, 0x0000, 0x0000, 0x1c80, 0xed4f, 0x3bdd, 0x0000, 0x0000, 0x0000,
    0x730c, 0xb0af, 0x3bc1, 0x0000, 0x0000, 0x0000, 0x6660, 0xc219, 0x3ba2, 0x0000,
    0x0000, 0x0000, 0x940c, 0xabe2, 0x3b83, 0x0000, 0x0000, 0x0000, 0xdffc, 0x8408,
    0x3b64, 0x0000, 0x0000, 0x0000, 0x6b98, 0xc402, 0x3b45, 0x0000, 0x0000, 0x0000,
    0x1818, 0x9cc4, 0x3b26, 0x0000, 0x0000, 0x0000, 0x5390, 0xaab6, 0x3b05, 0x0000,
    0x0000, 0x0000, 0xb070, 0xd464, 0x3ae9, 0x0000, 0x0000, 0x0000, 0x231a, 0x9ef0,
    0x3aca, 0x0000, 0x0000, 0x0000, 0x0670, 0xd1f1, 0x3aaa, 0x0000, 0x0000, 0x0000,
    0x7738, 0xd9f3, 0x3a8a, 0x0000, 0x0000, 0x0000, 0xa834, 0x8092, 0x3a6c, 0x0000,
    0x0000, 0x0000, 0xb45c, 0xce23, 0x3a4d, 0x0000, 0x0000, 0x0000, 0x36e8, 0xb0e5,
    0x3a2d, 0x0000, 0x0000, 0x0000, 0xd156, 0xaf44, 0x3a10, 0x0000, 0x0000, 0x0000,
    0x9f52, 0x8c82, 0x39f1, 0x0000, 0x0000, 0x0000, 0x829c, 0xff83, 0x39d1, 0x0000,
    0x0000, 0x0000, 0x7d06, 0xefc6, 0x39b3, 0x0000, 0x0000, 0x0000, 0x93e0, 0xb0b7,
    0x3992, 0x0000, 0x0000, 0x0000, 0xedde, 0xc193, 0x3975, 0x0000, 0x0000, 0x0000,
    0xbbc0, 0xcf49, 0x3952, 0x0000, 0x0000, 0x0000, 0xbdf0, 0xd63c, 0x3937, 0x0000,
    0x0000, 0x0000, 0x1f34, 0x9f3a, 0x3918, 0x0000, 0x0000, 0x0000, 0x3f8e, 0xe579,
    0x38f9, 0x0000, 0x0000, 0x0000, 0x90c8, 0xc3f8, 0x38d9, 0x0000, 0x0000, 0x0000,
    0x48c0, 0xf8f8, 0x38b7, 0x0000, 0x0000, 0x0000, 0xed56, 0xafa6, 0x389c, 0x0000,
    0x0000, 0x0000, 0x8218, 0xb969, 0x387d, 0x0000, 0x0000, 0x0000, 0x1852, 0xec57,
    0x385e, 0x0000, 0x0000, 0x0000, 0x670c, 0xd674, 0x383e, 0x0000, 0x0000, 0x0000,
    0xad40, 0xc2c4, 0x3820, 0x0000, 0x0000, 0x0000, 0x2e80, 0xa696, 0x3801, 0x0000,
    0x0000, 0x0000, 0xd800, 0xc467, 0x37dc, 0x0000, 0x0000, 0x0000, 0x3c72, 0xc5ae,
    0x37c3, 0x0000, 0x0000, 0x0000, 0xb006, 0xac69, 0x37a4, 0x0000, 0x0000, 0x0000,
    0x34a0, 0x8cdf, 0x3782, 0x0000, 0x0000, 0x0000, 0x9ed2, 0xd25e, 0x3766, 0x0000,
    0x0000, 0x0000, 0x6fec, 0xaaaa, 0x3747, 0x0000, 0x0000, 0x0000, 0x6040, 0xfb5c,
    0x3726, 0x0000, 0x0000, 0x0000, 0x764c, 0xa3fc, 0x3708, 0x0000, 0x0000, 0x0000,
    0xb254, 0x954e, 0x36e9, 0x0000, 0x0000, 0x0000, 0x3e1c, 0xf5dc, 0x36ca, 0x0000,
    0x0000, 0x0000, 0x7b06, 0xc635, 0x36ac, 0x0000, 0x0000, 0x0000, 0xa8ba, 0xd738,
    0x368d, 0x0000, 0x0000, 0x0000, 0x06cc, 0xb24e, 0x366d, 0x0000, 0x0000, 0x0000,
    0x7108, 0xac76, 0x364f, 0x0000, 0x0000, 0x0000, 0x2324, 0xa7cb, 0x3630, 0x0000,
    0x0000, 0x0000, 0xac40, 0xef15, 0x360f, 0x0000, 0x0000, 0x0000, 0xae46, 0xd516,
    0x35f2, 0x0000, 0x0000, 0x0000, 0x615e, 0xe003, 0x35d3, 0x0000, 0x0000, 0x0000,
    0x0cf0, 0xefe7, 0x35b1, 0x0000, 0x0000, 0x0000, 0xfb50, 0xf98c, 0x3595, 0x0000,
    0x0000, 0x0000, 0x0abc, 0xf333, 0x3575, 0x0000, 0x0000, 0x0000, 0xdd60, 0xca3f,
    0x3555, 0x0000, 0x0000, 0x0000, 0x7eb6, 0xd87f, 0x3538, 0x0000, 0x0000, 0x0000,
    0x44f4, 0xb291, 0x3519, 0x0000, 0x0000, 0x0000, 0xff80, 0xc982, 0x34f6, 0x0000,
    0x0000, 0x0000, 0x9de0, 0xd9b8, 0x34db, 0x0000, 0x0000, 0x0000, 0xcd42, 0x9366,
    0x34bc, 0x0000, 0x0000, 0x0000, 0xbef0, 0xfaee, 0x349d, 0x0000, 0x0000, 0x0000,
    0xdac4, 0xb6f1, 0x347d, 0x0000, 0x0000, 0x0000, 0xf140, 0x94de, 0x345d, 0x0000,
    0x0000, 0x0000, 0xa218, 0x8b4b, 0x343e, 0x0000, 0x0000, 0x0000, 0x6380, 0xa135,
    0x341e, 0x0000, 0x0000, 0x0000, 0xb184, 0x8cb2, 0x3402, 0x0000, 0x0000, 0x0000,
    0x196e, 0xdc61, 0x33e3, 0x0000, 0x0000, 0x0000, 0x0c00, 0xde05, 0x33c4, 0x0000,
    0x0000, 0x0000, 0xef9a, 0xbd38, 0x33a5, 0x0000, 0x0000, 0x0000, 0xc1a0, 0xdf00,
    0x3385, 0x0000, 0x0000, 0x0000, 0x1090, 0x9973, 0x3365, 0x0000, 0x0000, 0x0000,
    0x4882, 0x8301, 0x3348, 0x0000, 0x0000, 0x0000, 0x7abe, 0xadc7, 0x3329, 0x0000,
    0x0000, 0x0000, 0x7cba, 0xec2b, 0x330a, 0x0000, 0x0000, 0x0000, 0xa520, 0x8f21,
    0x32e9, 0x0000, 0x0000, 0x0000, 0x710c, 0x8d36, 0x32cc, 0x0000, 0x0000, 0x0000,
    0x5212, 0xc6ed, 0x32ad, 0x0000, 0x0000, 0x0000, 0x7308, 0xfd76, 0x328d, 0x0000,
    0x0000, 0x0000, 0x5014, 0xd548, 0x326f, 0x0000, 0x0000, 0x0000, 0xd3f2, 0xb499,
    0x3250, 0x0000, 0x0000, 0x0000, 0x7f74, 0xa606, 0x3230, 0x0000, 0x0000, 0x0000,
    0xf0a8, 0xd720, 0x3212, 0x0000, 0x0000, 0x0000, 0x185c, 0xe20f, 0x31f2, 0x0000,
    0x0000, 0x0000, 0xa5a8, 0x8738, 0x31d4, 0x0000, 0x0000, 0x0000, 0xdd74, 0xcafb,
    0x31b4, 0x0000, 0x0000, 0x0000, 0x98b6, 0xbd8e, 0x3196, 0x0000, 0x0000, 0x0000,
    0xe9de, 0x977f, 0x3177, 0x0000, 0x0000, 0x0000, 0x67c0, 0x818d, 0x3158, 0x0000,
    0x0000, 0x0000, 0xe52a, 0x9322, 0x3139, 0x0000, 0x0000, 0x0000, 0xe568, 0x9b6c,
    0x3119, 0x0000, 0x0000, 0x0000, 0x2358, 0xaa0a, 0x30fa, 0x0000, 0x0000, 0x0000,
    0xe480, 0xe13b, 0x30d9, 0x0000, 0x0000, 0x0000, 0x3024, 0x90a1, 0x30bd, 0x0000,
    0x0000, 0x0000, 0x9620, 0xda30, 0x309d, 0x0000, 0x0000, 0x0000, 0x898a, 0xb388,
    0x307f, 0x0000, 0x0000, 0x0000, 0xb24c, 0xc891, 0x3060, 0x0000, 0x0000, 0x0000,
    0x8056, 0xf98b, 0x3041, 0x0000, 0x0000, 0x0000, 0x72a4, 0xa1ea, 0x3021, 0x0000,
    0x0000, 0x0000, 0x6af8, 0x9488, 0x3001, 0x0000, 0x0000, 0x0000, 0xe00c, 0xdfcb,
    0x2fe4, 0x0000, 0x0000, 0x0000, 0xeeec, 0xc941, 0x2fc4, 0x0000, 0x0000, 0x0000,
    0x53e0, 0xe70f, 0x2fa4, 0x0000, 0x0000, 0x0000, 0x8f60, 0x9c07, 0x2f85, 0x0000,
    0x0000, 0x0000, 0xb328, 0xc3e7, 0x2f68, 0x0000, 0x0000, 0x0000, 0x9404, 0xf8c7,
    0x2f48, 0x0000, 0x0000, 0x0000, 0x38e0, 0xc99f, 0x2f29, 0x0000, 0x0000, 0x0000,
    0x9778, 0xd984, 0x2f09, 0x0000, 0x0000, 0x0000, 0xe700, 0xd142, 0x2eea, 0x0000,
    0x0000, 0x0000, 0xd904, 0x9443, 0x2ecd, 0x0000, 0x0000, 0x0000, 0xd4ba, 0xae7e,
    0x2eae, 0x0000, 0x0000, 0x0000, 0x8e5e, 0x8524, 0x2e8f, 0x0000, 0x0000, 0x0000,
    0xb550, 0xc9ed, 0x2e6e, 0x0000, 0x0000, 0x0000, 0x53b8, 0x8648, 0x2e51, 0x0000,
    0x0000, 0x0000, 0xdae4, 0x87f9, 0x2e32, 0x0000, 0x0000, 0x0000, 0x2942, 0xd966,
    0x2e13, 0x0000, 0x0000, 0x0000, 0x4f28, 0xcf3c, 0x2df3, 0x0000, 0x0000, 0x0000,
    0xfa40, 0xc4ef, 0x2dd1, 0x0000, 0x0000, 0x0000, 0x4424, 0xbca7, 0x2db5, 0x0000,
    0x0000, 0x0000, 0x2e62, 0xcdc5, 0x2d97, 0x0000, 0x0000, 0x0000, 0xed88, 0x996b,
    0x2d78, 0x0000, 0x0000, 0x0000, 0x7c30, 0xd97d, 0x2d56, 0x0000, 0x0000, 0x0000,
    0xed26, 0xbf6e, 0x2d3a, 0x0000, 0x0000, 0x0000, 0x2918, 0x921b, 0x2d1a, 0x0000,
    0x0000, 0x0000, 0x4e24, 0xe84e, 0x2cfb, 0x0000, 0x0000, 0x0000, 0x6dc0, 0x92ec,
    0x2cdd, 0x0000, 0x0000, 0x0000, 0x4f2c, 0xacf8, 0x2cbd, 0x0000, 0x0000, 0x0000,
    0xc634, 0xf094, 0x2c9e, 0x0000, 0x0000, 0x0000, 0xdc70, 0xe5d3, 0x2c7e, 0x0000,
    0x0000, 0x0000, 0x2180, 0xa600, 0x2c5b, 0x0000, 0x0000, 0x0000, 0x8480, 0xd680,
    0x2c3c, 0x0000, 0x0000, 0x0000, 0x8b24, 0xd63b, 0x2c22, 0x0000, 0x0000, 0x0000,
    0x02e0, 0xaa47, 0x2c00, 0x0000, 0x0000, 0x0000, 0x9ad0, 0xee84, 0x2be3, 0x0000,
    0x0000, 0x0000, 0xf7dc, 0xf699, 0x2bc6, 0x0000, 0x0000, 0x0000, 0xddde, 0xe490,
    0x2ba7, 0x0000, 0x0000, 0x0000, 0x34a0, 0xb4fd, 0x2b85, 0x0000, 0x0000, 0x0000,
    0x91b4, 0x8ef6, 0x2b68, 0x0000, 0x0000, 0x0000, 0xa3e0, 0xa2a7, 0x2b47, 0x0000,
    0x0000, 0x0000, 0xcce4, 0x82b3, 0x2b2a, 0x0000, 0x0000, 0x0000, 0xe4be, 0x8207,
    0x2b0c, 0x0000, 0x0000, 0x0000, 0x1d92, 0xab43, 0x2aed, 0x0000, 0x0000, 0x0000,
    0xe818, 0xf9f6, 0x2acd, 0x0000, 0x0000, 0x0000, 0xff12, 0xba80, 0x2aaf, 0x0000,
    0x0000, 0x0000, 0x5254, 0x8529, 0x2a90, 0x0000, 0x0000, 0x0000, 0x1b88, 0xe032,
    0x2a71, 0x0000, 0x0000, 0x0000, 0x3248, 0xd86d, 0x2a50, 0x0000, 0x0000, 0x0000,
    0x3140, 0xc9d5, 0x2a2e, 0x0000, 0x0000, 0x0000, 0x14e6, 0xbd47, 0x2a14, 0x0000,
    0x0000, 0x0000, 0x5c10, 0xe544, 0x29f4, 0x0000, 0x0000, 0x0000, 0x9f50, 0x90b6,
    0x29d4, 0x0000, 0x0000, 0x0000, 0x9850, 0xab55, 0x29b6, 0x0000, 0x0000, 0x0000,
    0x2750, 0x9d07, 0x2998, 0x0000, 0x0000, 0x0000, 0x6700, 0x8bbb, 0x2973, 0x0000,
    0x0000, 0x0000, 0x5dba, 0xed31, 0x295a, 0x0000, 0x0000, 0x0000, 0x61dc, 0x85fe,
    0x293a, 0x0000, 0x0000, 0x0000, 0x9ba2, 0xd6b4, 0x291c, 0x0000, 0x0000, 0x0000,
    0x2d30, 0xe3a5, 0x28fb, 0x0000, 0x0000, 0x0000, 0x6630, 0xb566, 0x28dd, 0x0000,
    0x0000, 0x0000, 0x5ad4, 0xa829, 0x28bf, 0x0000, 0x0000, 0x0000, 0x89d8, 0xe290,
    0x28a0, 0x0000, 0x0000, 0x0000, 0x3916, 0xc428, 0x2881, 0x0000, 0x0000, 0x0000,
    0x0490, 0xbea4, 0x2860, 0x0000, 0x0000, 0x0000, 0xee06, 0x80ee, 0x2843, 0x0000,
    0x0000, 0x0000, 0xfc00, 0xf327, 0x2820, 0x0000, 0x0000, 0x0000, 0xea40, 0xa871,
    0x2800, 0x0000, 0x0000, 0x0000, 0x63d8, 0x9c26, 0x27e4, 0x0000, 0x0000, 0x0000,
    0x07ba, 0xc0c9, 0x27c7, 0x0000, 0x0000, 0x0000, 0x3fa2, 0x9797, 0x27a8, 0x0000,
    0x0000, 0x0000, 0x21c6, 0xfeca, 0x2789, 0x0000, 0x0000, 0x0000, 0xde40, 0x860d,
    0x2768, 0x0000, 0x0000, 0x0000, 0x9cc8, 0x98ce, 0x2749, 0x0000, 0x0000, 0x0000,
    0x3778, 0xa31c, 0x272a, 0x0000, 0x0000, 0x0000, 0xe778, 0xf6e2, 0x270b, 0x0000,
    0x0000, 0x0000, 0x59b8, 0xf841, 0x26ed, 0x0000, 0x0000, 0x0000, 0x02e0, 0xad04,
    0x26cd, 0x0000, 0x0000, 0x0000, 0x5a92, 0x9380, 0x26b0, 0x0000, 0x0000, 0x0000,
    0xc740, 0x8886, 0x268d, 0x0000, 0x0000, 0x0000, 0x0680, 0xfaf8, 0x266c, 0x0000,
    0x0000, 0x0000, 0xfb60, 0x897f, 0x2653, 0x0000, 0x0000, 0x0000, 0x8760, 0xf903,
    0x2634, 0x0000, 0x0000, 0x0000, 0xad2a, 0xc2c8, 0x2615, 0x0000, 0x0000, 0x0000,
    0x2d86, 0x8aef, 0x25f6, 0x0000, 0x0000, 0x0000, 0x1ef4, 0xe627, 0x25d6, 0x0000,
    0x0000, 0x0000, 0x09e4, 0x8020, 0x25b7, 0x0000, 0x0000, 0x0000, 0x7548, 0xd227,
    0x2598, 0x0000, 0x0000, 0x0000, 0x75dc, 0xfb5b, 0x2579, 0x0000, 0x0000, 0x0000,
    0xea84, 0xc8b6, 0x255a, 0x0000, 0x0000, 0x0000, 0xe4d0, 0x8145, 0x253b, 0x0000,
    0x0000, 0x0000, 0x3640, 0x9768, 0x251c, 0x0000, 0x0000, 0x0000, 0x246a, 0xccec,
    0x24fe, 0x0000, 0x0000, 0x0000, 0x51d0, 0xa075, 0x24dd, 0x0000, 0x0000, 0x0000,
    0x4638, 0xa385, 0x24bf, 0x0000, 0x0000, 0x0000, 0xd788, 0xd776, 0x24a1, 0x0000,
    0x0000, 0x0000, 0x1370, 0x8997, 0x2482, 0x0000, 0x0000, 0x0000, 0x1e88, 0x9b67,
    0x2462, 0x0000, 0x0000, 0x0000, 0x6c08, 0xd975, 0x2444, 0x0000, 0x0000, 0x0000,
    0xfdb0, 0xcfc0, 0x2422, 0x0000, 0x0000, 0x0000, 0x3100, 0xc026, 0x2406, 0x0000,
    0x0000, 0x0000, 0xc5b4, 0xae64, 0x23e6, 0x0000, 0x0000, 0x0000, 0x2280, 0xf687,
    0x23c3, 0x0000, 0x0000, 0x0000, 0x2de0, 0x9006, 0x23a9, 0x0000, 0x0000, 0x0000,
    0x24bc, 0xf631, 0x238a, 0x0000, 0x0000, 0x0000, 0xb8d4, 0xa975, 0x236b, 0x0000,
    0x0000, 0x0000, 0xd9a4, 0xb949, 0x234b, 0x0000, 0x0000, 0x0000, 0xb54e, 0xbd39,
    0x232d, 0x0000, 0x0000, 0x0000, 0x4aac, 0x9a52, 0x230e, 0x0000, 0x0000, 0x0000,
    0xbbbc, 0xd085, 0x22ef, 0x0000, 0x0000, 0x0000, 0xdf18, 0xc633, 0x22cf, 0x0000,
    0x0000, 0x0000, 0x16d0, 0xeca5, 0x22af, 0x0000, 0x0000, 0x0000, 0xf2a0, 0xdf6f,
    0x228e, 0x0000, 0x0000, 0x0000, 0x8c44, 0xe86b, 0x2272, 0x0000, 0x0000, 0x0000,
    0x35c0, 0xbbf4, 0x2253, 0x0000, 0x0000, 0x0000, 0x0c40, 0xdafb, 0x2230, 0x0000,
    0x0000, 0x0000, 0x92dc, 0x9935, 0x2216, 0x0000, 0x0000, 0x0000, 0x0ca0, 0xbda6,
    0x21f3, 0x0000, 0x0000, 0x0000, 0x5958, 0xa6fd, 0x21d6, 0x0000, 0x0000, 0x0000,
    0xa3dc, 0x9d7f, 0x21b9, 0x0000, 0x0000, 0x0000, 0x79dc, 0xfcb5, 0x2199, 0x0000,
    0x0000, 0x0000, 0xf264, 0xcebb, 0x217b, 0x0000, 0x0000, 0x0000, 0x0abe, 0x8308,
    0x215c, 0x0000, 0x0000, 0x0000, 0x30ae, 0xb463, 0x213d, 0x0000, 0x0000, 0x0000,
    0x6228, 0xb040, 0x211c, 0x0000, 0x0000, 0x0000, 0xc9b2, 0xf43b, 0x20ff, 0x0000,
    0x0000, 0x0000, 0x3d8e, 0xa4b3, 0x20e0, 0x0000, 0x0000, 0x0000, 0x84e6, 0x8dab,
    0x20c1, 0x0000, 0x0000, 0x0000, 0xa124, 0x9b74, 0x20a1, 0x0000, 0x0000, 0x0000,
    0xc276, 0xd497, 0x2083, 0x0000, 0x0000, 0x0000, 0x6354, 0xa466, 0x2063, 0x0000,
    0x0000, 0x0000, 0x8654, 0xaf0a, 0x2044, 0x0000, 0x0000, 0x0000, 0x1d20, 0xfa5c,
    0x2024, 0x0000, 0x0000, 0x0000, 0xbcd0, 0xf3f0, 0x2004, 0x0000, 0x0000, 0x0000,
    0xedf0, 0xf0b6, 0x1fe7, 0x0000, 0x0000, 0x0000, 0x45bc, 0x9182, 0x1fc9, 0x0000,
    0x0000, 0x0000, 0xe254, 0xdc85, 0x1faa, 0x0000, 0x0000, 0x0000, 0xb898, 0xe9b1,
    0x1f8a, 0x0000, 0x0000, 0x0000, 0x0ebe, 0xe6f0, 0x1f6c, 0x0000, 0x0000, 0x0000,
    0xa9b8, 0xf584, 0x1f4c, 0x0000, 0x0000, 0x0000, 0x12e8, 0xdf6b, 0x1f2e, 0x0000,
    0x0000, 0x0000, 0x9f9e, 0xcd55, 0x1f0f, 0x0000, 0x0000, 0x0000, 0x05a0, 0xec3a,
    0x1eef, 0x0000, 0x0000, 0x0000, 0xd8e0, 0x96f8, 0x1ed1, 0x0000, 0x0000, 0x0000,
    0x3bd4, 0xccc6, 0x1eb1, 0x0000, 0x0000, 0x0000, 0x4910, 0xb87b, 0x1e93, 0x0000,
    0x0000, 0x0000, 0xbefc, 0xd40b, 0x1e73, 0x0000, 0x0000, 0x0000, 0x317e, 0xa406,
    0x1e55, 0x0000, 0x0000, 0x0000, 0x6bb2, 0xc2b2, 0x1e36, 0x0000, 0x0000, 0x0000,
    0xb87e, 0xbb78, 0x1e17, 0x0000, 0x0000, 0x0000, 0xa03c, 0xdbbd, 0x1df7, 0x0000,
    0x0000, 0x0000, 0x5b6c, 0xe3c8, 0x1dd9, 0x0000, 0x0000, 0x0000, 0x8968, 0xca8e,
    0x1dba, 0x0000, 0x0000, 0x0000, 0xc024, 0xe6ab, 0x1d9a, 0x0000, 0x0000, 0x0000,
    0x4110, 0xd4eb, 0x1d7a, 0x0000, 0x0000, 0x0000, 0xa168, 0xbdb5, 0x1d5d, 0x0000,
    0x0000, 0x0000, 0x012e, 0xa5fa, 0x1d3e, 0x0000, 0x0000, 0x0000, 0x6838, 0x9c1f,
    0x1d1e, 0x0000, 0x0000, 0x0000, 0xa158, 0xaa76, 0x1d00, 0x0000, 0x0000, 0x0000,
    0x090a, 0xbd95, 0x1ce1, 0x0000, 0x0000, 0x0000, 0xf73e, 0x8b6d, 0x1cc2, 0x0000,
    0x0000, 0x0000, 0x5fda, 0xbcbf, 0x1ca3, 0x0000, 0x0000, 0x0000, 0xdbe8, 0xb89f,
    0x1c84, 0x0000, 0x0000, 0x0000, 0x6e4c, 0x96c7, 0x1c64, 0x0000, 0x0000, 0x0000,
    0x19c2, 0xf2a4, 0x1c46, 0x0000, 0x0000, 0x0000, 0xb800, 0xf855, 0x1c1e, 0x0000,
    0x0000, 0x0000, 0x87fc, 0x85ff, 0x1c08, 0x0000, 0x0000, 0x0000, 0x1418, 0x839f,
    0x1be9, 0x0000, 0x0000, 0x0000, 0x6186, 0xd9d8, 0x1bca, 0x0000, 0x0000, 0x0000,
    0xf500, 0xabaa, 0x1ba6, 0x0000, 0x0000, 0x0000, 0x7b36, 0xdafe, 0x1b8c, 0x0000,
    0x0000, 0x0000, 0xf394, 0xe6d8, 0x1b6c, 0x0000, 0x0000, 0x0000, 0x6efc, 0x9e55,
    0x1b4e, 0x0000, 0x0000, 0x0000, 0x5e10, 0xc523, 0x1b2e, 0x0000, 0x0000, 0x0000,
    0x8210, 0xb6f9, 0x1b0d, 0x0000, 0x0000, 0x0000, 0x9ab0, 0x96e3, 0x1af1, 0x0000,
    0x0000, 0x0000, 0x3864, 0x92e7, 0x1ad1, 0x0000, 0x0000, 0x0000, 0x9878, 0xdc65,
    0x1ab1, 0x0000, 0x0000, 0x0000, 0xfa20, 0xd6cb, 0x1a94, 0x0000, 0x0000, 0x0000,
    0x6c00, 0xa4e4, 0x1a70, 0x0000, 0x0000, 0x0000, 0xab40, 0xb41b, 0x1a53, 0x0000,
    0x0000, 0x0000, 0x43a4, 0x8ede, 0x1a37, 0x0000, 0x0000, 0x0000, 0x22e0, 0x9314,
    0x1a15, 0x0000, 0x0000, 0x0000, 0x6170, 0xb949, 0x19f8, 0x0000, 0x0000, 0x0000,
    0x6b00, 0xe056, 0x19d8, 0x0000, 0x0000, 0x0000, 0x9ba8, 0xa94c, 0x19b9, 0x0000,
    0x0000, 0x0000, 0xfaa0, 0xaa16, 0x199b, 0x0000, 0x0000, 0x0000, 0x899a, 0xf627,
    0x197d, 0x0000, 0x0000, 0x0000, 0x9f20, 0xfb70, 0x195d, 0x0000, 0x0000, 0x0000,
    0xa4b8, 0xc176, 0x193e, 0x0000, 0x0000, 0x0000, 0xb21c, 0x85c3, 0x1920, 0x0000,
    0x0000, 0x0000, 0x50d2, 0x9b19, 0x1901, 0x0000, 0x0000, 0x0000, 0xd4b0, 0xb708,
    0x18e0, 0x0000, 0x0000, 0x0000, 0xfb88, 0xf510, 0x18c1, 0x0000, 0x0000, 0x0000,
    0x31ec, 0xdc8d, 0x18a3, 0x0000, 0x0000, 0x0000, 0x3c00, 0xbff9, 0x1885, 0x0000,
    0x0000, 0x0000, 0x5020, 0xc30b, 0x1862, 0x0000, 0x0000, 0x0000, 0xd4f0, 0xda0c,
    0x1844, 0x0000, 0x0000, 0x0000, 0x20d2, 0x99a5, 0x1828, 0x0000, 0x0000, 0x0000,
    0x852e, 0xd159, 0x1809, 0x0000, 0x0000, 0x0000, 0x7cd8, 0x97a1, 0x17e9, 0x0000,
    0x0000, 0x0000, 0x423a, 0x997b, 0x17cb, 0x0000, 0x0000, 0x0000, 0xc1c0, 0xbe7d,
    0x17a8, 0x0000, 0x0000, 0x0000, 0xe8bc, 0xdcdd, 0x178d, 0x0000, 0x0000, 0x0000,
    0x8b28, 0xae06, 0x176e, 0x0000, 0x0000, 0x0000, 0x102e, 0xb8d4, 0x174f, 0x0000,
    0x0000, 0x0000, 0xaa00, 0xaa5c, 0x172f, 0x0000, 0x0000, 0x0000, 0x51f0, 0x9fc0,
    0x170e, 0x0000, 0x0000, 0x0000, 0xf858, 0xe181, 0x16f2, 0x0000, 0x0000, 0x0000,
    0x91a8, 0x8162, 0x16d3, 0x0000, 0x0000, 0x0000, 0x5f40, 0xcb6f, 0x16b1, 0x0000,
    0x0000, 0x0000, 0xbb50, 0xe55f, 0x1693, 0x0000, 0x0000, 0x0000, 0xacd2, 0xd895,
    0x1676, 0x0000, 0x0000, 0x0000, 0xef30, 0x97bf, 0x1654, 0x0000, 0x0000, 0x0000,
    0xf700, 0xb3d7, 0x1633, 0x0000, 0x0000, 0x0000, 0x3454, 0xa7b5, 0x1619, 0x0000,
    0x0000, 0x0000, 0x6b00, 0xa929, 0x15f6, 0x0000, 0x0000, 0x0000, 0x9f04, 0x89f7,
    0x15db, 0x0000, 0x0000, 0x0000, 0xad78, 0xd985, 0x15bc, 0x0000, 0x0000, 0x0000,
    0xa46a, 0xae3f, 0x159d, 0x0000, 0x0000, 0x0000, 0x63a0, 0xd0da, 0x157c, 0x0000,
    0x0000, 0x0000, 0x5e90, 0x817d, 0x155e, 0x0000, 0x0000, 0x0000, 0x1494, 0xb13f,
    0x1540, 0x0000, 0x0000, 0x0000, 0x0090, 0x9c40, 0x1521, 0x0000, 0x0000, 0x0000,
    0xdd70, 0xcc86, 0x1500, 0x0000, 0x0000, 0x0000, 0x64f8, 0xdb6f, 0x14e1, 0x0000,
    0x0000, 0x0000, 0xe22c, 0xac17, 0x14c3, 0x0000, 0x0000, 0x0000, 0x60e0, 0xa9ad,
    0x14a3, 0x0000, 0x0000, 0x0000, 0x4640, 0xd658, 0x1481, 0x0000, 0x0000, 0x0000,
    0x6490, 0xa181, 0x1467, 0x0000, 0x0000, 0x0000, 0x1df4, 0xaaa2, 0x1447, 0x0000,
    0x0000, 0x0000, 0xb94a, 0x8f61, 0x1429, 0x0000, 0x0000, 0x0000, 0x5198, 0x9d83,
    0x1409, 0x0000, 0x0000, 0x0000, 0x0f7a, 0xa818, 0x13eb, 0x0000, 0x0000, 0x0000,
    0xc45e, 0xc06c, 0x13cc, 0x0000, 0x0000, 0x0000, 0x4ec0, 0xfa29, 0x13a8, 0x0000,
    0x0000, 0x0000, 0x6418, 0x8cad, 0x138c, 0x0000, 0x0000, 0x0000, 0xbcc8, 0xe7d1,
    0x136f, 0x0000, 0x0000, 0x0000, 0xc934, 0xf9b0, 0x134f, 0x0000, 0x0000, 0x0000,
    0x6ce0, 0x98df, 0x1331, 0x0000, 0x0000, 0x0000, 0x3516, 0xe5e9, 0x1312, 0x0000,
    0x0000, 0x0000, 0xc6c0, 0xef8b, 0x12ef, 0x0000, 0x0000, 0x0000, 0xaf02, 0x913d,
    0x12d4, 0x0000, 0x0000, 0x0000, 0xd230, 0xe1d5, 0x12b5, 0x0000, 0x0000, 0x0000,
    0xfba8, 0xc232, 0x1295, 0x0000, 0x0000, 0x0000, 0x7ba4, 0xabeb, 0x1277, 0x0000,
    0x0000, 0x0000, 0x6e5c, 0xc692, 0x1258, 0x0000, 0x0000, 0x0000, 0x76a2, 0x9756,
    0x1239, 0x0000, 0x0000, 0x0000, 0xe180, 0xe423, 0x1214, 0x0000, 0x0000, 0x0000,
    0x8c3c, 0x90f8, 0x11fb, 0x0000, 0x0000, 0x0000, 0x9f3c, 0x9fd2, 0x11dc, 0x0000,
    0x0000, 0x0000, 0x53e0, 0xb73e, 0x11bd, 0x0000, 0x0000, 0x0000, 0x45be, 0x88d6,
    0x119e, 0x0000, 0x0000, 0x0000, 0x111a, 0x8bc0, 0x117f, 0x0000, 0x0000, 0x0000,
    0xe26a, 0xd7ff, 0x1160, 0x0000, 0x0000, 0x0000, 0xfb60, 0xdd8d, 0x113f, 0x0000,
    0x0000, 0x0000, 0x9370, 0xc108, 0x1120, 0x0000, 0x0000, 0x0000, 0x9654, 0x8baf,
    0x1103, 0x0000, 0x0000, 0x0000, 0xd6ec, 0xd6b9, 0x10e4, 0x0000, 0x0000, 0x0000,
    0x23e4, 0xd7b7, 0x10c4, 0x0000, 0x0000, 0x0000, 0x1aa6, 0xa847, 0x10a6, 0x0000,
    0x0000, 0x0000, 0xbee6, 0x9fef, 0x1087, 0x0000, 0x0000, 0x0000, 0x26d0, 0xa6eb,
    0x1066, 0x0000, 0x0000, 0x0000, 0x5b86, 0xa880, 0x1049, 0x0000, 0x0000, 0x0000,
    0x125c, 0xd971, 0x1029, 0x0000, 0x0000, 0x0000, 0x1f78, 0x9d18, 0x100a, 0x0000,
    0x0000, 0x0000, 0x0e84, 0xb15b, 0x0feb, 0x0000, 0x0000, 0x0000, 0xd0c0, 0xc150,
    0x0fcc, 0x0000, 0x0000, 0x0000, 0xa330, 0xc40c, 0x0fad, 0x0000, 0x0000, 0x0000,
    0x5202, 0xfc2c, 0x0f8f, 0x0000, 0x0000, 0x0000, 0x3f7c, 0xecf5, 0x0f6f, 0x0000,
    0x0000, 0x0000, 0xef44, 0xfdfd, 0x0f50, 0x0000, 0x0000, 0x0000, 0x3f6c, 0xab1b,
    0x0f31, 0x0000, 0x0000, 0x0000, 0xf658, 0x89ec, 0x0f11, 0x0000, 0x0000, 0x0000,
    0xbfc8, 0x9ba8, 0x0ef4, 0x0000, 0x0000, 0x0000, 0x3d40, 0xbe21, 0x0ed5, 0x0000,
    0x0000, 0x0000, 0xbbc4, 0xc70d, 0x0eb6, 0x0000, 0x0000, 0x0000, 0x5158, 0xdb16,
    0x0e96, 0x0000, 0x0000, 0x0000, 0xb5a8, 0xa8d8, 0x0e78, 0x0000, 0x0000, 0x0000,
    0xcccc, 0xb40e, 0x0e58, 0x0000, 0x0000, 0x0000, 0x448c, 0xcb62, 0x0e3a, 0x0000,
    0x0000, 0x0000, 0xf12a, 0x8aed, 0x0e1b, 0x0000, 0x0000, 0x0000, 0x79d0, 0xc59c,
    0x0dfb, 0x0000, 0x0000, 0x0000, 0x06b4, 0xcdc9, 0x0ddd, 0x0000, 0x0000, 0x0000,
    0xae70, 0xa979, 0x0dbe, 0x0000, 0x0000, 0x0000, 0x317c, 0xa8fb, 0x0d9e, 0x0000,
    0x0000, 0x0000, 0x5fe0, 0x8a50, 0x0d7d, 0x0000, 0x0000, 0x0000, 0x70b6, 0xfdfa,
    0x0d61, 0x0000, 0x0000, 0x0000, 0x1640, 0x9dc7, 0x0d41, 0x0000, 0x0000, 0x0000,
    0x9a9c, 0xdc50, 0x0d23, 0x0000, 0x0000, 0x0000, 0x4fcc, 0x9a9b, 0x0d04, 0x0000,
    0x0000, 0x0000, 0x7e48, 0x8f77, 0x0ce5, 0x0000, 0x0000, 0x0000, 0x84e4, 0xd4b9,
    0x0cc6, 0x0000, 0x0000, 0x0000, 0x84e0, 0xbd10, 0x0ca6, 0x0000, 0x0000, 0x0000,
    0x1b0a, 0xc8d9, 0x0c88, 0x0000, 0x0000, 0x0000, 0x6a48, 0xfc81, 0x0c68, 0x0000,
    0x0000, 0x0000, 0x070a, 0xbef6, 0x0c4a, 0x0000, 0x0000, 0x0000, 0x8a70, 0xf096,
    0x0c2b, 0x0000, 0x0000, 0x0000, 0xecc2, 0xc994, 0x0c0c, 0x0000, 0x0000, 0x0000,
    0x1540, 0x9537, 0x0bea, 0x0000, 0x0000, 0x0000, 0x1b02, 0xab5b, 0x0bce, 0x0000,
    0x0000, 0x0000, 0x5dc0, 0xb0c8, 0x0bad, 0x0000, 0x0000, 0x0000, 0xc928, 0xe034,
    0x0b8f, 0x0000, 0x0000, 0x0000, 0x2d12, 0xb4b0, 0x0b71, 0x0000, 0x0000, 0x0000,
    0x8fc2, 0xbb94, 0x0b52, 0x0000, 0x0000, 0x0000, 0xe236, 0xe22f, 0x0b33, 0x0000,
    0x0000, 0x0000, 0xb97c, 0xbe9e, 0x0b13, 0x0000, 0x0000, 0x0000, 0xe1a6, 0xe16d,
    0x0af5, 0x0000, 0x0000, 0x0000, 0xd330, 0xbaf0, 0x0ad6, 0x0000, 0x0000, 0x0000,
    0xc0bc, 0xbbd0, 0x0ab7, 0x0000, 0x0000, 0x0000, 0x8e66, 0xdd9b, 0x0a98, 0x0000,
    0x0000, 0x0000, 0xc95c, 0xf799, 0x0a79, 0x0000, 0x0000, 0x0000, 0xdac0, 0xbe4c,
    0x0a55, 0x0000, 0x0000, 0x0000, 0xafc0, 0xc378, 0x0a37, 0x0000, 0x0000, 0x0000,
    0xa880, 0xe341, 0x0a19, 0x0000, 0x0000, 0x0000, 0xc242, 0x81f6, 0x09fd, 0x0000,
    0x0000, 0x0000, 0x7470, 0xc777, 0x09de, 0x0000, 0x0000, 0x0000, 0x62bc, 0xb684,
    0x09be, 0x0000, 0x0000, 0x0000, 0x43ac, 0x8c58, 0x099f, 0x0000, 0x0000, 0x0000,
    0xcc3c, 0xf9ac, 0x0981, 0x0000, 0x0000, 0x0000, 0x1526, 0xb670, 0x0962, 0x0000,
    0x0000, 0x0000, 0xc9fe, 0xdf50, 0x0943, 0x0000, 0x0000, 0x0000, 0x6ae6, 0xc065,
    0x0924, 0x0000, 0x0000, 0x0000, 0xb114, 0xcf29, 0x0905, 0x0000, 0x0000, 0x0000,
    0xd388, 0x922a, 0x08e4, 0x0000, 0x0000, 0x0000, 0xcf54, 0xb926, 0x08c7, 0x0000,
    0x0000, 0x0000, 0x3826, 0xe855, 0x08a8, 0x0000, 0x0000, 0x0000, 0xe7c8, 0x829b,
    0x0888, 0x0000, 0x0000, 0x0000, 0x546c, 0xa903, 0x086a, 0x0000, 0x0000, 0x0000,
    0x8768, 0x99cc, 0x0849, 0x0000, 0x0000, 0x0000, 0x00ac, 0xf529, 0x082b, 0x0000,
    0x0000, 0x0000, 0x2658, 0x9f0b, 0x080c, 0x0000, 0x0000, 0x0000, 0xfe5c, 0x9e21,
    0x07ee, 0x0000, 0x0000, 0x0000, 0x6da2, 0x9910, 0x07cf, 0x0000, 0x0000, 0x0000,
    0x9220, 0xf9b3, 0x07b0, 0x0000, 0x0000, 0x0000, 0x3d90, 0xa541, 0x0791, 0x0000,
    0x0000, 0x0000, 0x6e4c, 0xe7cc, 0x0771, 0x0000, 0x0000, 0x0000, 0xa8fa, 0xe80a,
    0x0753, 0x0000, 0x0000, 0x0000, 0x4e14, 0xc3a7, 0x0734, 0x0000, 0x0000, 0x0000,
    0xf7e0, 0xbad9, 0x0712, 0x0000, 0x0000, 0x0000, 0xfea0, 0xeff2, 0x06f5, 0x0000,
    0x0000, 0x0000, 0xcef6, 0xbd48, 0x06d7, 0x0000, 0x0000, 0x0000, 0x7544, 0xf559,
    0x06b7, 0x0000, 0x0000, 0x0000, 0x2388, 0xf655, 0x0698, 0x0000, 0x0000, 0x0000,
    0xe900, 0xad56, 0x0676, 0x0000, 0x0000, 0x0000, 0x2cc0, 0x8437, 0x0659, 0x0000,
    0x0000, 0x0000, 0x3068, 0xc544, 0x063b, 0x0000, 0x0000, 0x0000, 0xdc70, 0xe73c,
    0x061b, 0x0000, 0x0000, 0x0000, 0xee50, 0x9d49, 0x05fc, 0x0000, 0x0000, 0x0000,
    0x93d2, 0x81f6, 0x05df, 0x0000, 0x0000, 0x0000, 0x941c, 0xadff, 0x05bf, 0x0000,
    0x0000, 0x0000, 0x2ce2, 0x8e45, 0x05a1, 0x0000, 0x0000, 0x0000, 0x4a60, 0x95fd,
    0x0581, 0x0000, 0x0000, 0x0000, 0x79f8, 0xb83a, 0x0563, 0x0000, 0x0000, 0x0000,
    0xcb58, 0xa1f5, 0x0543, 0x0000, 0x0000, 0x0000, 0x2a3a, 0xdc36, 0x0525, 0x0000,
    0x0000, 0x0000, 0x14ee, 0x890e, 0x0506, 0x0000, 0x0000, 0x0000, 0x8f20, 0xc432,
    0x04e3, 0x0000, 0x0000, 0x0000, 0x8440, 0xb21d, 0x04c6, 0x0000, 0x0000, 0x0000,
    0x5430, 0xf698, 0x04a7, 0x0000, 0x0000, 0x0000, 0x04ae, 0x8b20, 0x048a, 0x0000,
    0x0000, 0x0000, 0x04d0, 0xe872, 0x046b, 0x0000, 0x0000, 0x0000, 0xc78e, 0x8893,
    0x044c, 0x0000, 0x0000, 0x0000, 0x0f78, 0x9895, 0x042b, 0x0000, 0x0000, 0x0000,
    0x11d4, 0xdf2e, 0x040d, 0x0000, 0x0000, 0x0000, 0xe84c, 0x89d5, 0x03ef, 0x0000,
    0x0000, 0x0000, 0xf7be, 0x8a67, 0x03d0, 0x0000, 0x0000, 0x0000, 0x95d0, 0xc906,
    0x03b1, 0x0000, 0x0000, 0x0000, 0x64ce, 0xd96c, 0x0392, 0x0000, 0x0000, 0x0000,
    0x97ba, 0xa16f, 0x0373, 0x0000, 0x0000, 0x0000, 0x463c, 0xc51a, 0x0354, 0x0000,
    0x0000, 0x0000, 0xef0a, 0xe93e, 0x0335, 0x0000, 0x0000, 0x0000, 0x526a, 0xa466,
    0x0316, 0x0000, 0x0000, 0x0000, 0x4140, 0xa94d, 0x02f5, 0x0000, 0x0000, 0x0000,
    0xb4ec, 0xce68, 0x02d8, 0x0000, 0x0000, 0x0000, 0x4fa2, 0x8490, 0x02b9, 0x0000,
    0x0000, 0x0000, 0x4e60, 0xca98, 0x0298, 0x0000, 0x0000, 0x0000, 0x08dc, 0xe09c,
    0x027a, 0x0000, 0x0000, 0x0000, 0x2b90, 0xc7e3, 0x025c, 0x0000, 0x0000, 0x0000,
    0x5a7c, 0xf8ef, 0x023c, 0x0000, 0x0000, 0x0000, 0x5022, 0x9d58, 0x021e, 0x0000,
    0x0000, 0x0000, 0x553a, 0xe242, 0x01ff, 0x0000, 0x0000, 0x0000, 0x7e6e, 0xb54d,
    0x01e0, 0x0000, 0x0000, 0x0000, 0xd2d4, 0xa88c, 0x01c1, 0x0000, 0x0000, 0x0000,
    0x75b6, 0xfe6d, 0x01a2, 0x0000, 0x0000, 0x0000, 0x3bb2, 0xf04c, 0x0183, 0x0000,
    0x0000, 0x0000, 0xc2d0, 0xc046, 0x0163, 0x0000, 0x0000, 0x0000, 0x250c, 0xf9d6,
    0x0145, 0x0000, 0x0000, 0x0000, 0xb7b4, 0x8a0d, 0x0126, 0x0000, 0x0000, 0x0000,
    0x1a72, 0xe4f5, 0x0107, 0x0000, 0x0000, 0x0000, 0x825c, 0xa9b8, 0x00e8, 0x0000,
    0x0000, 0x0000, 0x6c90, 0xc9ad, 0x00c6, 0x0000, 0x0000, 0x0000, 0x4d00, 0xd1bb,
    0x00aa, 0x0000, 0x0000, 0x0000, 0xa4a0, 0xee01, 0x0087, 0x0000, 0x0000, 0x0000,
    0x89a8, 0xbe9f, 0x006b, 0x0000, 0x0000, 0x0000, 0x038e, 0xc80c, 0x004d, 0x0000,
    0x0000, 0x0000, 0xfe26, 0x8384, 0x002e, 0x0000, 0x0000, 0x0000, 0xcd90, 0xca57,
    0x000e, 0x0000
};

void MacroAssembler::libm_reduce_pi04l(Register eax, Register ecx, Register edx, Register ebx, Register esi, Register edi, Register ebp, Register esp) {
  Label B1_1, B1_2, B1_3, B1_4, B1_5, B1_6, B1_7, B1_8, B1_9, B1_10, B1_11, B1_12;
  Label B1_13, B1_14, B1_15;

  assert_different_registers(ebx, eax, ecx, edx, esi, edi, ebp, esp);

  address zero_none  = (address)_zero_none;
  address _4onpi_d   = (address)__4onpi_d;
  address TWO_32H    = (address)_TWO_32H;
  address pi04_3d    = (address)_pi04_3d;
  address pi04_5d    = (address)_pi04_5d;
  address SCALE      = (address)_SCALE;
  address zeros      = (address)_zeros;
  address pi04_2d    = (address)_pi04_2d;
  address TWO_12H    = (address)_TWO_12H;
  address _4onpi_31l = (address)__4onpi_31l;

  bind(B1_1);
  push(ebp);
  movl(ebp, esp);
  andl(esp, -16);
  push(esi);
  push(edi);
  push(ebx);
  subl(esp, 20);
  movzwl(ebx, Address(ebp, 16));
  andl(ebx, 32767);
  movl(eax, Address(ebp, 20));
  cmpl(ebx, 16413);
  movl(esi, Address(ebp, 24));
  movl(Address(esp, 4), eax);
  jcc(Assembler::greaterEqual, B1_8);

  bind(B1_2);
  fld_x(Address(ebp, 8));
  fld_d(ExternalAddress(_4onpi_d));    //0x6dc9c883UL, 0x3ff45f30UL
  fmul(1);
  fstp_x(Address(esp, 8));
  movzwl(ecx, Address(esp, 16));
  negl(ecx);
  addl(ecx, 30);
  movl(eax, Address(esp, 12));
  shrl(eax);
  cmpl(Address(esp, 4), 0);
  jcc(Assembler::notEqual, B1_4);

  bind(B1_3);
  lea(ecx, Address(eax, 1));
  andl(ecx, -2);
  jmp(B1_5);

  bind(B1_4);
  movl(ecx, eax);
  addl(eax, Address(esp, 4));
  movl(edx, eax);
  andl(edx, 1);
  addl(ecx, edx);

  bind(B1_5);
  fld_d(ExternalAddress(TWO_32H));    //0x00000000UL, 0x41f80000UL
  cmpl(ebx, 16400);
  movl(Address(esp, 0), ecx);
  fild_s(Address(esp, 0));
  jcc(Assembler::greaterEqual, B1_7);

  bind(B1_6);
  fld_d(ExternalAddress(pi04_3d));    //0x54442d00UL, 0x3fe921fbUL
  fmul(1);
  fsubp(3);
  fxch(1);
  fmul(2);
  fld_s(2);
  fadd(1);
  fsubrp(1);
  fld_s(0);
  fxch(1);
  fsuba(3);
  fld_d(ExternalAddress(8 + pi04_3d));    //0x98cc5180UL, 0x3ce84698UL
  fmul(3);
  fsuba(2);
  fxch(1);
  fsub(2);
  fsubrp(1);
  faddp(3);
  fld_d(ExternalAddress(16 + pi04_3d));    //0xcbb5bf6cUL, 0xb9dfc8f8UL
  fmulp(2);
  fld_s(1);
  fsubr(1);
  fsuba(1);
  fxch(2);
  fsubp(1);
  faddp(2);
  fxch(1);
  jmp(B1_15);

  bind(B1_7);
  fld_d(ExternalAddress(pi04_5d));    //0x54400000UL, 0x3fe921fbUL
  fmul(1);
  fsubp(3);
  fxch(1);
  fmul(2);
  fld_s(2);
  fadd(1);
  fsubrp(1);
  fld_s(0);
  fxch(1);
  fsuba(3);
  fld_d(ExternalAddress(8 + pi04_5d));    //0x1a600000UL, 0x3dc0b461UL
  fmul(3);
  fsuba(2);
  fxch(1);
  fsub(2);
  fsubrp(1);
  faddp(3);
  fld_d(ExternalAddress(16 + pi04_5d));    //0x2e000000UL, 0x3b93198aUL
  fmul(2);
  fld_s(0);
  fsubr(2);
  fsuba(2);
  fxch(1);
  fsubp(2);
  fxch(1);
  faddp(3);
  fld_d(ExternalAddress(24 + pi04_5d));    //0x25200000UL, 0x396b839aUL
  fmul(2);
  fld_s(0);
  fsubr(2);
  fsuba(2);
  fxch(1);
  fsubp(2);
  fxch(1);
  faddp(3);
  fld_d(ExternalAddress(32 + pi04_5d));    //0x533e63a0UL, 0x37027044UL
  fmulp(2);
  fld_s(1);
  fsubr(1);
  fsuba(1);
  fxch(2);
  fsubp(1);
  faddp(2);
  fxch(1);
  jmp(B1_15);

  bind(B1_8);
  fld_x(Address(ebp, 8));
  addl(ebx, -16417);
  fmul_d(as_Address(ExternalAddress(SCALE)));    //0x00000000UL, 0x32600000UL
  movl(eax, -2078209981);
  imull(ebx);
  addl(edx, ebx);
  movl(ecx, ebx);
  sarl(edx, 4);
  sarl(ecx, 31);
  subl(edx, ecx);
  movl(eax, edx);
  shll(eax, 5);
  fstp_x(Address(ebp, 8));
  fld_x(Address(ebp, 8));
  subl(eax, edx);
  movl(Address(ebp, 8), 0);
  subl(ebx, eax);
  fld_x(Address(ebp, 8));
  cmpl(ebx, 17);
  fsuba(1);
  jcc(Assembler::less, B1_10);

  bind(B1_9);
  lea(eax, Address(noreg, edx, Address::times_8));
  lea(ecx, Address(eax, edx, Address::times_4));
  incl(edx);
  fld_x(Address(_4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmul(2);
  fld_x(Address(12 + _4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmul(2);
  fld_s(0);
  fadd(2);
  fsuba(2);
  fxch(1);
  faddp(2);
  fld_s(1);
  fadd(1);
  fstp_x(Address(esp, 8));
  andl(Address(esp, 8), -16777216);
  fld_x(Address(esp, 8));
  fsubp(1);
  jmp(B1_11);

  bind(B1_10);
  fld_d(ExternalAddress(zeros));    //0x00000000UL, 0x00000000UL
  fld_s(0);

  bind(B1_11);
  fld_s(0);
  lea(eax, Address(noreg, edx, Address::times_8));
  fld_s(3);
  lea(edx, Address(eax, edx, Address::times_4));
  fld_x(Address(_4onpi_31l, RelocationHolder::none).plus_disp(edx, Address::times_1));
  fmul(6);
  movl(Address(esp, 0), edx);
  fadda(2);
  fxch(2);
  fsuba(3);
  fxch(2);
  faddp(3);
  fxch(2);
  faddp(3);
  fld_x(Address(12 + _4onpi_31l, RelocationHolder::none).plus_disp(edx, Address::times_1));
  fmula(2);
  fld_s(2);
  fadd(2);
  fld_s(0);
  fxch(1);
  fsubra(3);
  fxch(3);
  fchs();
  faddp(4);
  fxch(3);
  faddp(4);
  fxch(2);
  fadd(3);
  fxch(2);
  fmul(5);
  fadda(2);
  fld_s(4);
  fld_x(Address(24 + _4onpi_31l, RelocationHolder::none).plus_disp(edx, Address::times_1));
  fmula(1);
  fxch(1);
  fadda(4);
  fxch(4);
  fstp_x(Address(esp, 8));
  movzwl(ebx, Address(esp, 16));
  andl(ebx, 32767);
  cmpl(ebx, 16415);
  jcc(Assembler::greaterEqual, B1_13);

  bind(B1_12);
  negl(ebx);
  addl(ebx, 30);
  movl(ecx, ebx);
  movl(eax, Address(esp, 12));
  shrl(eax);
  shll(eax);
  movl(Address(esp, 12), eax);
  movl(Address(esp, 8), 0);
  shrl(eax);
  jmp(B1_14);

  bind(B1_13);
  negl(ebx);
  addl(ebx, 30);
  movl(ecx, ebx);
  movl(edx, Address(esp, 8));
  shrl(edx);
  shll(edx);
  negl(ecx);
  movl(eax, Address(esp, 12));
  shll(eax);
  movl(ecx, ebx);
  movl(Address(esp, 8), edx);
  shrl(edx);
  orl(eax, edx);

  bind(B1_14);
  fld_x(Address(esp, 8));
  addl(eax, Address(esp, 4));
  fsubp(3);
  fmul(6);
  fld_s(4);
  movl(edx, eax);
  andl(edx, 1);
  fadd(3);
  movl(ecx, Address(esp, 0));
  fsuba(3);
  fxch(3);
  faddp(5);
  fld_s(1);
  fxch(3);
  fadd_d(Address(zero_none, RelocationHolder::none).plus_disp(edx, Address::times_8));
  fadda(3);
  fsub(3);
  faddp(2);
  fxch(1);
  faddp(4);
  fld_s(2);
  fadd(2);
  fsuba(2);
  fxch(3);
  faddp(2);
  fxch(1);
  faddp(3);
  fld_s(0);
  fadd(2);
  fsuba(2);
  fxch(1);
  faddp(2);
  fxch(1);
  faddp(2);
  fld_s(2);
  fld_x(Address(36 + _4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmula(1);
  fld_s(1);
  fadd(3);
  fsuba(3);
  fxch(2);
  faddp(3);
  fxch(2);
  faddp(3);
  fxch(1);
  fmul(4);
  fld_s(0);
  fadd(2);
  fsuba(2);
  fxch(1);
  faddp(2);
  fxch(1);
  faddp(2);
  fld_s(2);
  fld_x(Address(48 + _4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmula(1);
  fld_s(1);
  fadd(3);
  fsuba(3);
  fxch(2);
  faddp(3);
  fxch(2);
  faddp(3);
  fld_s(3);
  fxch(2);
  fmul(5);
  fld_x(Address(60 + _4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmula(3);
  fxch(3);
  faddp(1);
  fld_s(0);
  fadd(2);
  fsuba(2);
  fxch(1);
  faddp(2);
  fxch(1);
  faddp(3);
  fld_s(3);
  fxch(2);
  fmul(5);
  fld_x(Address(72 + _4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmula(3);
  fxch(3);
  faddp(1);
  fld_s(0);
  fadd(2);
  fsuba(2);
  fxch(1);
  faddp(2);
  fxch(1);
  faddp(3);
  fxch(1);
  fmulp(4);
  fld_x(Address(84 + _4onpi_31l, RelocationHolder::none).plus_disp(ecx, Address::times_1));
  fmulp(3);
  fxch(2);
  faddp(3);
  fld_s(2);
  fadd(2);
  fld_d(ExternalAddress(TWO_32H));    //0x00000000UL, 0x41f80000UL
  fmul(1);
  fadda(1);
  fsubp(1);
  fsuba(2);
  fxch(3);
  faddp(2);
  faddp(1);
  fld_d(ExternalAddress(pi04_2d));    //0x54400000UL, 0x3fe921fbUL
  fld_s(0);
  fmul(2);
  fxch(2);
  fadd(3);
  fxch(1);
  fmulp(3);
  fmul_d(as_Address(ExternalAddress(8 + pi04_2d)));    //0x1a626331UL, 0x3dc0b461UL
  faddp(1);

  bind(B1_15);
  fld_d(ExternalAddress(TWO_12H));    //0x00000000UL, 0x40b80000UL
  fld_s(2);
  fadd(2);
  fmula(1);
  fstp_x(Address(esp, 8));
  fld_x(Address(esp, 8));
  fadd(1);
  fsubrp(1);
  fst_d(Address(esi, 0));
  fsubp(2);
  faddp(1);
  fstp_d(Address(esi, 8));
  addl(esp, 20);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);
}


ATTRIBUTE_ALIGNED(16) static const jushort _SP[] =
{
    0xaaab, 0xaaaa, 0xaaaa, 0xaaaa, 0xbffc, 0x0000, 0x8887, 0x8888, 0x8888, 0x8888,
    0x3ff8, 0x0000, 0xc527, 0x0d00, 0x00d0, 0xd00d, 0xbff2, 0x0000, 0x45f6, 0xb616,
    0x1d2a, 0xb8ef, 0x3fec, 0x0000, 0x825b, 0x3997, 0x2b3f, 0xd732, 0xbfe5, 0x0000,
    0xbf33, 0x8bb4, 0x2fda, 0xb092, 0x3fde, 0x0000, 0x44a6, 0xed1a, 0x29ef, 0xd73e,
    0xbfd6, 0x0000, 0x8610, 0x307f, 0x62a1, 0xc921, 0x3fce, 0x0000
};

ATTRIBUTE_ALIGNED(16) static const jushort _CP[] =
{
    0x0000, 0x0000, 0x0000, 0x8000, 0xbffe, 0x0000, 0xaaa5, 0xaaaa, 0xaaaa, 0xaaaa,
    0x3ffa, 0x0000, 0x9c2f, 0x0b60, 0x60b6, 0xb60b, 0xbff5, 0x0000, 0xf024, 0x0cac,
    0x00d0, 0xd00d, 0x3fef, 0x0000, 0x03fe, 0x3f65, 0x7dbb, 0x93f2, 0xbfe9, 0x0000,
    0xd84d, 0xadee, 0xc698, 0x8f76, 0x3fe2, 0x0000, 0xdaba, 0xfe79, 0xea36, 0xc9c9,
    0xbfda, 0x0000, 0x3ac6, 0x0ba0, 0x07ce, 0xd585, 0x3fd2, 0x0000
};

void MacroAssembler::libm_sincos_huge(XMMRegister xmm0, XMMRegister xmm1, Register eax, Register ecx, Register edx, Register ebx, Register esi, Register edi, Register ebp, Register esp) {
  Label B1_1, B1_2, B1_3, B1_4, B1_5, B1_6, B1_7, B1_8, B1_9, B1_10, B1_11, B1_12;
  Label B1_13, B1_14, B1_15, B1_16, B1_17, B1_18, B1_19, B1_20, B1_21, B1_22, B1_23;
  Label B1_24, B1_25, B1_26, B1_27, B1_28, B1_29, B1_30, B1_31, B1_32, B1_33, B1_34;
  Label B1_35, B1_36, B1_37, B1_38, B1_39, B1_40, B1_41, B1_42, B1_43, B1_46;

  assert_different_registers(ebx, eax, ecx, edx, esi, edi, ebp, esp);

  address CP = (address)_CP;
  address SP = (address)_SP;

  bind(B1_1);
  push(ebp);
  movl(ebp, esp);
  andl(esp, -64);
  push(esi);
  push(edi);
  push(ebx);
  subl(esp, 52);
  movl(eax, Address(ebp, 16));
  movl(edx, Address(ebp, 20));
  movl(Address(esp, 32), eax);
  movl(Address(esp, 36), edx);

  bind(B1_2);
  fnstcw(Address(esp, 30));

  bind(B1_3);
  movsd(xmm1, Address(ebp, 8));
  movl(esi, Address(ebp, 12));
  movl(eax, esi);
  andl(eax, 2147483647);
  andps(xmm1, ExternalAddress(L_2IL0FLOATPACKET_0));    //0xffffffffUL, 0x7fffffffUL, 0x00000000UL, 0x00000000UL
  shrl(esi, 31);
  movl(Address(esp, 40), eax);
  cmpl(eax, 1104150528);
  movsd(Address(ebp, 8), xmm1);
  jcc(Assembler::aboveEqual, B1_11);

  bind(B1_4);
  movsd(xmm0, ExternalAddress(PI4_INV));    //0x6dc9c883UL, 0x3ff45f30UL
  mulsd(xmm0, xmm1);
  movzwl(edx, Address(esp, 30));
  movl(eax, edx);
  andl(eax, 768);
  movsd(Address(esp, 0), xmm0);
  cmpl(eax, 768);
  jcc(Assembler::equal, B1_42);

  bind(B1_5);
  orl(edx, -64768);
  movw(Address(esp, 28), edx);

  bind(B1_6);
  fldcw(Address(esp, 28));

  bind(B1_7);
  movsd(xmm1, Address(ebp, 8));
  movl(ebx, 1);

  bind(B1_8);
  movl(Address(esp, 12), ebx);
  movl(ebx, Address(esp, 4));
  movl(eax, ebx);
  movl(Address(esp, 8), esi);
  movl(esi, ebx);
  shrl(esi, 20);
  andl(eax, 1048575);
  movl(ecx, esi);
  orl(eax, 1048576);
  negl(ecx);
  movl(edx, eax);
  addl(ecx, 19);
  addl(esi, 13);
  movl(Address(esp, 24), ecx);
  shrl(edx);
  movl(ecx, esi);
  shll(eax);
  movl(ecx, Address(esp, 24));
  movl(esi, Address(esp, 0));
  shrl(esi);
  orl(eax, esi);
  cmpl(ebx, 1094713344);
  movsd(Address(esp, 16), xmm1);
  fld_d(Address(esp, 16));
  cmov32(Assembler::below, eax, edx);
  movl(esi, Address(esp, 8));
  lea(edx, Address(eax, 1));
  movl(ebx, edx);
  andl(ebx, -2);
  movl(Address(esp, 16), ebx);
  fild_s(Address(esp, 16));
  movl(ebx, Address(esp, 12));
  cmpl(Address(esp, 40), 1094713344);
  jcc(Assembler::aboveEqual, B1_10);

  bind(B1_9);
  fld_d(ExternalAddress(PI4X3));    //0x54443000UL, 0xbfe921fbUL
  fmul(1);
  faddp(2);
  fld_d(ExternalAddress(PI4X3 + 8));    //0x3b39a000UL, 0x3d373dcbUL
  fmul(1);
  faddp(2);
  fld_d(ExternalAddress(PI4X3 + 16));    //0xe0e68948UL, 0xba845c06UL
  fmulp(1);
  faddp(1);
  jmp(B1_17);

  bind(B1_10);
  fld_d(ExternalAddress(PI4X4));    //0x54400000UL, 0xbfe921fbUL
  fmul(1);
  faddp(2);
  fld_d(ExternalAddress(PI4X4 + 8));    //0x1a600000UL, 0xbdc0b461UL
  fmul(1);
  faddp(2);
  fld_d(ExternalAddress(PI4X4 + 16));    //0x2e000000UL, 0xbb93198aUL
  fmul(1);
  faddp(2);
  fld_d(ExternalAddress(PI4X4 + 24));    //0x252049c1UL, 0xb96b839aUL
  fmulp(1);
  faddp(1);
  jmp(B1_17);

  bind(B1_11);
  movzwl(edx, Address(esp, 30));
  movl(eax, edx);
  andl(eax, 768);
  cmpl(eax, 768);
  jcc(Assembler::equal, B1_43);
  bind(B1_12);
  orl(edx, -64768);
  movw(Address(esp, 28), edx);

  bind(B1_13);
  fldcw(Address(esp, 28));

  bind(B1_14);
  movsd(xmm1, Address(ebp, 8));
  movl(ebx, 1);

  bind(B1_15);
  movsd(Address(esp, 16), xmm1);
  fld_d(Address(esp, 16));
  addl(esp, -32);
  lea(eax, Address(esp, 32));
  fstp_x(Address(esp, 0));
  movl(Address(esp, 12), 0);
  movl(Address(esp, 16), eax);
  call(RuntimeAddress(CAST_FROM_FN_PTR(address, StubRoutines::dlibm_reduce_pi04l())));

  bind(B1_46);
  addl(esp, 32);

  bind(B1_16);
  fld_d(Address(esp, 0));
  lea(edx, Address(eax, 1));
  fld_d(Address(esp, 8));
  faddp(1);

  bind(B1_17);
  movl(ecx, edx);
  addl(eax, 3);
  shrl(ecx, 2);
  andl(ecx, 1);
  shrl(eax, 2);
  xorl(esi, ecx);
  movl(ecx, Address(esp, 36));
  andl(eax, 1);
  andl(ecx, 3);
  cmpl(ecx, 3);
  jcc(Assembler::notEqual, B1_25);

  bind(B1_18);
  fld_x(ExternalAddress(84 + SP));    //0x8610, 0x307f, 0x62
  fld_s(1);
  fmul((2));
  testb(edx, 2);
  fmula((1));
  fld_x(ExternalAddress(72 + SP));    //0x44a6, 0xed1a, 0x29
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(60 + SP));    //0xbf33, 0x8bb4, 0x2f
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(48 + SP));    //0x825b, 0x3997, 0x2b
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(36 + SP));    //0x45f6, 0xb616, 0x1d
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(24 + SP));    //0xc527, 0x0d00, 0x00
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(12 + SP));    //0x8887, 0x8888, 0x88
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(SP));    //0xaaab, 0xaaaa, 0xaa
  faddp(2);
  fmula(1);
  fld_x(ExternalAddress(84 + CP));    //0x3ac6, 0x0ba0, 0x07
  fmul(1);
  fld_x(ExternalAddress(72 + CP));    //0xdaba, 0xfe79, 0xea
  faddp(1);
  fmul(1);
  fld_x(ExternalAddress(62 + CP));    //0xd84d, 0xadee, 0xc6
  faddp(1);
  fmul(1);
  fld_x(ExternalAddress(48 + CP));    //0x03fe, 0x3f65, 0x7d
  faddp(1);
  fmul(1);
  fld_x(ExternalAddress(36 + CP));    //0xf024, 0x0cac, 0x00
  faddp(1);
  fmul(1);
  fld_x(ExternalAddress(24 + CP));    //0x9c2f, 0x0b60, 0x60
  faddp(1);
  fmul(1);
  fld_x(ExternalAddress(12 + CP));    //0xaaa5, 0xaaaa, 0xaa
  faddp(1);
  fmul(1);
  fld_x(ExternalAddress(CP));    //0x0000, 0x0000, 0x00
  faddp(1);
  fmulp(1);
  fld_d(Address(ONES, RelocationHolder::none).plus_disp(esi, Address::times_8));
  fld_d(Address(ONES, RelocationHolder::none).plus_disp(eax, Address::times_8));
  jcc(Assembler::equal, B1_22);

  bind(B1_19);
  fmulp(4);
  testl(ebx, ebx);
  fxch(2);
  fmul(3);
  movl(eax, Address(esp, 2));
  faddp(3);
  fxch(2);
  fstp_d(Address(eax, 0));
  fmula(1);
  faddp(1);
  fstp_d(Address(eax, 8));
  jcc(Assembler::equal, B1_21);

  bind(B1_20);
  fldcw(Address(esp, 30));

  bind(B1_21);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);

  bind(B1_22);
  fxch(1);
  fmulp(4);
  testl(ebx, ebx);
  fxch(2);
  fmul(3);
  movl(eax, Address(esp, 32));
  faddp(3);
  fxch(2);
  fstp_d(Address(eax, 8));
  fmula(1);
  faddp(1);
  fstp_d(Address(eax, 0));
  jcc(Assembler::equal, B1_24);

  bind(B1_23);
  fldcw(Address(esp, 30));

  bind(B1_24);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);

  bind(B1_25);
  testb(Address(esp, 36), 2);
  jcc(Assembler::equal, B1_33);

  bind(B1_26);
  fld_s(0);
  testb(edx, 2);
  fmul(1);
  fld_s(0);
  fmul(1);
  jcc(Assembler::equal, B1_30);

  bind(B1_27);
  fstp_d(2);
  fld_x(ExternalAddress(84 + CP));    //0x3ac6, 0x0ba0, 0x07
  testl(ebx, ebx);
  fmul(2);
  fld_x(ExternalAddress(72 + CP));    //0xdaba, 0xfe79, 0xea
  fmul(3);
  fld_x(ExternalAddress(60 + CP));    //0xd84d, 0xadee, 0xc6
  movl(eax, Address(rsp, 32));
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(48 + CP));    //0x03fe, 0x3f65, 0x7d
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(36 + CP));    //0xf024, 0x0cac, 0x00
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(24 + CP));    //0x9c2f, 0x0b60, 0x60
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(12 + CP));    //0xaaa5, 0xaaaa, 0xaa
  faddp(2);
  fxch(1);
  fmulp(3);
  fld_x(ExternalAddress(CP));    //0x0000, 0x0000, 0x00
  faddp(1);
  fmulp(1);
  faddp(1);
  fld_d(Address(ONES, RelocationHolder::none).plus_disp(rsi, Address::times_8));
  fmula(1);
  faddp(1);
  fstp_d(Address(eax, 8));
  jcc(Assembler::equal, B1_29);

  bind(B1_28);
  fldcw(Address(esp, 30));

  bind(B1_29);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);

  bind(B1_30);
  fld_x(ExternalAddress(84 + SP));    //0x8610, 0x307f, 0x62
  testl(ebx, ebx);
  fmul(1);
  fld_x(ExternalAddress(72 + SP));    //0x44a6, 0xed1a, 0x29
  fmul(2);
  fld_x(ExternalAddress(60 + SP));    //0xbf33, 0x8bb4, 0x2f
  movl(eax, Address(rsp, 32));
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(48 + SP));    //0x825b, 0x3997, 0x2b
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(36 + SP));    //0x45f6, 0xb616, 0x1d
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(24 + SP));    //0xc527, 0x0d00, 0x00
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(12 + SP));    //0x8887, 0x8888, 0x88
  faddp(2);
  fxch(1);
  fmulp(2);
  fld_x(ExternalAddress(SP));    //0xaaab, 0xaaaa, 0xaa
  faddp(1);
  fmulp(2);
  faddp(1);
  fld_d(Address(ONES, RelocationHolder::none).plus_disp(rsi, Address::times_8));
  fmulp(2);
  fmul(1);
  faddp(1);
  fstp_d(Address(eax, 8));
  jcc(Assembler::equal, B1_32);

  bind(B1_31);
  fldcw(Address(esp, 30));

  bind(B1_32);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);

  bind(B1_33);
  testb(Address(esp, 36), 1);
  jcc(Assembler::equal, B1_41);

  bind(B1_34);
  fld_s(0);
  testb(edx, 2);
  fmul(1);
  fld_s(0);
  fmul(1);
  jcc(Assembler::equal, B1_38);

  bind(B1_35);
  fld_x(ExternalAddress(84 + SP));    //0x8610, 0x307f, 0x62
  testl(ebx, ebx);
  fmul(1);
  fld_x(ExternalAddress(72 + SP));    //0x44a6, 0xed1a, 0x29
  fmul(2);
  fld_x(ExternalAddress(60 + SP));    //0xbf33, 0x8bb4, 0x2f
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(48 + SP));    //0x825b, 0x3997, 0x2b
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(36 + SP));    //0x45f6, 0xb616, 0x1d
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(24 + SP));    //0xc527, 0x0d00, 0x00
  faddp(2);
  fxch(1);
  fmul(2);
  fld_x(ExternalAddress(12 + SP));    //0x8887, 0x8888, 0x88
  faddp(2);
  fxch(1);
  fmulp(2);
  fld_x(ExternalAddress(SP));    //0xaaab, 0xaaaa, 0xaa
  faddp(1);
  fmulp(2);
  faddp(1);
  fld_d(Address(ONES, RelocationHolder::none).plus_disp(eax, Address::times_8));
  fmulp(2);
  fmul(1);
  movl(eax, Address(esp, 32));
  faddp(1);
  fstp_d(Address(eax, 0));
  jcc(Assembler::equal, B1_37);

  bind(B1_36);
  fldcw(Address(esp, 30));

  bind(B1_37);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);

  bind(B1_38);
  fstp_d(2);
  fld_x(ExternalAddress(84 + CP));    //0x3ac6, 0x0ba0, 0x07
  testl(ebx, ebx);
  fmul(2);
  fld_x(ExternalAddress(72 + CP));    //0xdaba, 0xfe79, 0xea
  fmul(3);
  fld_x(ExternalAddress(60 + CP));    //0xd84d, 0xadee, 0xc6
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(48 + CP));    //0x03fe, 0x3f65, 0x7d
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(36 + CP));    //0xf024, 0x0cac, 0x00
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(24 + CP));    //0x9c2f, 0x0b60, 0x60
  faddp(2);
  fxch(1);
  fmul(3);
  fld_x(ExternalAddress(12 + CP));    //0xaaa5, 0xaaaa, 0xaa
  faddp(2);
  fxch(1);
  fmulp(3);
  fld_x(ExternalAddress(CP));    //0x0000, 0x0000, 0x00
  faddp(1);
  fmulp(1);
  faddp(1);
  fld_d(Address(ONES, RelocationHolder::none).plus_disp(eax, Address::times_8));
  fmula(1);
  movl(eax, Address(esp, 32));
  faddp(1);
  fstp_d(Address(eax, 0));
  jcc(Assembler::equal, B1_40);

  bind(B1_39);
  fldcw(Address(esp, 30));
  bind(B1_40);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);
  bind(B1_41);
  fstp_d(0);
  addl(esp, 52);
  pop(ebx);
  pop(edi);
  pop(esi);
  movl(esp, ebp);
  pop(ebp);
  ret(0);
  bind(B1_42);
  xorl(ebx, ebx);
  jmp(B1_8);
  bind(B1_43);
  xorl(ebx, ebx);
  jmp(B1_15);
}

ATTRIBUTE_ALIGNED(16) static const juint _static_const_table_sin[] =
{
    0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
    0x00000000UL, 0x00000000UL, 0x3ff00000UL, 0x176d6d31UL, 0xbf73b92eUL,
    0xbc29b42cUL, 0x3fb917a6UL, 0xe0000000UL, 0xbc3e2718UL, 0x00000000UL,
    0x3ff00000UL, 0x011469fbUL, 0xbf93ad06UL, 0x3c69a60bUL, 0x3fc8f8b8UL,
    0xc0000000UL, 0xbc626d19UL, 0x00000000UL, 0x3ff00000UL, 0x939d225aUL,
    0xbfa60beaUL, 0x2ed59f06UL, 0x3fd29406UL, 0xa0000000UL, 0xbc75d28dUL,
    0x00000000UL, 0x3ff00000UL, 0x866b95cfUL, 0xbfb37ca1UL, 0xa6aea963UL,
    0x3fd87de2UL, 0xe0000000UL, 0xbc672cedUL, 0x00000000UL, 0x3ff00000UL,
    0x73fa1279UL, 0xbfbe3a68UL, 0x3806f63bUL, 0x3fde2b5dUL, 0x20000000UL,
    0x3c5e0d89UL, 0x00000000UL, 0x3ff00000UL, 0x5bc57974UL, 0xbfc59267UL,
    0x39ae68c8UL, 0x3fe1c73bUL, 0x20000000UL, 0x3c8b25ddUL, 0x00000000UL,
    0x3ff00000UL, 0x53aba2fdUL, 0xbfcd0dfeUL, 0x25091dd6UL, 0x3fe44cf3UL,
    0x20000000UL, 0x3c68076aUL, 0x00000000UL, 0x3ff00000UL, 0x99fcef32UL,
    0x3fca8279UL, 0x667f3bcdUL, 0x3fe6a09eUL, 0x20000000UL, 0xbc8bdd34UL,
    0x00000000UL, 0x3fe00000UL, 0x94247758UL, 0x3fc133ccUL, 0x6b151741UL,
    0x3fe8bc80UL, 0x20000000UL, 0xbc82c5e1UL, 0x00000000UL, 0x3fe00000UL,
    0x9ae68c87UL, 0x3fac73b3UL, 0x290ea1a3UL, 0x3fea9b66UL, 0xe0000000UL,
    0x3c39f630UL, 0x00000000UL, 0x3fe00000UL, 0x7f909c4eUL, 0xbf9d4a2cUL,
    0xf180bdb1UL, 0x3fec38b2UL, 0x80000000UL, 0xbc76e0b1UL, 0x00000000UL,
    0x3fe00000UL, 0x65455a75UL, 0xbfbe0875UL, 0xcf328d46UL, 0x3fed906bUL,
    0x20000000UL, 0x3c7457e6UL, 0x00000000UL, 0x3fe00000UL, 0x76acf82dUL,
    0x3fa4a031UL, 0x56c62ddaUL, 0x3fee9f41UL, 0xe0000000UL, 0x3c8760b1UL,
    0x00000000UL, 0x3fd00000UL, 0x0e5967d5UL, 0xbfac1d1fUL, 0xcff75cb0UL,
    0x3fef6297UL, 0x20000000UL, 0x3c756217UL, 0x00000000UL, 0x3fd00000UL,
    0x0f592f50UL, 0xbf9ba165UL, 0xa3d12526UL, 0x3fefd88dUL, 0x40000000UL,
    0xbc887df6UL, 0x00000000UL, 0x3fc00000UL, 0x00000000UL, 0x00000000UL,
    0x00000000UL, 0x3ff00000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
    0x00000000UL, 0x0f592f50UL, 0x3f9ba165UL, 0xa3d12526UL, 0x3fefd88dUL,
    0x40000000UL, 0xbc887df6UL, 0x00000000UL, 0xbfc00000UL, 0x0e5967d5UL,
    0x3fac1d1fUL, 0xcff75cb0UL, 0x3fef6297UL, 0x20000000UL, 0x3c756217UL,
    0x00000000UL, 0xbfd00000UL, 0x76acf82dUL, 0xbfa4a031UL, 0x56c62ddaUL,
    0x3fee9f41UL, 0xe0000000UL, 0x3c8760b1UL, 0x00000000UL, 0xbfd00000UL,
    0x65455a75UL, 0x3fbe0875UL, 0xcf328d46UL, 0x3fed906bUL, 0x20000000UL,
    0x3c7457e6UL, 0x00000000UL, 0xbfe00000UL, 0x7f909c4eUL, 0x3f9d4a2cUL,
    0xf180bdb1UL, 0x3fec38b2UL, 0x80000000UL, 0xbc76e0b1UL, 0x00000000UL,
    0xbfe00000UL, 0x9ae68c87UL, 0xbfac73b3UL, 0x290ea1a3UL, 0x3fea9b66UL,
    0xe0000000UL, 0x3c39f630UL, 0x00000000UL, 0xbfe00000UL, 0x94247758UL,
    0xbfc133ccUL, 0x6b151741UL, 0x3fe8bc80UL, 0x20000000UL, 0xbc82c5e1UL,
    0x00000000UL, 0xbfe00000UL, 0x99fcef32UL, 0xbfca8279UL, 0x667f3bcdUL,
    0x3fe6a09eUL, 0x20000000UL, 0xbc8bdd34UL, 0x00000000UL, 0xbfe00000UL,
    0x53aba2fdUL, 0x3fcd0dfeUL, 0x25091dd6UL, 0x3fe44cf3UL, 0x20000000UL,
    0x3c68076aUL, 0x00000000UL, 0xbff00000UL, 0x5bc57974UL, 0x3fc59267UL,
    0x39ae68c8UL, 0x3fe1c73bUL, 0x20000000UL, 0x3c8b25ddUL, 0x00000000UL,
    0xbff00000UL, 0x73fa1279UL, 0x3fbe3a68UL, 0x3806f63bUL, 0x3fde2b5dUL,
    0x20000000UL, 0x3c5e0d89UL, 0x00000000UL, 0xbff00000UL, 0x866b95cfUL,
    0x3fb37ca1UL, 0xa6aea963UL, 0x3fd87de2UL, 0xe0000000UL, 0xbc672cedUL,
    0x00000000UL, 0xbff00000UL, 0x939d225aUL, 0x3fa60beaUL, 0x2ed59f06UL,
    0x3fd29406UL, 0xa0000000UL, 0xbc75d28dUL, 0x00000000UL, 0xbff00000UL,
    0x011469fbUL, 0x3f93ad06UL, 0x3c69a60bUL, 0x3fc8f8b8UL, 0xc0000000UL,
    0xbc626d19UL, 0x00000000UL, 0xbff00000UL, 0x176d6d31UL, 0x3f73b92eUL,
    0xbc29b42cUL, 0x3fb917a6UL, 0xe0000000UL, 0xbc3e2718UL, 0x00000000UL,
    0xbff00000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
    0x00000000UL, 0x00000000UL, 0x00000000UL, 0xbff00000UL, 0x176d6d31UL,
    0x3f73b92eUL, 0xbc29b42cUL, 0xbfb917a6UL, 0xe0000000UL, 0x3c3e2718UL,
    0x00000000UL, 0xbff00000UL, 0x011469fbUL, 0x3f93ad06UL, 0x3c69a60bUL,
    0xbfc8f8b8UL, 0xc0000000UL, 0x3c626d19UL, 0x00000000UL, 0xbff00000UL,
    0x939d225aUL, 0x3fa60beaUL, 0x2ed59f06UL, 0xbfd29406UL, 0xa0000000UL,
    0x3c75d28dUL, 0x00000000UL, 0xbff00000UL, 0x866b95cfUL, 0x3fb37ca1UL,
    0xa6aea963UL, 0xbfd87de2UL, 0xe0000000UL, 0x3c672cedUL, 0x00000000UL,
    0xbff00000UL, 0x73fa1279UL, 0x3fbe3a68UL, 0x3806f63bUL, 0xbfde2b5dUL,
    0x20000000UL, 0xbc5e0d89UL, 0x00000000UL, 0xbff00000UL, 0x5bc57974UL,
    0x3fc59267UL, 0x39ae68c8UL, 0xbfe1c73bUL, 0x20000000UL, 0xbc8b25ddUL,
    0x00000000UL, 0xbff00000UL, 0x53aba2fdUL, 0x3fcd0dfeUL, 0x25091dd6UL,
    0xbfe44cf3UL, 0x20000000UL, 0xbc68076aUL, 0x00000000UL, 0xbff00000UL,
    0x99fcef32UL, 0xbfca8279UL, 0x667f3bcdUL, 0xbfe6a09eUL, 0x20000000UL,
    0x3c8bdd34UL, 0x00000000UL, 0xbfe00000UL, 0x94247758UL, 0xbfc133ccUL,
    0x6b151741UL, 0xbfe8bc80UL, 0x20000000UL, 0x3c82c5e1UL, 0x00000000UL,
    0xbfe00000UL, 0x9ae68c87UL, 0xbfac73b3UL, 0x290ea1a3UL, 0xbfea9b66UL,
    0xe0000000UL, 0xbc39f630UL, 0x00000000UL, 0xbfe00000UL, 0x7f909c4eUL,
    0x3f9d4a2cUL, 0xf180bdb1UL, 0xbfec38b2UL, 0x80000000UL, 0x3c76e0b1UL,
    0x00000000UL, 0xbfe00000UL, 0x65455a75UL, 0x3fbe0875UL, 0xcf328d46UL,
    0xbfed906bUL, 0x20000000UL, 0xbc7457e6UL, 0x00000000UL, 0xbfe00000UL,
    0x76acf82dUL, 0xbfa4a031UL, 0x56c62ddaUL, 0xbfee9f41UL, 0xe0000000UL,
    0xbc8760b1UL, 0x00000000UL, 0xbfd00000UL, 0x0e5967d5UL, 0x3fac1d1fUL,
    0xcff75cb0UL, 0xbfef6297UL, 0x20000000UL, 0xbc756217UL, 0x00000000UL,
    0xbfd00000UL, 0x0f592f50UL, 0x3f9ba165UL, 0xa3d12526UL, 0xbfefd88dUL,
    0x40000000UL, 0x3c887df6UL, 0x00000000UL, 0xbfc00000UL, 0x00000000UL,
    0x00000000UL, 0x00000000UL, 0xbff00000UL, 0x00000000UL, 0x00000000UL,
    0x00000000UL, 0x00000000UL, 0x0f592f50UL, 0xbf9ba165UL, 0xa3d12526UL,
    0xbfefd88dUL, 0x40000000UL, 0x3c887df6UL, 0x00000000UL, 0x3fc00000UL,
    0x0e5967d5UL, 0xbfac1d1fUL, 0xcff75cb0UL, 0xbfef6297UL, 0x20000000UL,
    0xbc756217UL, 0x00000000UL, 0x3fd00000UL, 0x76acf82dUL, 0x3fa4a031UL,
    0x56c62ddaUL, 0xbfee9f41UL, 0xe0000000UL, 0xbc8760b1UL, 0x00000000UL,
    0x3fd00000UL, 0x65455a75UL, 0xbfbe0875UL, 0xcf328d46UL, 0xbfed906bUL,
    0x20000000UL, 0xbc7457e6UL, 0x00000000UL, 0x3fe00000UL, 0x7f909c4eUL,
    0xbf9d4a2cUL, 0xf180bdb1UL, 0xbfec38b2UL, 0x80000000UL, 0x3c76e0b1UL,
    0x00000000UL, 0x3fe00000UL, 0x9ae68c87UL, 0x3fac73b3UL, 0x290ea1a3UL,
    0xbfea9b66UL, 0xe0000000UL, 0xbc39f630UL, 0x00000000UL, 0x3fe00000UL,
    0x94247758UL, 0x3fc133ccUL, 0x6b151741UL, 0xbfe8bc80UL, 0x20000000UL,
    0x3c82c5e1UL, 0x00000000UL, 0x3fe00000UL, 0x99fcef32UL, 0x3fca8279UL,
    0x667f3bcdUL, 0xbfe6a09eUL, 0x20000000UL, 0x3c8bdd34UL, 0x00000000UL,
    0x3fe00000UL, 0x53aba2fdUL, 0xbfcd0dfeUL, 0x25091dd6UL, 0xbfe44cf3UL,
    0x20000000UL, 0xbc68076aUL, 0x00000000UL, 0x3ff00000UL, 0x5bc57974UL,
    0xbfc59267UL, 0x39ae68c8UL, 0xbfe1c73bUL, 0x20000000UL, 0xbc8b25ddUL,
    0x00000000UL, 0x3ff00000UL, 0x73fa1279UL, 0xbfbe3a68UL, 0x3806f63bUL,
    0xbfde2b5dUL, 0x20000000UL, 0xbc5e0d89UL, 0x00000000UL, 0x3ff00000UL,
    0x866b95cfUL, 0xbfb37ca1UL, 0xa6aea963UL, 0xbfd87de2UL, 0xe0000000UL,
    0x3c672cedUL, 0x00000000UL, 0x3ff00000UL, 0x939d225aUL, 0xbfa60beaUL,
    0x2ed59f06UL, 0xbfd29406UL, 0xa0000000UL, 0x3c75d28dUL, 0x00000000UL,
    0x3ff00000UL, 0x011469fbUL, 0xbf93ad06UL, 0x3c69a60bUL, 0xbfc8f8b8UL,
    0xc0000000UL, 0x3c626d19UL, 0x00000000UL, 0x3ff00000UL, 0x176d6d31UL,
    0xbf73b92eUL, 0xbc29b42cUL, 0xbfb917a6UL, 0xe0000000UL, 0x3c3e2718UL,
    0x00000000UL, 0x3ff00000UL, 0x55555555UL, 0xbfc55555UL, 0x00000000UL,
    0xbfe00000UL, 0x11111111UL, 0x3f811111UL, 0x55555555UL, 0x3fa55555UL,
    0x1a01a01aUL, 0xbf2a01a0UL, 0x16c16c17UL, 0xbf56c16cUL, 0xa556c734UL,
    0x3ec71de3UL, 0x1a01a01aUL, 0x3efa01a0UL, 0x1a600000UL, 0x3d90b461UL,
    0x1a600000UL, 0x3d90b461UL, 0x54400000UL, 0x3fb921fbUL, 0x00000000UL,
    0x00000000UL, 0x2e037073UL, 0x3b63198aUL, 0x00000000UL, 0x00000000UL,
    0x6dc9c883UL, 0x40245f30UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
    0x43380000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x43600000UL,
    0x00000000UL, 0x00000000UL, 0x00000000UL, 0x3c800000UL, 0x00000000UL,
    0x00000000UL, 0xffffffffUL, 0x3fefffffUL, 0x00000000UL, 0x00000000UL,
    0x00000000UL, 0x80000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
    0x80000000UL, 0x00000000UL, 0x80000000UL, 0x00000000UL, 0x3fe00000UL,
    0x00000000UL, 0x3fe00000UL
};

void MacroAssembler::fast_sin(XMMRegister xmm0, XMMRegister xmm1, XMMRegister xmm2, XMMRegister xmm3,
                              XMMRegister xmm4, XMMRegister xmm5, XMMRegister xmm6, XMMRegister xmm7,
                              Register eax, Register ebx, Register edx) {

  Label L_2TAG_PACKET_0_0_2, L_2TAG_PACKET_1_0_2, L_2TAG_PACKET_2_0_2, L_2TAG_PACKET_3_0_2;
  Label L_2TAG_PACKET_4_0_2;

  assert_different_registers(eax, ebx, edx);

  address static_const_table_sin = (address)_static_const_table_sin;

  subl(rsp, 120);
  movl(Address(rsp, 56), ebx);
  lea(ebx, ExternalAddress(static_const_table_sin));
  movsd(xmm0, Address(rsp, 128));
  pextrw(eax, xmm0, 3);
  andl(eax, 32767);
  subl(eax, 12336);
  cmpl(eax, 4293);
  jcc(Assembler::above, L_2TAG_PACKET_0_0_2);
  movsd(xmm1, Address(ebx, 2160));
  mulsd(xmm1, xmm0);
  movsd(xmm5, Address(ebx, 2272));
  movdqu(xmm4, Address(ebx, 2256));
  pand(xmm4, xmm0);
  por(xmm5, xmm4);
  movsd(xmm3, Address(ebx, 2128));
  movdqu(xmm2, Address(ebx, 2112));
  addpd(xmm1, xmm5);
  cvttsd2sil(edx, xmm1);
  cvtsi2sdl(xmm1, edx);
  mulsd(xmm3, xmm1);
  unpcklpd(xmm1, xmm1);
  addl(edx, 1865216);
  movdqu(xmm4, xmm0);
  andl(edx, 63);
  movdqu(xmm5, Address(ebx, 2096));
  lea(eax, Address(ebx, 0));
  shll(edx, 5);
  addl(eax, edx);
  mulpd(xmm2, xmm1);
  subsd(xmm0, xmm3);
  mulsd(xmm1, Address(ebx, 2144));
  subsd(xmm4, xmm3);
  movsd(xmm7, Address(eax, 8));
  unpcklpd(xmm0, xmm0);
  movapd(xmm3, xmm4);
  subsd(xmm4, xmm2);
  mulpd(xmm5, xmm0);
  subpd(xmm0, xmm2);
  movdqu(xmm6, Address(ebx, 2064));
  mulsd(xmm7, xmm4);
  subsd(xmm3, xmm4);
  mulpd(xmm5, xmm0);
  mulpd(xmm0, xmm0);
  subsd(xmm3, xmm2);
  movdqu(xmm2, Address(eax, 0));
  subsd(xmm1, xmm3);
  movsd(xmm3, Address(eax, 24));
  addsd(xmm2, xmm3);
  subsd(xmm7, xmm2);
  mulsd(xmm2, xmm4);
  mulpd(xmm6, xmm0);
  mulsd(xmm3, xmm4);
  mulpd(xmm2, xmm0);
  mulpd(xmm0, xmm0);
  addpd(xmm5, Address(ebx, 2080));
  mulsd(xmm4, Address(eax, 0));
  addpd(xmm6, Address(ebx, 2048));
  mulpd(xmm5, xmm0);
  movapd(xmm0, xmm3);
  addsd(xmm3, Address(eax, 8));
  mulpd(xmm1, xmm7);
  movapd(xmm7, xmm4);
  addsd(xmm4, xmm3);
  addpd(xmm6, xmm5);
  movsd(xmm5, Address(eax, 8));
  subsd(xmm5, xmm3);
  subsd(xmm3, xmm4);
  addsd(xmm1, Address(eax, 16));
  mulpd(xmm6, xmm2);
  addsd(xmm5, xmm0);
  addsd(xmm3, xmm7);
  addsd(xmm1, xmm5);
  addsd(xmm1, xmm3);
  addsd(xmm1, xmm6);
  unpckhpd(xmm6, xmm6);
  addsd(xmm1, xmm6);
  addsd(xmm4, xmm1);
  movsd(Address(rsp, 0), xmm4);
  fld_d(Address(rsp, 0));
  jmp(L_2TAG_PACKET_1_0_2);

  bind(L_2TAG_PACKET_0_0_2);
  jcc(Assembler::greater, L_2TAG_PACKET_2_0_2);
  shrl(eax, 4);
  cmpl(eax, 268434685);
  jcc(Assembler::notEqual, L_2TAG_PACKET_3_0_2);
  movsd(Address(rsp, 0), xmm0);
  fld_d(Address(rsp, 0));
  jmp(L_2TAG_PACKET_1_0_2);

  bind(L_2TAG_PACKET_3_0_2);
  movsd(xmm3, Address(ebx, 2192));
  mulsd(xmm3, xmm0);
  subsd(xmm3, xmm0);
  mulsd(xmm3, Address(ebx, 2208));
  movsd(Address(rsp, 0), xmm0);
  fld_d(Address(rsp, 0));
  jmp(L_2TAG_PACKET_1_0_2);

  bind(L_2TAG_PACKET_2_0_2);
  movl(eax, Address(rsp, 132));
  andl(eax, 2146435072);
  cmpl(eax, 2146435072);
  jcc(Assembler::equal, L_2TAG_PACKET_4_0_2);
  subl(rsp, 32);
  movsd(Address(rsp, 0), xmm0);
  lea(eax, Address(rsp, 40));
  movl(Address(rsp, 8), eax);
  movl(eax, 2);
  movl(Address(rsp, 12), eax);
  call(RuntimeAddress(CAST_FROM_FN_PTR(address, StubRoutines::dlibm_sin_cos_huge())));
  addl(rsp, 32);
  fld_d(Address(rsp, 16));
  jmp(L_2TAG_PACKET_1_0_2);
  bind(L_2TAG_PACKET_4_0_2);
  fld_d(Address(rsp, 128));
  fmul_d(Address(ebx, 2240));
  bind(L_2TAG_PACKET_1_0_2);
  movl(ebx, Address(rsp, 56));
}
