/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 *
 * Copyright 2024 GNOME Foundation, Inc.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Authors:
 *  - Philip Withnall <pwithnall@gnome.org>
 */

#include "config.h"

#include <glib.h>
#include <glib-object.h>
#include <glib-unix.h>
#include <glib/gi18n-lib.h>
#include <gio/gio.h>
#include <locale.h>
#include <libgsystemservice/peer-manager.h>
#include <libgsystemservice/peer-manager-dbus.h>
#include <libgsystemservice/service.h>
#include <libmalcontent-timer/child-timer-service.h>
#include <libmalcontent-timer/parent-timer-service.h>
#include <libmalcontent-timer/timer-service.h>
#include <libmalcontent-timer/timer-store.h>


static void mct_timer_service_dispose (GObject *object);
static void mct_timer_service_finalize (GObject *object);

static void mct_timer_service_startup_async (GssService          *service,
                                             GCancellable        *cancellable,
                                             GAsyncReadyCallback  callback,
                                             gpointer             user_data);
static void mct_timer_service_startup_finish (GssService    *service,
                                              GAsyncResult  *result,
                                              GError       **error);
static void mct_timer_service_shutdown (GssService *service);

static void notify_busy_cb (GObject    *obj,
                            GParamSpec *pspec,
                            gpointer    user_data);

/**
 * MctTimerService:
 *
 * The core implementation of the timer daemon, which exposes its D-Bus
 * API on the bus.
 *
 * Since: 0.14.0
 */
struct _MctTimerService
{
  GssService parent;

  char *state_directory_path;  /* (owned) (nullable) */
  MctTimerStore *timer_store;  /* (owned) */
  MctChildTimerService *child_timer_service;  /* (owned) */
  gulong child_timer_service_notify_busy_id;
  MctParentTimerService *parent_timer_service;  /* (owned) */
  unsigned long parent_timer_service_notify_busy_id;

  gboolean busy;
};

G_DEFINE_TYPE (MctTimerService, mct_timer_service, GSS_TYPE_SERVICE)

static void
mct_timer_service_class_init (MctTimerServiceClass *klass)
{
  GObjectClass *object_class = (GObjectClass *) klass;
  GssServiceClass *service_class = (GssServiceClass *) klass;

  object_class->dispose = mct_timer_service_dispose;
  object_class->finalize = mct_timer_service_finalize;

  service_class->startup_async = mct_timer_service_startup_async;
  service_class->startup_finish = mct_timer_service_startup_finish;
  service_class->shutdown = mct_timer_service_shutdown;
}

static void
mct_timer_service_init (MctTimerService *self)
{
  g_autoptr(GOptionGroup) group = NULL;
  const GOptionEntry options[] =
    {
      { "state-directory", 0, G_OPTION_FLAG_NONE, G_OPTION_ARG_FILENAME, &self->state_directory_path,
        N_("Directory to store data in (default: $STATE_DIRECTORY)"), N_("DIR") },
      { NULL}
    };

  /* Add an additional command line option for the state directory. */
  group = g_option_group_new ("service", _("Service Options:"), _("Show service help options"), NULL, NULL);
  g_option_group_add_entries (group, options);
  gss_service_add_option_group (GSS_SERVICE (self), group);
}

static void
mct_timer_service_dispose (GObject *object)
{
  MctTimerService *self = MCT_TIMER_SERVICE (object);

  g_clear_signal_handler (&self->child_timer_service_notify_busy_id, self->child_timer_service);
  g_clear_object (&self->child_timer_service);

  g_clear_signal_handler (&self->parent_timer_service_notify_busy_id, self->parent_timer_service);
  g_clear_object (&self->parent_timer_service);

  g_clear_object (&self->timer_store);

  /* Chain up to the parent class */
  G_OBJECT_CLASS (mct_timer_service_parent_class)->dispose (object);
}

static void
mct_timer_service_finalize (GObject *object)
{
  MctTimerService *self = MCT_TIMER_SERVICE (object);

  g_clear_pointer (&self->state_directory_path, g_free);

  /* Chain up to the parent class */
  G_OBJECT_CLASS (mct_timer_service_parent_class)->finalize (object);
}

static void startup_user_manager_cb (GObject      *source_object,
                                     GAsyncResult *result,
                                     void         *user_data);

static void
mct_timer_service_startup_async (GssService          *service,
                                 GCancellable        *cancellable,
                                 GAsyncReadyCallback  callback,
                                 gpointer             user_data)
{
  MctTimerService *self = MCT_TIMER_SERVICE (service);
  GDBusConnection *connection = gss_service_get_dbus_connection (GSS_SERVICE (self));
  g_autoptr(MctUserManager) user_manager = NULL;
  g_autoptr(GTask) task = NULL;

  task = g_task_new (service, cancellable, callback, user_data);
  g_task_set_source_tag (task, mct_timer_service_startup_async);

  /* Load the user manager first as it’s the only async operation we need to do */
  user_manager = mct_user_manager_new (connection);

  mct_user_manager_load_async (user_manager, cancellable, startup_user_manager_cb, g_steal_pointer (&task));
}

static void
startup_user_manager_cb (GObject      *source_object,
                         GAsyncResult *result,
                         void         *user_data)
{
  MctUserManager *user_manager = MCT_USER_MANAGER (source_object);
  g_autoptr(GTask) task = g_steal_pointer (&user_data);
  MctTimerService *self = MCT_TIMER_SERVICE (g_task_get_source_object (task));
  GDBusConnection *connection = gss_service_get_dbus_connection (GSS_SERVICE (self));
  const char *state_directory_env;
  g_autoptr(GFile) state_directory = NULL;
  g_autoptr(GFile) store_directory = NULL;
  g_autoptr(GssPeerManager) peer_manager = NULL;
  g_autoptr(MctManager) policy_manager = NULL;
  g_autoptr(GError) local_error = NULL;

  if (!mct_user_manager_load_finish (user_manager, result, &local_error))
    {
      g_task_return_error (task, g_steal_pointer (&local_error));
      return;
    }

  /* Work out where the state directory is. If running under systemd, it’ll be
   * passed in as `STATE_DIRECTORY` due to our use of the `StateDirectory=`
   * option (see systemd.exec(5)). We prefer the command line option, if
   * specified, as that’s used for debugging. */
  state_directory_env = g_getenv ("STATE_DIRECTORY");
  if (self->state_directory_path != NULL)
    state_directory = g_file_new_for_commandline_arg (self->state_directory_path);
  else if (state_directory_env != NULL)
    state_directory = g_file_new_for_commandline_arg (state_directory_env);
  else
    {
      g_task_return_new_error (task, GSS_SERVICE_ERROR, GSS_SERVICE_ERROR_INVALID_ENVIRONMENT,
                               _("No state directory specified using $STATE_DIRECTORY or --state-directory"));
      return;
    }

  store_directory = g_file_get_child (state_directory, "store");
  self->timer_store = mct_timer_store_new (store_directory);
  peer_manager = GSS_PEER_MANAGER (gss_peer_manager_dbus_new (connection));
  policy_manager = mct_manager_new (connection);

  self->child_timer_service =
      mct_child_timer_service_new (connection,
                                   "/org/freedesktop/MalcontentTimer1",
                                   self->timer_store,
                                   peer_manager,
                                   policy_manager);
  self->child_timer_service_notify_busy_id =
      g_signal_connect (self->child_timer_service, "notify::busy",
                        (GCallback) notify_busy_cb, self);

  self->parent_timer_service =
      mct_parent_timer_service_new (connection,
                                    "/org/freedesktop/MalcontentTimer1",
                                    self->timer_store,
                                    user_manager,
                                    peer_manager);
  self->parent_timer_service_notify_busy_id =
      g_signal_connect (self->parent_timer_service, "notify::busy",
                        (GCallback) notify_busy_cb, self);

  notify_busy_cb (NULL, NULL, self);

  if (!mct_child_timer_service_register (self->child_timer_service, &local_error) ||
      !mct_parent_timer_service_register (self->parent_timer_service, &local_error))
    g_task_return_error (task, g_steal_pointer (&local_error));
  else
    g_task_return_boolean (task, TRUE);
}

static void
mct_timer_service_startup_finish (GssService    *service,
                                  GAsyncResult  *result,
                                  GError       **error)
{
  g_task_propagate_boolean (G_TASK (result), error);
}

static void
notify_busy_cb (GObject    *obj,
                GParamSpec *pspec,
                gpointer    user_data)
{
  MctTimerService *self = MCT_TIMER_SERVICE (user_data);

  gboolean was_busy = self->busy;
  gboolean child_now_busy = mct_child_timer_service_get_busy (self->child_timer_service);
  gboolean parent_now_busy = mct_parent_timer_service_get_busy (self->parent_timer_service);
  gboolean now_busy = child_now_busy || parent_now_busy;

  g_debug ("%s: was_busy: %s, now_busy: %s (child_now_busy: %s, parent_now_busy: %s)",
           G_STRFUNC,
           was_busy ? "yes" : "no",
           now_busy ? "yes" : "no",
           child_now_busy ? "yes" : "no",
           parent_now_busy ? "yes" : "no");

  if (was_busy && !now_busy)
    gss_service_release (GSS_SERVICE (self));
  else if (!was_busy && now_busy)
    gss_service_hold (GSS_SERVICE (self));

  self->busy = now_busy;
}

static void
mct_timer_service_shutdown (GssService *service)
{
  MctTimerService *self = MCT_TIMER_SERVICE (service);

  mct_child_timer_service_unregister (self->child_timer_service);
  mct_parent_timer_service_unregister (self->parent_timer_service);
}

/**
 * mct_timer_service_new:
 *
 * Create a new [class@Malcontent.TimerService].
 *
 * Returns: (transfer full): a new [class@Malcontent.TimerService]
 * Since: 0.14.0
 */
MctTimerService *
mct_timer_service_new (void)
{
  return g_object_new (MCT_TYPE_TIMER_SERVICE,
                       "bus-type", G_BUS_TYPE_SYSTEM,
                       "service-id", "org.freedesktop.MalcontentTimer1",
                       "inactivity-timeout", 30000  /* ms */,
                       "translation-domain", GETTEXT_PACKAGE,
                       "parameter-string", _("— record session and app usage information for parental controls"),
                       "summary", _("Record session and app usage information "
                                    "provided by user login sessions to "
                                    "monitor and enforce configured parental "
                                    "controls screen time limits on accounts."),
                       NULL);
}

