/*
    SPDX-FileCopyrightText: 2019 - 2022 UnionTech Software Technology Co., Ltd.

    SPDX-License-Identifier: LGPL-3.0-or-later
*/

#pragma once

#include <array>
#include <cmath>
#include <cstdint>
#include <numbers>

namespace SolarUtils
{
constexpr double J2000 = 2451545.0;
struct MoonEclipticParameter {
    double Lp;
    double D;
    double M;
    double Mp;
    double F;
    double E;
};

struct MoonEclipticLongitudeCoeff {
    double D;
    double M;
    double Mp;
    double F;
    double EiA;
    double ErA;
};

struct EarthNutationParameter {
    double D;
    double M;
    double Mp;
    double F;
    double Omega;
};

struct NuationCoefficient {
    double D;
    double M;
    double Mp;
    double F;
    double Omega;
    double Sine1;
    double Sine2;
    double Cosine1;
    double Cosine2;
};

constexpr std::array<MoonEclipticLongitudeCoeff, 60> s_moonLongitude{
    MoonEclipticLongitudeCoeff{0, 0, 1, 0, 6288744, -20905355},
    {2, 0, -1, 0, 1274027, -3699111},
    {2, 0, 0, 0, 658314, -2955968},
    {0, 0, 2, 0, 213618, -569925},
    {0, 1, 0, 0, -185116, 48888},
    {0, 0, 0, 2, -114332, -3149},
    {2, 0, -2, 0, 58793, 246158},
    {2, -1, -1, 0, 57066, -152138},
    {2, 0, 1, 0, 53322, -170733},
    {2, -1, 0, 0, 45758, -204586},
    {0, 1, -1, 0, -40923, -129620},
    {1, 0, 0, 0, -34720, 108743},
    {0, 1, 1, 0, -30383, 104755},
    {2, 0, 0, -2, 15327, 10321},
    {0, 0, 1, 2, -12528, 0},
    {0, 0, 1, -2, 10980, 79661},
    {4, 0, -1, 0, 10675, -34782},
    {0, 0, 3, 0, 10034, -23210},
    {4, 0, -2, 0, 8548, -21636},
    {2, 1, -1, 0, -7888, 24208},
    {2, 1, 0, 0, -6766, 30824},
    {1, 0, -1, 0, -5163, -8379},
    {1, 1, 0, 0, 4987, -16675},
    {2, -1, 1, 0, 4036, -12831},
    {2, 0, 2, 0, 3994, -10445},
    {4, 0, 0, 0, 3861, -11650},
    {2, 0, -3, 0, 3665, 14403},
    {0, 1, -2, 0, -2689, -7003},
    {2, 0, -1, 2, -2602, 0},
    {2, -1, -2, 0, 2390, 10056},
    {1, 0, 1, 0, -2348, 6322},
    {2, -2, 0, 0, 2236, -9884},
    {0, 1, 2, 0, -2120, 5751},
    {0, 2, 0, 0, -2069, 0},
    {2, -2, -1, 0, 2048, -4950},
    {2, 0, 1, -2, -1773, 4130},
    {2, 0, 0, 2, -1595, 0},
    {4, -1, -1, 0, 1215, -3958},
    {0, 0, 2, 2, -1110, 0},
    {3, 0, -1, 0, -892, 3258},
    {2, 1, 1, 0, -810, 2616},
    {4, -1, -2, 0, 759, -1897},
    {0, 2, -1, 0, -713, -2117},
    {2, 2, -1, 0, -700, 2354},
    {2, 1, -2, 0, 691, 0},
    {2, -1, 0, -2, 596, 0},
    {4, 0, 1, 0, 549, -1423},
    {0, 0, 4, 0, 537, -1117},
    {4, -1, 0, 0, 520, -1571},
    {1, 0, -2, 0, -487, -1739},
    {2, 1, 0, -2, -399, 0},
    {0, 0, 2, -2, -381, -4421},
    {1, 1, 1, 0, 351, 0},
    {3, 0, -2, 0, -340, 0},
    {4, 0, -3, 0, 330, 0},
    {2, -1, 2, 0, 327, 0},
    {0, 2, 1, 0, -323, 1165},
    {1, 1, -1, 0, 299, 0},
    {2, 0, 3, 0, 294, 0},
    {2, 0, -1, -2, 0, 8752},
};

constexpr std::array<NuationCoefficient, 63> s_nuation{
    NuationCoefficient{0, 0, 0, 0, 1, -171996, -174.2, 92025, 8.9},
    NuationCoefficient{-2, 0, 0, 2, 2, -13187, -1.6, 5736, -3.1},
    NuationCoefficient{0, 0, 0, 2, 2, -2274, -0.2, 977, -0.5},
    NuationCoefficient{0, 0, 0, 0, 2, 2062, 0.2, -895, 0.5},
    NuationCoefficient{0, 1, 0, 0, 0, 1426, -3.4, 54, -0.1},
    NuationCoefficient{0, 0, 1, 0, 0, 712, 0.1, -7, 0},
    NuationCoefficient{-2, 1, 0, 2, 2, -517, 1.2, 224, -0.6},
    NuationCoefficient{0, 0, 0, 2, 1, -386, -0.4, 200, 0},
    NuationCoefficient{0, 0, 1, 2, 2, -301, 0, 129, -0.1},
    NuationCoefficient{-2, -1, 0, 2, 2, 217, -0.5, -95, 0.3},
    NuationCoefficient{-2, 0, 1, 0, 0, -158, 0, 0, 0},
    NuationCoefficient{-2, 0, 0, 2, 1, 129, 0.1, -70, 0},
    NuationCoefficient{0, 0, -1, 2, 2, 123, 0, -53, 0},
    NuationCoefficient{2, 0, 0, 0, 0, 63, 0, 0, 0},
    NuationCoefficient{0, 0, 1, 0, 1, 63, 0.1, -33, 0},
    NuationCoefficient{2, 0, -1, 2, 2, -59, 0, 26, 0},
    NuationCoefficient{0, 0, -1, 0, 1, -58, -0.1, 32, 0},
    NuationCoefficient{0, 0, 1, 2, 1, -51, 0, 27, 0},
    NuationCoefficient{-2, 0, 2, 0, 0, 48, 0, 0, 0},
    NuationCoefficient{0, 0, -2, 2, 1, 46, 0, -24, 0},
    NuationCoefficient{2, 0, 0, 2, 2, -38, 0, 16, 0},
    NuationCoefficient{0, 0, 2, 2, 2, -31, 0, 13, 0},
    NuationCoefficient{0, 0, 2, 0, 0, 29, 0, 0, 0},
    NuationCoefficient{-2, 0, 1, 2, 2, 29, 0, -12, 0},
    NuationCoefficient{0, 0, 0, 2, 0, 26, 0, 0, 0},
    NuationCoefficient{-2, 0, 0, 2, 0, -22, 0, 0, 0},
    NuationCoefficient{0, 0, -1, 2, 1, 21, 0, -10, 0},
    NuationCoefficient{0, 2, 0, 0, 0, 17, -0.1, 0, 0},
    NuationCoefficient{2, 0, -1, 0, 1, 16, 0, -8, 0},
    NuationCoefficient{-2, 2, 0, 2, 2, -16, 0.1, 7, 0},
    NuationCoefficient{0, 1, 0, 0, 1, -15, 0, 9, 0},
    NuationCoefficient{-2, 0, 1, 0, 1, -13, 0, 7, 0},
    NuationCoefficient{0, -1, 0, 0, 1, -12, 0, 6, 0},
    NuationCoefficient{0, 0, 2, -2, 0, 11, 0, 0, 0},
    NuationCoefficient{2, 0, -1, 2, 1, -10, 0, 5, 0},
    NuationCoefficient{2, 0, 1, 2, 2, -8, 0, 3, 0},
    NuationCoefficient{0, 1, 0, 2, 2, 7, 0, -3, 0},
    NuationCoefficient{-2, 1, 1, 0, 0, -7, 0, 0, 0},
    NuationCoefficient{0, -1, 0, 2, 2, -7, 0, 3, 0},
    NuationCoefficient{2, 0, 0, 2, 1, -7, 0, 3, 0},
    NuationCoefficient{2, 0, 1, 0, 0, 6, 0, 0, 0},
    NuationCoefficient{-2, 0, 2, 2, 2, 6, 0, -3, 0},
    NuationCoefficient{-2, 0, 1, 2, 1, 6, 0, -3, 0},
    NuationCoefficient{2, 0, -2, 0, 1, -6, 0, 3, 0},
    NuationCoefficient{2, 0, 0, 0, 1, -6, 0, 3, 0},
    NuationCoefficient{0, -1, 1, 0, 0, 5, 0, 0, 0},
    NuationCoefficient{-2, -1, 0, 2, 1, -5, 0, 3, 0},
    NuationCoefficient{-2, 0, 0, 0, 1, -5, 0, 3, 0},
    NuationCoefficient{0, 0, 2, 2, 1, -5, 0, 3, 0},
    NuationCoefficient{-2, 0, 2, 0, 1, 4, 0, 0, 0},
    NuationCoefficient{-2, 1, 0, 2, 1, 4, 0, 0, 0},
    NuationCoefficient{0, 0, 1, -2, 0, 4, 0, 0, 0},
    NuationCoefficient{-1, 0, 1, 0, 0, -4, 0, 0, 0},
    NuationCoefficient{-2, 1, 0, 0, 0, -4, 0, 0, 0},
    NuationCoefficient{1, 0, 0, 0, 0, -4, 0, 0, 0},
    NuationCoefficient{0, 0, 1, 2, 0, 3, 0, 0, 0},
    NuationCoefficient{0, 0, -2, 2, 2, -3, 0, 0, 0},
    NuationCoefficient{-1, -1, 1, 0, 0, -3, 0, 0, 0},
    NuationCoefficient{0, 1, 1, 0, 0, -3, 0, 0, 0},
    NuationCoefficient{0, -1, 1, 2, 2, -3, 0, 0, 0},
    NuationCoefficient{2, -1, -1, 2, 2, -3, 0, 0, 0},
    NuationCoefficient{0, 0, 3, 2, 2, -3, 0, 0, 0},
    NuationCoefficient{2, -1, 0, 2, 2, -3, 0, 0, 0},
};

constexpr double getEarthL0(double t)
{
    double result = 0.0;
    result += 1.75347045673;
    result += 0.03341656456 * std::cos(4.66925680417 + 6283.0758499914 * t);
    result += 0.00034894275 * std::cos(4.62610241759 + 12566.1516999828 * t);
    result += 3.417571e-05 * std::cos(2.82886579606 + 3.523118349 * t);
    result += 3.497056e-05 * std::cos(2.74411800971 + 5753.3848848968 * t);
    result += 3.135896e-05 * std::cos(3.62767041758 + 77713.7714681205 * t);
    result += 2.676218e-05 * std::cos(4.41808351397 + 7860.4193924392 * t);
    result += 2.342687e-05 * std::cos(6.13516237631 + 3930.2096962196 * t);
    result += 1.273166e-05 * std::cos(2.03709655772 + 529.6909650946 * t);
    result += 1.324292e-05 * std::cos(0.74246356352 + 11506.7697697936 * t);
    result += 9.01855e-06 * std::cos(2.04505443513 + 26.2983197998 * t);
    result += 1.199167e-05 * std::cos(1.10962944315 + 1577.3435424478 * t);
    result += 8.57223e-06 * std::cos(3.50849156957 + 398.1490034082 * t);
    result += 7.79786e-06 * std::cos(1.17882652114 + 5223.6939198022 * t);
    result += 9.9025e-06 * std::cos(5.23268129594 + 5884.9268465832 * t);
    result += 7.53141e-06 * std::cos(2.53339053818 + 5507.5532386674 * t);
    result += 5.05264e-06 * std::cos(4.58292563052 + 18849.2275499742 * t);
    result += 4.92379e-06 * std::cos(4.20506639861 + 775.522611324 * t);
    result += 3.56655e-06 * std::cos(2.91954116867 + 0.0673103028 * t);
    result += 2.84125e-06 * std::cos(1.89869034186 + 796.2980068164 * t);
    result += 2.4281e-06 * std::cos(0.34481140906 + 5486.777843175 * t);
    result += 3.17087e-06 * std::cos(5.84901952218 + 11790.6290886588 * t);
    result += 2.71039e-06 * std::cos(0.31488607649 + 10977.078804699 * t);
    result += 2.0616e-06 * std::cos(4.80646606059 + 2544.3144198834 * t);
    result += 2.05385e-06 * std::cos(1.86947813692 + 5573.1428014331 * t);
    result += 2.02261e-06 * std::cos(2.45767795458 + 6069.7767545534 * t);
    result += 1.26184e-06 * std::cos(1.0830263021 + 20.7753954924 * t);
    result += 1.55516e-06 * std::cos(0.83306073807 + 213.299095438 * t);
    result += 1.15132e-06 * std::cos(0.64544911683 + 0.9803210682 * t);
    result += 1.02851e-06 * std::cos(0.63599846727 + 4694.0029547076 * t);
    result += 1.01724e-06 * std::cos(4.26679821365 + 7.1135470008 * t);
    result += 9.9206e-07 * std::cos(6.20992940258 + 2146.1654164752 * t);
    result += 1.32212e-06 * std::cos(3.41118275555 + 2942.4634232916 * t);
    result += 9.7607e-07 * std::cos(0.6810127227 + 155.4203994342 * t);
    result += 8.5128e-07 * std::cos(1.29870743025 + 6275.9623029906 * t);
    result += 7.4651e-07 * std::cos(1.75508916159 + 5088.6288397668 * t);
    result += 1.01895e-06 * std::cos(0.97569221824 + 15720.8387848784 * t);
    result += 8.4711e-07 * std::cos(3.67080093025 + 71430.6956181291 * t);
    result += 7.3547e-07 * std::cos(4.67926565481 + 801.8209311238 * t);
    result += 7.3874e-07 * std::cos(3.50319443167 + 3154.6870848956 * t);
    result += 7.8756e-07 * std::cos(3.03698313141 + 12036.4607348882 * t);
    result += 7.9637e-07 * std::cos(1.807913307 + 17260.1546546904 * t);
    result += 8.5803e-07 * std::cos(5.98322631256 + 161000.685737674 * t);
    result += 5.6963e-07 * std::cos(2.78430398043 + 6286.5989683404 * t);
    result += 6.1148e-07 * std::cos(1.81839811024 + 7084.8967811152 * t);
    result += 6.9627e-07 * std::cos(0.83297596966 + 9437.762934887 * t);
    result += 5.6116e-07 * std::cos(4.38694880779 + 14143.4952424306 * t);
    result += 6.2449e-07 * std::cos(3.97763880587 + 8827.3902698748 * t);
    result += 5.1145e-07 * std::cos(0.28306864501 + 5856.4776591154 * t);
    result += 5.5577e-07 * std::cos(3.47006009062 + 6279.5527316424 * t);
    result += 4.1036e-07 * std::cos(5.36817351402 + 8429.2412664666 * t);
    result += 5.1605e-07 * std::cos(1.33282746983 + 1748.016413067 * t);
    result += 5.1992e-07 * std::cos(0.18914945834 + 12139.5535091068 * t);
    result += 4.9e-07 * std::cos(0.48735065033 + 1194.4470102246 * t);
    result += 3.92e-07 * std::cos(6.16832995016 + 10447.3878396044 * t);
    result += 3.5566e-07 * std::cos(1.77597314691 + 6812.766815086 * t);
    result += 3.677e-07 * std::cos(6.04133859347 + 10213.285546211 * t);
    result += 3.6596e-07 * std::cos(2.56955238628 + 1059.3819301892 * t);
    result += 3.3291e-07 * std::cos(0.59309499459 + 17789.845619785 * t);
    result += 3.5954e-07 * std::cos(1.70876111898 + 2352.8661537718 * t);
    result += 4.0938e-07 * std::cos(2.39850881707 + 19651.048481098 * t);
    result += 3.0047e-07 * std::cos(2.73975123935 + 1349.8674096588 * t);
    result += 3.0412e-07 * std::cos(0.44294464135 + 83996.8473181119 * t);
    result += 2.3663e-07 * std::cos(0.48473567763 + 8031.0922630584 * t);
    result += 2.3574e-07 * std::cos(2.06527720049 + 3340.6124266998 * t);
    result += 2.1089e-07 * std::cos(4.14825464101 + 951.7184062506 * t);
    result += 2.4738e-07 * std::cos(0.21484762138 + 3.5904286518 * t);
    result += 2.5352e-07 * std::cos(3.16470953405 + 4690.4798363586 * t);
    result += 2.282e-07 * std::cos(5.22197888032 + 4705.7323075436 * t);
    result += 2.1419e-07 * std::cos(1.42563735525 + 16730.4636895958 * t);
    result += 2.1891e-07 * std::cos(5.55594302562 + 553.5694028424 * t);
    result += 1.7481e-07 * std::cos(4.56052900359 + 135.0650800354 * t);
    result += 1.9925e-07 * std::cos(5.22208471269 + 12168.0026965746 * t);
    result += 1.986e-07 * std::cos(5.77470167653 + 6309.3741697912 * t);
    result += 2.03e-07 * std::cos(0.37133792946 + 283.8593188652 * t);
    result += 1.4421e-07 * std::cos(4.19315332546 + 242.728603974 * t);
    result += 1.6225e-07 * std::cos(5.98837722564 + 11769.8536931664 * t);
    result += 1.5077e-07 * std::cos(4.19567181073 + 6256.7775301916 * t);
    result += 1.9124e-07 * std::cos(3.82219996949 + 23581.2581773176 * t);
    result += 1.8888e-07 * std::cos(5.38626880969 + 149854.400134808 * t);
    result += 1.4346e-07 * std::cos(3.72355084422 + 38.0276726358 * t);
    result += 1.7898e-07 * std::cos(2.21490735647 + 13367.9726311066 * t);
    result += 1.2054e-07 * std::cos(2.62229588349 + 955.5997416086 * t);
    result += 1.1287e-07 * std::cos(0.17739328092 + 4164.311989613 * t);
    result += 1.3971e-07 * std::cos(4.40138139996 + 6681.2248533996 * t);
    result += 1.3621e-07 * std::cos(1.88934471407 + 7632.9432596502 * t);
    result += 1.2503e-07 * std::cos(1.13052412208 + 5.5229243074 * t);
    result += 1.0498e-07 * std::cos(5.35909518669 + 1592.5960136328 * t);
    result += 9.803e-08 * std::cos(0.99947478995 + 11371.7046897582 * t);
    result += 9.22e-08 * std::cos(4.57138609781 + 4292.3308329504 * t);
    result += 1.0327e-07 * std::cos(6.19982566125 + 6438.4962494256 * t);
    result += 1.2003e-07 * std::cos(1.003514567 + 632.7837393132 * t);
    result += 1.0827e-07 * std::cos(0.32734520222 + 103.0927742186 * t);
    result += 8.356e-08 * std::cos(4.53902685948 + 25132.3033999656 * t);
    result += 1.0005e-07 * std::cos(6.0291496328 + 5746.271337896 * t);
    result += 8.409e-08 * std::cos(3.29946744189 + 7234.794256242 * t);
    result += 8.006e-08 * std::cos(5.82145271907 + 28.4491874678 * t);
    result += 1.0523e-07 * std::cos(0.93871805506 + 11926.2544136688 * t);
    result += 7.686e-08 * std::cos(3.12142363172 + 7238.6755916 * t);
    result += 9.378e-08 * std::cos(2.62414241032 + 5760.4984318976 * t);
    result += 8.127e-08 * std::cos(6.11228001785 + 4732.0306273434 * t);
    result += 9.232e-08 * std::cos(0.48343968736 + 522.5774180938 * t);
    result += 9.802e-08 * std::cos(5.24413991147 + 27511.4678735372 * t);
    result += 7.871e-08 * std::cos(0.99590177926 + 5643.1785636774 * t);
    result += 8.123e-08 * std::cos(6.2705301365 + 426.598190876 * t);
    result += 9.048e-08 * std::cos(5.33686335897 + 6386.16862421 * t);
    result += 8.62e-08 * std::cos(4.16538210888 + 7058.5984613154 * t);
    result += 6.297e-08 * std::cos(4.71724819317 + 6836.6452528338 * t);
    result += 7.575e-08 * std::cos(3.97382858911 + 11499.6562227928 * t);
    result += 7.756e-08 * std::cos(2.95729056763 + 23013.5395395872 * t);
    result += 7.314e-08 * std::cos(0.60652505806 + 11513.8833167944 * t);
    result += 5.955e-08 * std::cos(2.87641047971 + 6283.14316029419 * t);
    result += 6.534e-08 * std::cos(5.79072926033 + 18073.7049386502 * t);
    result += 7.188e-08 * std::cos(3.99831508699 + 74.7815985673 * t);
    result += 7.346e-08 * std::cos(4.38582365437 + 316.3918696566 * t);
    result += 5.413e-08 * std::cos(5.39199024641 + 419.4846438752 * t);
    result += 5.127e-08 * std::cos(2.36062848786 + 10973.55568635 * t);
    result += 7.056e-08 * std::cos(0.32258441903 + 263.0839233728 * t);
    result += 6.625e-08 * std::cos(3.66475158672 + 17298.1823273262 * t);
    result += 6.762e-08 * std::cos(5.91132535899 + 90955.5516944961 * t);
    result += 4.938e-08 * std::cos(5.73672165674 + 9917.6968745098 * t);
    result += 5.547e-08 * std::cos(2.45152597661 + 12352.8526045448 * t);
    result += 5.958e-08 * std::cos(3.32051344676 + 6283.0085396886 * t);
    result += 4.471e-08 * std::cos(2.06385999536 + 7079.3738568078 * t);
    result += 6.153e-08 * std::cos(1.45823331144 + 233141.314404362 * t);
    result += 4.348e-08 * std::cos(4.4234217548 + 5216.5803728014 * t);
    result += 6.123e-08 * std::cos(1.07494905258 + 19804.8272915828 * t);
    result += 4.488e-08 * std::cos(3.6528503715 + 206.1855484372 * t);
    result += 4.02e-08 * std::cos(0.83995823171 + 20.3553193988 * t);
    result += 5.188e-08 * std::cos(4.06503864016 + 6208.2942514241 * t);
    result += 5.307e-08 * std::cos(0.38217636096 + 31441.6775697568 * t);
    result += 3.785e-08 * std::cos(2.34369213733 + 3.881335358 * t);
    result += 4.497e-08 * std::cos(3.27230796845 + 11015.1064773348 * t);
    result += 4.132e-08 * std::cos(0.92128915753 + 3738.761430108 * t);
    result += 3.521e-08 * std::cos(5.97844807108 + 3894.1818295422 * t);
    result += 4.215e-08 * std::cos(1.90601120623 + 245.8316462294 * t);
    result += 3.701e-08 * std::cos(5.03069397926 + 536.8045120954 * t);
    result += 3.865e-08 * std::cos(1.82634360607 + 11856.2186514245 * t);
    result += 3.652e-08 * std::cos(1.01838584934 + 16200.7727245012 * t);
    result += 3.39e-08 * std::cos(0.97785123922 + 8635.9420037632 * t);
    result += 3.737e-08 * std::cos(2.95380107829 + 3128.3887650958 * t);
    result += 3.507e-08 * std::cos(3.71291946325 + 6290.1893969922 * t);
    result += 3.086e-08 * std::cos(3.64646921512 + 10.6366653498 * t);
    result += 3.397e-08 * std::cos(1.10590684017 + 14712.317116458 * t);
    result += 3.334e-08 * std::cos(0.83684924911 + 6496.3749454294 * t);
    result += 2.805e-08 * std::cos(2.58504514144 + 14314.1681130498 * t);
    result += 3.65e-08 * std::cos(1.08344142571 + 88860.0570709867 * t);
    result += 3.388e-08 * std::cos(3.20185096055 + 5120.6011455836 * t);
    result += 3.252e-08 * std::cos(3.47859752062 + 6133.5126528568 * t);
    result += 2.553e-08 * std::cos(3.94869034189 + 1990.745017041 * t);
    result += 3.52e-08 * std::cos(2.05559692878 + 244287.600007228 * t);
    result += 2.565e-08 * std::cos(1.560717849 + 23543.2305046818 * t);
    result += 2.621e-08 * std::cos(3.85639359951 + 266.6070417218 * t);
    result += 2.955e-08 * std::cos(3.39692949667 + 9225.539273283 * t);
    result += 2.876e-08 * std::cos(6.02635617464 + 154717.609887683 * t);
    result += 2.395e-08 * std::cos(1.16131956403 + 10984.1923516998 * t);
    result += 3.161e-08 * std::cos(1.32798718453 + 10873.9860304804 * t);
    result += 3.163e-08 * std::cos(5.08946464629 + 21228.3920235458 * t);
    result += 2.361e-08 * std::cos(4.27212906992 + 6040.3472460174 * t);
    result += 3.03e-08 * std::cos(1.80209931347 + 35371.8872659764 * t);
    result += 2.343e-08 * std::cos(3.576898605 + 10969.9652576982 * t);
    result += 2.618e-08 * std::cos(2.57870156528 + 22483.8485744926 * t);
    result += 2.113e-08 * std::cos(3.71393780256 + 65147.6197681377 * t);
    result += 2.019e-08 * std::cos(0.81393923319 + 170.6728706192 * t);
    result += 2.003e-08 * std::cos(0.38091017375 + 6172.869528772 * t);
    result += 2.506e-08 * std::cos(3.74379142438 + 10575.4066829418 * t);
    result += 2.381e-08 * std::cos(0.10581361289 + 7.046236698 * t);
    result += 1.949e-08 * std::cos(4.86892513469 + 36.0278666774 * t);
    result += 2.074e-08 * std::cos(4.2279477457 + 5650.2921106782 * t);
    result += 1.924e-08 * std::cos(5.5946054986 + 6282.0955289232 * t);
    result += 1.949e-08 * std::cos(1.07002512703 + 5230.807466803 * t);
    result += 1.988e-08 * std::cos(5.19736046771 + 6262.300454499 * t);
    result += 1.887e-08 * std::cos(3.74365662683 + 23.8784377478 * t);
    result += 1.787e-08 * std::cos(1.25929682929 + 12559.038152982 * t);
    result += 1.883e-08 * std::cos(1.90364058477 + 15.252471185 * t);
    result += 1.816e-08 * std::cos(3.68083868442 + 15110.4661198662 * t);
    result += 1.701e-08 * std::cos(4.4110589538 + 110.2063212194 * t);
    result += 1.99e-08 * std::cos(3.93295788548 + 6206.8097787158 * t);
    result += 2.103e-08 * std::cos(0.75354917468 + 13521.7514415914 * t);
    result += 1.774e-08 * std::cos(0.48747535361 + 1551.045222648 * t);
    result += 1.882e-08 * std::cos(0.86684493432 + 22003.9146348698 * t);
    result += 1.924e-08 * std::cos(1.22898324132 + 709.9330485583 * t);
    result += 2.009e-08 * std::cos(4.6285092198 + 6037.244203762 * t);
    result += 1.924e-08 * std::cos(0.60231842508 + 6284.0561710596 * t);
    result += 1.596e-08 * std::cos(3.98332956992 + 13916.0191096416 * t);
    result += 1.664e-08 * std::cos(4.41939715469 + 8662.240323563 * t);
    result += 1.971e-08 * std::cos(1.04560500503 + 18209.3302636602 * t);
    result += 1.942e-08 * std::cos(4.31335979989 + 6244.9428143536 * t);
    result += 1.476e-08 * std::cos(0.93271367331 + 2379.1644735716 * t);
    result += 1.81e-08 * std::cos(0.49112137707 + 1.4844727083 * t);
    result += 1.346e-08 * std::cos(1.51574702235 + 4136.9104335162 * t);
    result += 1.528e-08 * std::cos(5.61835711404 + 6127.6554505572 * t);
    result += 1.791e-08 * std::cos(3.22187270126 + 39302.096962196 * t);
    result += 1.747e-08 * std::cos(3.05638656738 + 18319.5365848796 * t);
    result += 1.431e-08 * std::cos(4.51153808594 + 20426.571092422 * t);
    result += 1.695e-08 * std::cos(0.22047718414 + 25158.6017197654 * t);
    result += 1.242e-08 * std::cos(4.46665769933 + 17256.6315363414 * t);
    result += 1.463e-08 * std::cos(4.69242679213 + 14945.3161735544 * t);
    result += 1.205e-08 * std::cos(1.86912144659 + 4590.910180489 * t);
    result += 1.192e-08 * std::cos(2.74227166898 + 12569.6748183318 * t);
    result += 1.222e-08 * std::cos(5.18120087482 + 5333.9002410216 * t);
    result += 1.39e-08 * std::cos(5.42894648983 + 143571.324284816 * t);
    result += 1.473e-08 * std::cos(1.70479245805 + 11712.9553182308 * t);
    result += 1.362e-08 * std::cos(2.61069503292 + 6062.6632075526 * t);
    result += 1.148e-08 * std::cos(6.0300180054 + 3634.6210245184 * t);
    result += 1.198e-08 * std::cos(5.15294130422 + 10177.2576795336 * t);
    result += 1.266e-08 * std::cos(0.11421493643 + 18422.6293590982 * t);
    result += 1.411e-08 * std::cos(1.09908857534 + 3496.032826134 * t);
    result += 1.349e-08 * std::cos(2.99805109633 + 17654.7805397496 * t);
    result += 1.253e-08 * std::cos(2.79850152848 + 167283.761587666 * t);
    result += 1.311e-08 * std::cos(1.60942984879 + 5481.2549188676 * t);
    result += 1.079e-08 * std::cos(6.20304501787 + 3.2863574178 * t);
    result += 1.181e-08 * std::cos(1.20653776978 + 131.5419616864 * t);
    result += 1.254e-08 * std::cos(5.45103277798 + 6076.8903015542 * t);
    result += 1.035e-08 * std::cos(2.32142722747 + 7342.4577801806 * t);
    result += 1.117e-08 * std::cos(0.38838354256 + 949.1756089698 * t);
    result += 9.66e-09 * std::cos(3.18341890851 + 11087.2851259184 * t);
    result += 1.171e-08 * std::cos(3.39635049962 + 12562.6285816338 * t);
    result += 1.121e-08 * std::cos(0.72627490378 + 220.4126424388 * t);
    result += 1.024e-08 * std::cos(2.19378315386 + 11403.676995575 * t);
    result += 8.88e-09 * std::cos(3.91173199285 + 4686.8894077068 * t);
    result += 9.1e-09 * std::cos(1.98802695087 + 735.8765135318 * t);
    result += 8.3e-09 * std::cos(0.48984915507 + 24072.9214697764 * t);
    result += 1.096e-08 * std::cos(6.17377835617 + 5436.9930152402 * t);
    result += 9.08e-09 * std::cos(0.44959639433 + 7477.522860216 * t);
    result += 9.74e-09 * std::cos(1.52996238356 + 9623.6882766912 * t);
    result += 8.4e-09 * std::cos(1.79543266333 + 5429.8794682394 * t);
    result += 7.78e-09 * std::cos(6.17699177946 + 38.1330356378 * t);
    result += 7.76e-09 * std::cos(4.09855402433 + 14.2270940016 * t);
    result += 1.068e-08 * std::cos(4.64200173735 + 43232.3066584156 * t);
    result += 9.54e-09 * std::cos(1.49988435748 + 1162.4747044078 * t);
    result += 9.07e-09 * std::cos(0.86986870809 + 10344.2950653858 * t);
    result += 9.31e-09 * std::cos(4.06044689031 + 28766.924424484 * t);
    result += 7.39e-09 * std::cos(5.04368197372 + 639.897286314 * t);
    result += 9.37e-09 * std::cos(3.4688469896 + 1589.0728952838 * t);
    result += 7.63e-09 * std::cos(5.86304932998 + 16858.4825329332 * t);
    result += 9.53e-09 * std::cos(4.20801492835 + 11190.377900137 * t);
    result += 7.08e-09 * std::cos(1.7289998894 + 13095.8426650774 * t);
    result += 9.69e-09 * std::cos(1.64439522215 + 29088.811415985 * t);
    result += 7.17e-09 * std::cos(0.16688678895 + 11.729352836 * t);
    result += 9.62e-09 * std::cos(3.53092337542 + 12416.5885028482 * t);
    result += 7.47e-09 * std::cos(5.77866940346 + 12592.4500197826 * t);
    result += 6.72e-09 * std::cos(1.91095796194 + 3.9321532631 * t);
    result += 6.71e-09 * std::cos(5.46240843677 + 18052.9295431578 * t);
    result += 6.75e-09 * std::cos(6.28311558823 + 4535.0594369244 * t);
    result += 6.84e-09 * std::cos(0.3997501208 + 5849.3641121146 * t);
    result += 7.99e-09 * std::cos(0.29851185294 + 12132.439962106 * t);
    result += 7.58e-09 * std::cos(0.96370823331 + 1052.2683831884 * t);
    result += 7.82e-09 * std::cos(5.33878339919 + 13517.8701062334 * t);
    result += 7.3e-09 * std::cos(1.70106160291 + 17267.2682016912 * t);
    result += 7.49e-09 * std::cos(2.59599901875 + 11609.8625440122 * t);
    result += 7.34e-09 * std::cos(2.78417782952 + 640.8776073822 * t);
    result += 6.88e-09 * std::cos(5.15048287468 + 16496.3613962024 * t);
    result += 7.7e-09 * std::cos(1.62469589333 + 4701.1165017084 * t);
    result += 6.33e-09 * std::cos(2.20587893893 + 25934.1243310894 * t);
    result += 7.6e-09 * std::cos(4.21317219403 + 377.3736079158 * t);
    result += 5.84e-09 * std::cos(2.13420121623 + 10557.5941608238 * t);
    result += 5.74e-09 * std::cos(0.24250054587 + 9779.1086761254 * t);
    result += 5.73e-09 * std::cos(3.16435264609 + 533.2140834436 * t);
    result += 6.85e-09 * std::cos(3.19344289472 + 12146.6670561076 * t);
    result += 6.75e-09 * std::cos(0.96179233959 + 10454.5013866052 * t);
    result += 6.48e-09 * std::cos(1.46327342555 + 6268.8487559898 * t);
    result += 5.89e-09 * std::cos(2.50543543638 + 3097.88382272579 * t);
    result += 5.51e-09 * std::cos(5.28099026956 + 9388.0059094152 * t);
    result += 6.96e-09 * std::cos(3.65342150016 + 4804.209275927 * t);
    result += 6.69e-09 * std::cos(2.51030077026 + 2388.8940204492 * t);
    result += 5.5e-09 * std::cos(0.06883864342 + 20199.094959633 * t);
    result += 6.29e-09 * std::cos(4.13350995675 + 45892.730433157 * t);
    result += 6.78e-09 * std::cos(6.09190163533 + 135.62532501 * t);
    result += 5.93e-09 * std::cos(1.50136257618 + 226858.23855437 * t);
    result += 5.42e-09 * std::cos(3.58573645173 + 6148.010769956 * t);
    result += 6.82e-09 * std::cos(5.02203067788 + 17253.0411076896 * t);
    result += 5.65e-09 * std::cos(4.2930923861 + 11933.3679606696 * t);
    result += 4.86e-09 * std::cos(0.77746204893 + 27.4015560968 * t);
    result += 5.03e-09 * std::cos(0.58963565969 + 15671.0817594066 * t);
    result += 6.16e-09 * std::cos(4.06539884128 + 227.476132789 * t);
    result += 5.83e-09 * std::cos(6.12695541996 + 18875.525869774 * t);
    result += 5.37e-09 * std::cos(2.1505644098 + 21954.157609398 * t);
    result += 6.69e-09 * std::cos(6.06986269566 + 47162.5163546352 * t);
    result += 4.75e-09 * std::cos(0.4034384211 + 6915.8595893046 * t);
    result += 5.4e-09 * std::cos(2.83444222174 + 5326.7866940208 * t);
    result += 5.3e-09 * std::cos(5.26359885263 + 10988.808157535 * t);
    result += 5.82e-09 * std::cos(3.24533095664 + 153.7788104848 * t);
    result += 6.41e-09 * std::cos(3.24711791371 + 2107.0345075424 * t);
    result += 6.21e-09 * std::cos(3.09698523779 + 33019.0211122046 * t);
    result += 4.66e-09 * std::cos(3.14982372198 + 10440.2742926036 * t);
    result += 4.66e-09 * std::cos(0.90708835657 + 5966.6839803348 * t);
    result += 5.28e-09 * std::cos(0.8192645447 + 813.5502839598 * t);
    result += 6.03e-09 * std::cos(3.81378921927 + 316428.228673915 * t);
    result += 5.59e-09 * std::cos(1.81894804124 + 17996.0311682222 * t);
    result += 4.37e-09 * std::cos(2.28625594435 + 6303.8512454838 * t);
    result += 5.18e-09 * std::cos(4.86069178322 + 20597.2439630412 * t);
    result += 4.24e-09 * std::cos(6.23520018693 + 6489.2613984286 * t);
    result += 5.18e-09 * std::cos(6.17617826756 + 0.2438174835 * t);
    result += 4.04e-09 * std::cos(5.72804304258 + 5642.1982426092 * t);
    result += 4.58e-09 * std::cos(1.34117773915 + 6287.0080032545 * t);
    result += 5.48e-09 * std::cos(5.6845445832 + 155427.542936241 * t);
    result += 5.47e-09 * std::cos(1.03391472061 + 3646.3503773544 * t);
    result += 4.28e-09 * std::cos(4.69800981138 + 846.0828347512 * t);
    result += 4.13e-09 * std::cos(6.02520699406 + 6279.4854213396 * t);
    result += 5.34e-09 * std::cos(3.03030638223 + 66567.4858652543 * t);
    result += 3.83e-09 * std::cos(1.49056949125 + 19800.9459562248 * t);
    result += 4.1e-09 * std::cos(5.28319622279 + 18451.078546566 * t);
    result += 3.52e-09 * std::cos(4.68891600359 + 4907.3020501456 * t);
    result += 4.8e-09 * std::cos(5.36572651091 + 348.924420448 * t);
    result += 3.44e-09 * std::cos(5.89157452896 + 6546.1597733642 * t);
    result += 3.4e-09 * std::cos(0.3755742644 + 13119.7211028252 * t);
    result += 4.34e-09 * std::cos(4.98417785901 + 6702.5604938666 * t);
    result += 3.32e-09 * std::cos(2.68902519126 + 29296.6153895786 * t);
    result += 4.48e-09 * std::cos(2.16478480251 + 5905.7022420756 * t);
    result += 3.44e-09 * std::cos(2.06546633735 + 49.7570254718 * t);
    result += 3.15e-09 * std::cos(1.24023811803 + 4061.2192153944 * t);
    result += 3.24e-09 * std::cos(2.30897526929 + 5017.508371365 * t);
    result += 4.13e-09 * std::cos(0.17171692962 + 6286.6662786432 * t);
    result += 4.31e-09 * std::cos(3.86601101393 + 12489.8856287072 * t);
    result += 3.49e-09 * std::cos(4.55372342974 + 4933.2084403326 * t);
    result += 3.23e-09 * std::cos(0.41971136084 + 10770.8932562618 * t);
    result += 3.41e-09 * std::cos(2.68612860807 + 11.0457002639 * t);
    result += 3.16e-09 * std::cos(3.52936906658 + 17782.7320727842 * t);
    result += 3.15e-09 * std::cos(5.63357264999 + 568.8218740274 * t);
    result += 3.4e-09 * std::cos(3.83571212349 + 10660.6869350424 * t);
    result += 2.97e-09 * std::cos(0.62691416712 + 20995.3929664494 * t);
    result += 4.05e-09 * std::cos(1.00085779471 + 16460.333529525 * t);
    result += 4.14e-09 * std::cos(1.21998752076 + 51092.7260508548 * t);
    result += 3.36e-09 * std::cos(4.71465945226 + 6179.9830757728 * t);
    result += 3.61e-09 * std::cos(3.71227508354 + 28237.2334593894 * t);
    result += 3.85e-09 * std::cos(6.21925225757 + 24356.7807886416 * t);
    result += 3.27e-09 * std::cos(1.05606504715 + 11919.140866668 * t);
    result += 3.27e-09 * std::cos(6.14222420989 + 6254.6266625236 * t);
    result += 2.68e-09 * std::cos(2.47224339737 + 664.75604513 * t);
    result += 2.69e-09 * std::cos(1.86207884109 + 23141.5583829246 * t);
    result += 3.45e-09 * std::cos(0.93461290184 + 6058.7310542895 * t);
    result += 2.96e-09 * std::cos(4.5168755718 + 6418.1409300268 * t);
    result += 3.53e-09 * std::cos(4.50033653082 + 36949.2308084242 * t);
    result += 2.6e-09 * std::cos(4.04963546305 + 6525.8044539654 * t);
    result += 2.98e-09 * std::cos(2.20046722622 + 156137.475984799 * t);
    result += 2.53e-09 * std::cos(3.49900838384 + 29864.334027309 * t);
    result += 2.54e-09 * std::cos(2.44901693835 + 5331.3574437408 * t);
    result += 2.96e-09 * std::cos(0.84347588787 + 5729.506447149 * t);
    result += 2.98e-09 * std::cos(1.29194706125 + 22805.7355659936 * t);
    result += 2.41e-09 * std::cos(2.00721280805 + 16737.5772365966 * t);
    result += 3.11e-09 * std::cos(1.23668016334 + 6281.5913772831 * t);
    result += 2.4e-09 * std::cos(2.51650377121 + 6245.0481773556 * t);
    result += 3.32e-09 * std::cos(3.55576945724 + 7668.6374249425 * t);
    result += 2.64e-09 * std::cos(4.44052061202 + 12964.300703391 * t);
    result += 2.57e-09 * std::cos(1.79654471948 + 11080.1715789176 * t);
    result += 2.6e-09 * std::cos(3.3307759842 + 5888.4499649322 * t);
    result += 2.85e-09 * std::cos(0.3088636143 + 11823.1616394502 * t);
    result += 2.9e-09 * std::cos(5.70141882483 + 77.673770428 * t);
    result += 2.55e-09 * std::cos(4.0093966444 + 5881.4037282342 * t);
    result += 2.53e-09 * std::cos(4.73318493678 + 16723.350142595 * t);
    result += 2.28e-09 * std::cos(0.95333661324 + 5540.0857894588 * t);
    result += 3.19e-09 * std::cos(1.38633229189 + 163096.180361183 * t);
    result += 2.24e-09 * std::cos(1.65156322696 + 10027.9031957292 * t);
    result += 2.26e-09 * std::cos(0.34106460604 + 17796.9591667858 * t);
    result += 2.36e-09 * std::cos(4.19817431922 + 19.66976089979 * t);
    result += 2.8e-09 * std::cos(4.1408026897 + 12539.853380183 * t);
    result += 2.75e-09 * std::cos(5.50306930248 + 32.5325507914 * t);
    result += 2.23e-09 * std::cos(5.23334210294 + 56.8983749356 * t);
    result += 2.17e-09 * std::cos(6.08587881787 + 6805.6532680852 * t);
    result += 2.8e-09 * std::cos(4.52472044653 + 6016.4688082696 * t);
    result += 2.27e-09 * std::cos(5.06509843737 + 6277.552925684 * t);
    result += 2.26e-09 * std::cos(5.17755154305 + 11720.0688652316 * t);
    result += 2.45e-09 * std::cos(3.96486270306 + 22.7752014508 * t);
    result += 2.2e-09 * std::cos(4.7207808197 + 6.62855890001 * t);
    result += 2.07e-09 * std::cos(5.71701403951 + 41.5507909848 * t);
    result += 2.04e-09 * std::cos(3.9122741125 + 2699.7348193176 * t);
    result += 2.09e-09 * std::cos(0.86881969011 + 6321.1035226272 * t);
    result += 2e-09 * std::cos(2.11984445273 + 4274.5183108324 * t);
    result += 2e-09 * std::cos(5.39839888163 + 6019.9919266186 * t);
    result += 2.09e-09 * std::cos(5.67606291663 + 11293.4706743556 * t);
    result += 2.52e-09 * std::cos(1.64965729351 + 9380.9596727172 * t);
    result += 2.75e-09 * std::cos(5.04826903506 + 73.297125859 * t);
    result += 2.08e-09 * std::cos(1.88207277133 + 11300.5842213564 * t);
    result += 2.72e-09 * std::cos(0.74640926842 + 1975.492545856 * t);
    result += 1.99e-09 * std::cos(3.30836672397 + 22743.4093795164 * t);
    result += 2.69e-09 * std::cos(4.48560812155 + 64471.9912417449 * t);
    result += 1.92e-09 * std::cos(2.17464236325 + 5863.5912061162 * t);
    result += 2.28e-09 * std::cos(5.85373115869 + 128.0188433374 * t);
    result += 2.61e-09 * std::cos(2.64321183295 + 55022.9357470744 * t);
    result += 2.2e-09 * std::cos(5.75012110079 + 29.429508536 * t);
    result += 1.87e-09 * std::cos(4.03230554718 + 467.9649903544 * t);
    result += 2e-09 * std::cos(5.60556112058 + 1066.49547719 * t);
    result += 2.31e-09 * std::cos(1.09802712785 + 12341.8069042809 * t);
    result += 1.99e-09 * std::cos(0.295006252 + 149.5631971346 * t);
    result += 2.49e-09 * std::cos(5.10473210814 + 7875.6718636242 * t);
    result += 2.08e-09 * std::cos(0.93013835019 + 14919.0178537546 * t);
    result += 1.79e-09 * std::cos(0.87104393079 + 12721.572099417 * t);
    result += 2.03e-09 * std::cos(1.56920753653 + 28286.9904848612 * t);
    result += 1.79e-09 * std::cos(2.47036386443 + 16062.1845261168 * t);
    result += 1.98e-09 * std::cos(3.54061588502 + 30.914125635 * t);
    result += 1.71e-09 * std::cos(3.45356518113 + 5327.4761083828 * t);
    result += 1.83e-09 * std::cos(0.72325421604 + 6272.0301497275 * t);
    result += 2.16e-09 * std::cos(2.97174580686 + 19402.7969528166 * t);
    result += 1.68e-09 * std::cos(2.51550550242 + 23937.856389741 * t);
    result += 1.95e-09 * std::cos(0.09045393425 + 156.4007205024 * t);
    result += 1.79e-09 * std::cos(4.4947179809 + 31415.379249957 * t);
    result += 2.16e-09 * std::cos(0.42177594328 + 23539.7073863328 * t);
    result += 1.89e-09 * std::cos(0.37542530191 + 9814.6041002912 * t);
    result += 2.18e-09 * std::cos(2.36835880025 + 16627.3709153772 * t);
    result += 1.66e-09 * std::cos(4.23182968446 + 16840.6700108152 * t);
    result += 2e-09 * std::cos(2.02153258098 + 16097.6799502826 * t);
    result += 1.69e-09 * std::cos(0.91318727 + 95.9792272178 * t);
    result += 2.11e-09 * std::cos(5.73370637657 + 151.8972810852 * t);
    result += 2.04e-09 * std::cos(0.42643085174 + 515.463871093 * t);
    result += 2.12e-09 * std::cos(3.00233538977 + 12043.574281889 * t);
    result += 1.92e-09 * std::cos(5.46153589821 + 6379.0550772092 * t);
    result += 1.65e-09 * std::cos(1.38698167064 + 4171.4255366138 * t);
    result += 1.6e-09 * std::cos(6.23798383332 + 202.2533951741 * t);
    result += 2.15e-09 * std::cos(0.20889073407 + 5621.8429232104 * t);
    result += 1.81e-09 * std::cos(4.12439203622 + 13341.6743113068 * t);
    result += 1.53e-09 * std::cos(1.24460848836 + 29826.3063546732 * t);
    result += 1.5e-09 * std::cos(3.12999753018 + 799.8211251654 * t);
    result += 1.75e-09 * std::cos(4.55671604437 + 239424.390254353 * t);
    result += 1.92e-09 * std::cos(1.33928820063 + 394.6258850592 * t);
    result += 1.49e-09 * std::cos(2.65697593276 + 21.335640467 * t);
    result += 1.46e-09 * std::cos(5.58021191726 + 412.3710968744 * t);
    result += 1.56e-09 * std::cos(3.75650175503 + 12323.4230960088 * t);
    result += 1.43e-09 * std::cos(3.75708566606 + 58864.5439181463 * t);
    result += 1.43e-09 * std::cos(3.28248547724 + 29.8214381488 * t);
    result += 1.44e-09 * std::cos(1.07862546598 + 1265.5674786264 * t);
    result += 1.48e-09 * std::cos(0.23389236655 + 10021.8372800994 * t);
    result += 1.93e-09 * std::cos(5.92751083086 + 40879.4405046438 * t);
    result += 1.4e-09 * std::cos(4.97612440269 + 158.9435177832 * t);
    result += 1.48e-09 * std::cos(2.61640453469 + 17157.0618804718 * t);
    result += 1.41e-09 * std::cos(3.66871308723 + 26084.0218062162 * t);
    result += 1.47e-09 * std::cos(5.09968173403 + 661.232926781 * t);
    result += 1.46e-09 * std::cos(4.96885605695 + 57375.8019008462 * t);
    result += 1.42e-09 * std::cos(0.78678347839 + 12779.4507954208 * t);
    result += 1.34e-09 * std::cos(4.79432636012 + 111.1866422876 * t);
    result += 1.4e-09 * std::cos(1.27748013377 + 107.6635239386 * t);
    result += 1.69e-09 * std::cos(2.74893543762 + 26735.9452622132 * t);
    result += 1.65e-09 * std::cos(3.95288000638 + 6357.8574485587 * t);
    result += 1.83e-09 * std::cos(5.43418358741 + 369.6998159404 * t);
    result += 1.34e-09 * std::cos(3.09132862833 + 17.812522118 * t);
    result += 1.32e-09 * std::cos(3.05633896779 + 22490.9621214934 * t);
    result += 1.34e-09 * std::cos(4.09472795832 + 6599.467719648 * t);
    result += 1.81e-09 * std::cos(4.22950689891 + 966.9708774356 * t);
    result += 1.52e-09 * std::cos(5.28885894415 + 12669.2444742014 * t);
    result += 1.5e-09 * std::cos(5.86819430908 + 97238.6275444875 * t);
    result += 1.42e-09 * std::cos(5.87266532526 + 22476.7350274918 * t);
    result += 1.45e-09 * std::cos(5.07330784304 + 87.30820453981 * t);
    result += 1.33e-09 * std::cos(5.65471067133 + 31.9723058168 * t);
    result += 1.24e-09 * std::cos(2.83326217072 + 12566.2190102856 * t);
    result += 1.35e-09 * std::cos(3.12861731644 + 32217.2001810808 * t);
    result += 1.37e-09 * std::cos(0.86487461904 + 9924.8104215106 * t);
    result += 1.72e-09 * std::cos(1.98369595114 + 174242.46596405 * t);
    result += 1.7e-09 * std::cos(4.41115280254 + 327574.514276781 * t);
    result += 1.51e-09 * std::cos(0.46542099527 + 39609.6545831656 * t);
    result += 1.48e-09 * std::cos(2.13439571118 + 491.6632924588 * t);
    result += 1.53e-09 * std::cos(3.78801830344 + 17363.247428909 * t);
    result += 1.65e-09 * std::cos(5.31654110459 + 16943.7627850338 * t);
    result += 1.65e-09 * std::cos(4.06747587817 + 58953.145443294 * t);
    result += 1.18e-09 * std::cos(0.63846333239 + 6.0659156298 * t);
    result += 1.59e-09 * std::cos(0.86086959274 + 221995.028801495 * t);
    result += 1.19e-09 * std::cos(5.96432932413 + 1385.8952763362 * t);
    result += 1.14e-09 * std::cos(5.16516114595 + 25685.872802808 * t);
    result += 1.12e-09 * std::cos(3.39403722178 + 21393.5419698576 * t);
    result += 1.12e-09 * std::cos(4.92889233335 + 56.8032621698 * t);
    result += 1.19e-09 * std::cos(2.40637635942 + 18635.9284545362 * t);
    result += 1.15e-09 * std::cos(0.23374479051 + 418.9243989006 * t);
    result += 1.22e-09 * std::cos(0.93575234049 + 24492.4061136516 * t);
    result += 1.15e-09 * std::cos(4.58880032176 + 26709.6469424134 * t);
    result += 1.3e-09 * std::cos(4.85539251 + 22345.2603761082 * t);
    result += 1.4e-09 * std::cos(1.09413073202 + 44809.6502008634 * t);
    result += 1.12e-09 * std::cos(6.05401806281 + 433.7117378768 * t);
    result += 1.04e-09 * std::cos(1.54931540602 + 127.9515330346 * t);
    result += 1.05e-09 * std::cos(4.82620858888 + 33794.5437235286 * t);
    result += 1.02e-09 * std::cos(4.12448497391 + 15664.0355227086 * t);
    result += 1.07e-09 * std::cos(4.67919356465 + 77690.7595057385 * t);
    result += 1.18e-09 * std::cos(4.5232017012 + 19004.6479494084 * t);
    result += 1.07e-09 * std::cos(5.71774478555 + 77736.7834305025 * t);
    result += 1.43e-09 * std::cos(1.81201813018 + 4214.0690150848 * t);
    result += 1.25e-09 * std::cos(1.14419195615 + 625.6701923124 * t);
    result += 1.24e-09 * std::cos(3.27736514057 + 12566.08438968 * t);
    result += 1.1e-09 * std::cos(1.08682570828 + 2787.0430238574 * t);
    result += 1.05e-09 * std::cos(1.78318141871 + 18139.2945014159 * t);
    result += 1.02e-09 * std::cos(4.75119578149 + 12242.6462833254 * t);
    result += 1.37e-09 * std::cos(1.43510636754 + 86464.6133168312 * t);
    result += 1.01e-09 * std::cos(4.91289409429 + 401.6721217572 * t);
    result += 1.29e-09 * std::cos(1.23567904485 + 12029.3471878874 * t);
    result += 1.38e-09 * std::cos(2.45654707999 + 7576.560073574 * t);
    result += 1.03e-09 * std::cos(0.40004073416 + 90279.9231681033 * t);
    result += 1.08e-09 * std::cos(0.9898977494 + 5636.0650166766 * t);
    result += 1.17e-09 * std::cos(5.17362872063 + 34520.3093093808 * t);
    result += 1e-09 * std::cos(3.95534628189 + 5547.1993364596 * t);
    result += 9.8e-10 * std::cos(1.28118280598 + 21548.9623692918 * t);
    result += 9.7e-10 * std::cos(3.34717130592 + 16310.9790457206 * t);
    result += 9.8e-10 * std::cos(4.37041908717 + 34513.2630726828 * t);
    result += 1.25e-09 * std::cos(2.7216443296 + 24065.8079227756 * t);
    result += 1.02e-09 * std::cos(0.66938025772 + 10239.5838660108 * t);
    result += 1.19e-09 * std::cos(1.21689479331 + 1478.8665740644 * t);
    result += 9.4e-10 * std::cos(1.99595224256 + 13362.4497067992 * t);
    result += 9.4e-10 * std::cos(4.30965982872 + 26880.3198130326 * t);
    result += 9.5e-10 * std::cos(2.89807657534 + 34911.412076091 * t);
    result += 1.06e-09 * std::cos(1.0015665359 + 16522.6597160022 * t);
    result += 9.7e-10 * std::cos(0.89642320201 + 71980.6335747312 * t);
    result += 1.16e-09 * std::cos(4.19967201116 + 206.7007372966 * t);
    result += 9.9e-10 * std::cos(1.37437847718 + 1039.0266107904 * t);
    result += 1.26e-09 * std::cos(3.21642544972 + 305281.943071049 * t);
    result += 9.4e-10 * std::cos(0.6899787606 + 7834.1210726394 * t);
    result += 9.4e-10 * std::cos(5.58132218606 + 3104.9300594238 * t);
    result += 9.5e-10 * std::cos(3.0382374111 + 8982.810669309 * t);
    result += 1.08e-09 * std::cos(0.52696637156 + 276.7457718644 * t);
    result += 1.24e-09 * std::cos(3.43899862683 + 172146.97134054 * t);
    result += 1.02e-09 * std::cos(1.04031728553 + 95143.1329209781 * t);
    result += 1.04e-09 * std::cos(3.39218586218 + 290.972865866 * t);
    result += 1.1e-09 * std::cos(3.68205877433 + 22380.755800274 * t);
    result += 1.17e-09 * std::cos(0.78475956902 + 83286.9142695536 * t);
    result += 8.3e-10 * std::cos(0.18241793425 + 15141.390794312 * t);
    result += 8.9e-10 * std::cos(4.45371820659 + 792.7748884674 * t);
    result += 8.2e-10 * std::cos(4.80703651241 + 6819.8803620868 * t);
    result += 8.7e-10 * std::cos(3.43122851097 + 27707.5424942948 * t);
    result += 1.01e-09 * std::cos(5.32081603011 + 2301.58581590939 * t);
    result += 8.2e-10 * std::cos(0.87060089842 + 10241.2022911672 * t);
    result += 8.6e-10 * std::cos(4.61919461931 + 36147.4098773004 * t);
    result += 9.5e-10 * std::cos(2.87032884659 + 23020.653086588 * t);
    result += 8.8e-10 * std::cos(3.2113316569 + 33326.5787331742 * t);
    result += 8e-10 * std::cos(1.84900424847 + 21424.4666443034 * t);
    result += 1.01e-09 * std::cos(4.18796434479 + 30666.1549584328 * t);
    result += 1.07e-09 * std::cos(5.77864921649 + 34115.1140692746 * t);
    result += 1.04e-09 * std::cos(1.08739495962 + 6288.5987742988 * t);
    result += 1.1e-09 * std::cos(3.32898859416 + 72140.6286666874 * t);
    result += 8.7e-10 * std::cos(4.40657711727 + 142.1786270362 * t);
    result += 1.09e-09 * std::cos(1.94546030825 + 24279.1070182136 * t);
    result += 8.7e-10 * std::cos(4.32472045435 + 742.9900605326 * t);
    result += 1.07e-09 * std::cos(4.91580912547 + 277.0349937414 * t);
    result += 8.8e-10 * std::cos(2.10180220766 + 26482.1708096244 * t);
    result += 8.6e-10 * std::cos(4.01887374432 + 12491.3701014155 * t);
    result += 1.06e-09 * std::cos(5.49092372854 + 62883.3551395136 * t);
    result += 8e-10 * std::cos(6.19781316983 + 6709.6740408674 * t);
    result += 8.8e-10 * std::cos(2.09872810657 + 238004.524157236 * t);
    result += 8.3e-10 * std::cos(4.90662164029 + 51.28033786241 * t);
    result += 9.5e-10 * std::cos(4.13387406591 + 18216.443810661 * t);
    result += 7.8e-10 * std::cos(6.0694939168 + 148434.534037691 * t);
    result += 7.9e-10 * std::cos(3.03048221644 + 838.9692877504 * t);
    result += 7.4e-10 * std::cos(5.49813051211 + 29026.4852295078 * t);
    result += 7.3e-10 * std::cos(3.05008665738 + 567.7186377304 * t);
    result += 8.4e-10 * std::cos(0.46604373274 + 45.1412196366 * t);
    result += 9.3e-10 * std::cos(2.52267536308 + 48739.859897083 * t);
    result += 7.6e-10 * std::cos(1.76418124905 + 41654.9631159678 * t);
    result += 6.7e-10 * std::cos(5.77851227793 + 6311.5250374592 * t);
    result += 6.2e-10 * std::cos(3.32967880172 + 15508.6151232744 * t);
    result += 7.9e-10 * std::cos(5.59773841328 + 71960.3865832237 * t);
    result += 5.7e-10 * std::cos(3.90629505268 + 5999.2165311262 * t);
    result += 6.1e-10 * std::cos(0.05695043232 + 7856.89627409019 * t);
    result += 6.1e-10 * std::cos(5.63297958433 + 7863.9425107882 * t);
    result += 6.5e-10 * std::cos(3.72178394016 + 12573.2652469836 * t);
    result += 5.7e-10 * std::cos(4.18217219541 + 26087.9031415742 * t);
    result += 6.6e-10 * std::cos(3.92262333487 + 69853.3520756813 * t);
    result += 5.3e-10 * std::cos(5.51119362045 + 77710.2483497715 * t);
    result += 5.3e-10 * std::cos(4.88573986961 + 77717.2945864695 * t);
    result += 6.2e-10 * std::cos(2.88876342225 + 9411.4646150872 * t);
    result += 5.1e-10 * std::cos(1.12657183874 + 82576.9812209953 * t);
    result += 4.5e-10 * std::cos(2.95671076719 + 24602.612434871 * t);
    result += 4e-10 * std::cos(5.55145719241 + 12565.1713789146 * t);
    result += 3.9e-10 * std::cos(1.20838190039 + 18842.1140029734 * t);
    result += 4.5e-10 * std::cos(3.18590558749 + 45585.1728121874 * t);
    result += 4.9e-10 * std::cos(2.44790934886 + 13613.804277336 * t);
    return result;
}

constexpr double getEarthL1(double t)
{
    double result = 0.0;
    result += 6283.31966747491;
    result += 0.00206058863 * std::cos(2.67823455584 + 6283.0758499914 * t);
    result += 4.30343e-05 * std::cos(2.63512650414 + 12566.1516999828 * t);
    result += 4.25264e-06 * std::cos(1.59046980729 + 3.523118349 * t);
    result += 1.08977e-06 * std::cos(2.96618001993 + 1577.3435424478 * t);
    result += 9.3478e-07 * std::cos(2.59212835365 + 18849.2275499742 * t);
    result += 1.19261e-06 * std::cos(5.79557487799 + 26.2983197998 * t);
    result += 7.2122e-07 * std::cos(1.13846158196 + 529.6909650946 * t);
    result += 6.7768e-07 * std::cos(1.87472304791 + 398.1490034082 * t);
    result += 6.7327e-07 * std::cos(4.40918235168 + 5507.5532386674 * t);
    result += 5.9027e-07 * std::cos(2.8879703846 + 5223.6939198022 * t);
    result += 5.5976e-07 * std::cos(2.17471680261 + 155.4203994342 * t);
    result += 4.5407e-07 * std::cos(0.39803079805 + 796.2980068164 * t);
    result += 3.6369e-07 * std::cos(0.46624739835 + 775.522611324 * t);
    result += 2.8958e-07 * std::cos(2.64707383882 + 7.1135470008 * t);
    result += 1.9097e-07 * std::cos(1.84628332577 + 5486.777843175 * t);
    result += 2.0844e-07 * std::cos(5.34138275149 + 0.9803210682 * t);
    result += 1.8508e-07 * std::cos(4.96855124577 + 213.299095438 * t);
    result += 1.6233e-07 * std::cos(0.03216483047 + 2544.3144198834 * t);
    result += 1.7293e-07 * std::cos(2.99116864949 + 6275.9623029906 * t);
    result += 1.5832e-07 * std::cos(1.43049285325 + 2146.1654164752 * t);
    result += 1.4615e-07 * std::cos(1.20532366323 + 10977.078804699 * t);
    result += 1.1877e-07 * std::cos(3.25804815607 + 5088.6288397668 * t);
    result += 1.1514e-07 * std::cos(2.07502418155 + 4694.0029547076 * t);
    result += 9.721e-08 * std::cos(4.23925472239 + 1349.8674096588 * t);
    result += 9.969e-08 * std::cos(1.30262991097 + 6286.5989683404 * t);
    result += 9.452e-08 * std::cos(2.69957062864 + 242.728603974 * t);
    result += 1.2461e-07 * std::cos(2.83432285512 + 1748.016413067 * t);
    result += 1.1808e-07 * std::cos(5.2737979048 + 1194.4470102246 * t);
    result += 8.577e-08 * std::cos(5.64475868067 + 951.7184062506 * t);
    result += 1.0641e-07 * std::cos(0.76614199202 + 553.5694028424 * t);
    result += 7.576e-08 * std::cos(5.30062664886 + 2352.8661537718 * t);
    result += 5.834e-08 * std::cos(1.76649917904 + 1059.3819301892 * t);
    result += 6.385e-08 * std::cos(2.65033984967 + 9437.762934887 * t);
    result += 5.223e-08 * std::cos(5.66135767624 + 71430.6956181291 * t);
    result += 5.305e-08 * std::cos(0.90857521574 + 3154.6870848956 * t);
    result += 6.101e-08 * std::cos(4.66632584188 + 4690.4798363586 * t);
    result += 4.33e-08 * std::cos(0.24102555403 + 6812.766815086 * t);
    result += 5.041e-08 * std::cos(1.42490103709 + 6438.4962494256 * t);
    result += 4.259e-08 * std::cos(0.77355900599 + 10447.3878396044 * t);
    result += 5.198e-08 * std::cos(1.85353197345 + 801.8209311238 * t);
    result += 3.744e-08 * std::cos(2.00119516488 + 8031.0922630584 * t);
    result += 3.558e-08 * std::cos(2.42901552681 + 14143.4952424306 * t);
    result += 3.372e-08 * std::cos(3.86210700128 + 1592.5960136328 * t);
    result += 3.374e-08 * std::cos(0.88776219727 + 12036.4607348882 * t);
    result += 3.175e-08 * std::cos(3.18785710594 + 4705.7323075436 * t);
    result += 3.221e-08 * std::cos(0.61599835472 + 8429.2412664666 * t);
    result += 4.132e-08 * std::cos(5.23992859705 + 7084.8967811152 * t);
    result += 2.97e-08 * std::cos(6.07026318493 + 4292.3308329504 * t);
    result += 2.9e-08 * std::cos(2.32464208411 + 20.3553193988 * t);
    result += 3.504e-08 * std::cos(4.79975694359 + 6279.5527316424 * t);
    result += 2.95e-08 * std::cos(1.43108874817 + 5746.271337896 * t);
    result += 2.697e-08 * std::cos(4.80368225199 + 7234.794256242 * t);
    result += 2.531e-08 * std::cos(6.22290682655 + 6836.6452528338 * t);
    result += 2.745e-08 * std::cos(0.93466065396 + 5760.4984318976 * t);
    result += 3.25e-08 * std::cos(3.39954640038 + 7632.9432596502 * t);
    result += 2.277e-08 * std::cos(5.00277837672 + 17789.845619785 * t);
    result += 2.075e-08 * std::cos(3.95534978634 + 10213.285546211 * t);
    result += 2.061e-08 * std::cos(2.22411683077 + 5856.4776591154 * t);
    result += 2.252e-08 * std::cos(5.67166499885 + 11499.6562227928 * t);
    result += 2.148e-08 * std::cos(5.20184578235 + 11513.8833167944 * t);
    result += 1.886e-08 * std::cos(0.53198320577 + 3340.6124266998 * t);
    result += 1.875e-08 * std::cos(4.73511970207 + 83996.8473181119 * t);
    result += 2.06e-08 * std::cos(2.54987293999 + 25132.3033999656 * t);
    result += 1.794e-08 * std::cos(1.47435409831 + 4164.311989613 * t);
    result += 1.778e-08 * std::cos(3.02473091781 + 5.5229243074 * t);
    result += 2.029e-08 * std::cos(0.90960209983 + 6256.7775301916 * t);
    result += 2.075e-08 * std::cos(2.26767270157 + 522.5774180938 * t);
    result += 1.772e-08 * std::cos(3.02622802353 + 5753.3848848968 * t);
    result += 1.569e-08 * std::cos(6.12410242782 + 5216.5803728014 * t);
    result += 1.59e-08 * std::cos(4.63713748247 + 3.2863574178 * t);
    result += 1.542e-08 * std::cos(4.20004448567 + 13367.9726311066 * t);
    result += 1.427e-08 * std::cos(1.19088061711 + 3894.1818295422 * t);
    result += 1.375e-08 * std::cos(3.09301252193 + 135.0650800354 * t);
    result += 1.359e-08 * std::cos(4.24532506641 + 426.598190876 * t);
    result += 1.34e-08 * std::cos(5.76511818622 + 6040.3472460174 * t);
    result += 1.284e-08 * std::cos(3.08524663344 + 5643.1785636774 * t);
    result += 1.25e-08 * std::cos(3.07748157144 + 11926.2544136688 * t);
    result += 1.551e-08 * std::cos(3.07665451458 + 6681.2248533996 * t);
    result += 1.268e-08 * std::cos(2.09196018331 + 6290.1893969922 * t);
    result += 1.144e-08 * std::cos(3.24444699514 + 12168.0026965746 * t);
    result += 1.248e-08 * std::cos(3.44504937285 + 536.8045120954 * t);
    result += 1.118e-08 * std::cos(2.31829670425 + 16730.4636895958 * t);
    result += 1.105e-08 * std::cos(5.31966001019 + 23.8784377478 * t);
    result += 1.051e-08 * std::cos(3.75015946014 + 7860.4193924392 * t);
    result += 1.025e-08 * std::cos(2.44688534235 + 1990.745017041 * t);
    result += 9.62e-09 * std::cos(0.81771017882 + 3.881335358 * t);
    result += 9.1e-09 * std::cos(0.41727865299 + 7079.3738568078 * t);
    result += 8.83e-09 * std::cos(5.16833917651 + 11790.6290886588 * t);
    result += 9.57e-09 * std::cos(4.07673573735 + 6127.6554505572 * t);
    result += 1.11e-08 * std::cos(3.90096793825 + 11506.7697697936 * t);
    result += 8.02e-09 * std::cos(3.88778875582 + 10973.55568635 * t);
    result += 7.8e-09 * std::cos(2.39934293755 + 1589.0728952838 * t);
    result += 7.58e-09 * std::cos(1.30034364248 + 103.0927742186 * t);
    result += 7.49e-09 * std::cos(4.962758033 + 6496.3749454294 * t);
    result += 7.65e-09 * std::cos(3.36312388424 + 36.0278666774 * t);
    result += 9.15e-09 * std::cos(5.41543742089 + 206.1855484372 * t);
    result += 7.76e-09 * std::cos(2.57589093871 + 11371.7046897582 * t);
    result += 7.72e-09 * std::cos(3.98369209464 + 955.5997416086 * t);
    result += 7.49e-09 * std::cos(5.17890001805 + 10969.9652576982 * t);
    result += 8.06e-09 * std::cos(0.34218864254 + 9917.6968745098 * t);
    result += 7.28e-09 * std::cos(5.20962563787 + 38.0276726358 * t);
    result += 6.85e-09 * std::cos(2.77592961854 + 20.7753954924 * t);
    result += 6.36e-09 * std::cos(4.28242193632 + 28.4491874678 * t);
    result += 6.08e-09 * std::cos(5.63278508906 + 10984.1923516998 * t);
    result += 7.04e-09 * std::cos(5.60738823665 + 3738.761430108 * t);
    result += 6.85e-09 * std::cos(0.38876148682 + 15.252471185 * t);
    result += 6.01e-09 * std::cos(0.73489602442 + 419.4846438752 * t);
    result += 7.16e-09 * std::cos(2.65279791438 + 6309.3741697912 * t);
    result += 5.84e-09 * std::cos(5.54502568227 + 17298.1823273262 * t);
    result += 6.5e-09 * std::cos(1.13379656406 + 7058.5984613154 * t);
    result += 6.88e-09 * std::cos(2.59683891779 + 3496.032826134 * t);
    result += 4.85e-09 * std::cos(0.44467180946 + 12352.8526045448 * t);
    result += 5.28e-09 * std::cos(2.74936967681 + 3930.2096962196 * t);
    result += 5.97e-09 * std::cos(5.27668281777 + 10575.4066829418 * t);
    result += 5.83e-09 * std::cos(3.1892906781 + 4732.0306273434 * t);
    result += 5.26e-09 * std::cos(5.01697321546 + 5884.9268465832 * t);
    result += 5.4e-09 * std::cos(1.29175137075 + 640.8776073822 * t);
    result += 4.73e-09 * std::cos(5.4995330697 + 5230.807466803 * t);
    result += 4.06e-09 * std::cos(5.21248452189 + 220.4126424388 * t);
    result += 3.95e-09 * std::cos(1.87474483222 + 16200.7727245012 * t);
    result += 3.7e-09 * std::cos(3.84921354713 + 18073.7049386502 * t);
    result += 3.67e-09 * std::cos(0.88533542778 + 6283.14316029419 * t);
    result += 3.79e-09 * std::cos(0.37983009325 + 10177.2576795336 * t);
    result += 3.56e-09 * std::cos(3.84145204913 + 11712.9553182308 * t);
    result += 3.74e-09 * std::cos(5.01577520608 + 7.046236698 * t);
    result += 3.81e-09 * std::cos(4.30250406634 + 6062.6632075526 * t);
    result += 4.71e-09 * std::cos(0.86381834647 + 6069.7767545534 * t);
    result += 3.67e-09 * std::cos(1.32943839763 + 6283.0085396886 * t);
    result += 4.6e-09 * std::cos(5.19667219575 + 6284.0561710596 * t);
    result += 3.33e-09 * std::cos(5.54256205741 + 4686.8894077068 * t);
    result += 3.41e-09 * std::cos(4.36522989934 + 7238.6755916 * t);
    result += 3.36e-09 * std::cos(4.00205876835 + 3097.88382272579 * t);
    result += 3.59e-09 * std::cos(6.22679790284 + 245.8316462294 * t);
    result += 3.07e-09 * std::cos(2.35299010924 + 170.6728706192 * t);
    result += 3.43e-09 * std::cos(3.77164927143 + 6076.8903015542 * t);
    result += 2.96e-09 * std::cos(5.44152227481 + 17260.1546546904 * t);
    result += 3.28e-09 * std::cos(0.13837875384 + 11015.1064773348 * t);
    result += 2.68e-09 * std::cos(1.1390455063 + 12569.6748183318 * t);
    result += 2.63e-09 * std::cos(0.00538633678 + 4136.9104335162 * t);
    result += 2.82e-09 * std::cos(5.0439983748 + 7477.522860216 * t);
    result += 2.88e-09 * std::cos(3.13401177517 + 12559.038152982 * t);
    result += 2.59e-09 * std::cos(0.93882269387 + 5642.1982426092 * t);
    result += 2.92e-09 * std::cos(1.98420020514 + 12132.439962106 * t);
    result += 2.47e-09 * std::cos(3.84244798532 + 5429.8794682394 * t);
    result += 2.45e-09 * std::cos(5.70467521726 + 65147.6197681377 * t);
    result += 2.41e-09 * std::cos(0.99480969552 + 3634.6210245184 * t);
    result += 2.46e-09 * std::cos(3.06168069935 + 110.2063212194 * t);
    result += 2.39e-09 * std::cos(6.11855909114 + 11856.2186514245 * t);
    result += 2.63e-09 * std::cos(0.66348415419 + 21228.3920235458 * t);
    result += 2.62e-09 * std::cos(1.51070507866 + 12146.6670561076 * t);
    result += 2.3e-09 * std::cos(1.75927314884 + 9779.1086761254 * t);
    result += 2.23e-09 * std::cos(2.00967043606 + 6172.869528772 * t);
    result += 2.46e-09 * std::cos(1.10411690865 + 6282.0955289232 * t);
    result += 2.21e-09 * std::cos(3.03945240854 + 8635.9420037632 * t);
    result += 2.14e-09 * std::cos(4.03840869663 + 14314.1681130498 * t);
    result += 2.36e-09 * std::cos(5.4691507058 + 13916.0191096416 * t);
    result += 2.24e-09 * std::cos(4.68408089456 + 24072.9214697764 * t);
    result += 2.12e-09 * std::cos(2.13695625494 + 5849.3641121146 * t);
    result += 2.07e-09 * std::cos(3.07724246401 + 11.729352836 * t);
    result += 2.07e-09 * std::cos(6.10306282747 + 23543.2305046818 * t);
    result += 2.66e-09 * std::cos(1.00709566823 + 2388.8940204492 * t);
    result += 2.17e-09 * std::cos(6.27837036335 + 17267.2682016912 * t);
    result += 2.04e-09 * std::cos(2.34615348695 + 266.6070417218 * t);
    result += 1.95e-09 * std::cos(5.55015549753 + 6133.5126528568 * t);
    result += 1.88e-09 * std::cos(2.52667166175 + 6525.8044539654 * t);
    result += 1.85e-09 * std::cos(0.90960768344 + 18319.5365848796 * t);
    result += 1.77e-09 * std::cos(1.73429218289 + 154717.609887683 * t);
    result += 1.87e-09 * std::cos(4.76483647432 + 4535.0594369244 * t);
    result += 1.86e-09 * std::cos(4.63080493407 + 10440.2742926036 * t);
    result += 2.15e-09 * std::cos(2.8125545456 + 7342.4577801806 * t);
    result += 1.72e-09 * std::cos(1.45551888559 + 9225.539273283 * t);
    result += 1.62e-09 * std::cos(3.30661909388 + 639.897286314 * t);
    result += 1.68e-09 * std::cos(2.17671416605 + 27.4015560968 * t);
    result += 1.6e-09 * std::cos(1.68164180475 + 15110.4661198662 * t);
    result += 1.58e-09 * std::cos(0.13519771874 + 13095.8426650774 * t);
    result += 1.83e-09 * std::cos(0.56281322071 + 13517.8701062334 * t);
    result += 1.79e-09 * std::cos(3.58450811616 + 87.30820453981 * t);
    result += 1.52e-09 * std::cos(2.84070476818 + 5650.2921106782 * t);
    result += 1.82e-09 * std::cos(0.44065530624 + 17253.0411076896 * t);
    result += 1.6e-09 * std::cos(5.95767264171 + 4701.1165017084 * t);
    result += 1.42e-09 * std::cos(1.4629013752 + 11087.2851259184 * t);
    result += 1.42e-09 * std::cos(2.04464036087 + 20426.571092422 * t);
    result += 1.31e-09 * std::cos(5.40912137746 + 2699.7348193176 * t);
    result += 1.44e-09 * std::cos(2.07312090485 + 25158.6017197654 * t);
    result += 1.47e-09 * std::cos(6.15106982168 + 9623.6882766912 * t);
    result += 1.41e-09 * std::cos(5.55739979498 + 10454.5013866052 * t);
    result += 1.35e-09 * std::cos(0.06098110407 + 16723.350142595 * t);
    result += 1.24e-09 * std::cos(5.81218025669 + 17256.6315363414 * t);
    result += 1.24e-09 * std::cos(2.36293551623 + 4933.2084403326 * t);
    result += 1.26e-09 * std::cos(3.47435905118 + 22483.8485744926 * t);
    result += 1.59e-09 * std::cos(5.63954754618 + 5729.506447149 * t);
    result += 1.23e-09 * std::cos(3.92815963256 + 17996.0311682222 * t);
    result += 1.48e-09 * std::cos(3.02509280598 + 1551.045222648 * t);
    result += 1.2e-09 * std::cos(5.91904349732 + 6206.8097787158 * t);
    result += 1.34e-09 * std::cos(3.11122937825 + 21954.157609398 * t);
    result += 1.19e-09 * std::cos(5.5214112345 + 709.9330485583 * t);
    result += 1.22e-09 * std::cos(3.00813429479 + 19800.9459562248 * t);
    result += 1.27e-09 * std::cos(1.37618620001 + 14945.3161735544 * t);
    result += 1.41e-09 * std::cos(2.56889468729 + 1052.2683831884 * t);
    result += 1.23e-09 * std::cos(2.83671175442 + 11919.140866668 * t);
    result += 1.18e-09 * std::cos(0.81934438215 + 5331.3574437408 * t);
    result += 1.51e-09 * std::cos(2.68731829165 + 11769.8536931664 * t);
    result += 1.19e-09 * std::cos(5.08835797638 + 5481.2549188676 * t);
    result += 1.53e-09 * std::cos(2.46021790779 + 11933.3679606696 * t);
    result += 1.08e-09 * std::cos(1.04936452145 + 11403.676995575 * t);
    result += 1.28e-09 * std::cos(0.99794735107 + 8827.3902698748 * t);
    result += 1.44e-09 * std::cos(2.54869747042 + 227.476132789 * t);
    result += 1.5e-09 * std::cos(4.50631437136 + 2379.1644735716 * t);
    result += 1.07e-09 * std::cos(1.79272017026 + 13119.7211028252 * t);
    result += 1.07e-09 * std::cos(4.43556814486 + 18422.6293590982 * t);
    result += 1.09e-09 * std::cos(0.29269062317 + 16737.5772365966 * t);
    result += 1.41e-09 * std::cos(3.18979826258 + 6262.300454499 * t);
    result += 1.22e-09 * std::cos(4.23040027813 + 29.429508536 * t);
    result += 1.11e-09 * std::cos(5.16954029551 + 17782.7320727842 * t);
    result += 1e-09 * std::cos(3.52213872761 + 18052.9295431578 * t);
    result += 1.08e-09 * std::cos(1.08514212991 + 16858.4825329332 * t);
    result += 1.06e-09 * std::cos(1.9608524841 + 74.7815985673 * t);
    result += 1.1e-09 * std::cos(2.30582372873 + 16460.333529525 * t);
    result += 9.7e-10 * std::cos(3.5091894021 + 5333.9002410216 * t);
    result += 9.9e-10 * std::cos(3.56417337974 + 735.8765135318 * t);
    result += 9.4e-10 * std::cos(5.01857894228 + 3128.3887650958 * t);
    result += 9.7e-10 * std::cos(1.65579893894 + 533.2140834436 * t);
    result += 9.2e-10 * std::cos(0.89217162285 + 29296.6153895786 * t);
    result += 1.23e-09 * std::cos(3.16062050433 + 9380.9596727172 * t);
    result += 1.02e-09 * std::cos(1.20493500565 + 23020.653086588 * t);
    result += 8.8e-10 * std::cos(2.21296088224 + 12721.572099417 * t);
    result += 8.9e-10 * std::cos(1.5426472031 + 20199.094959633 * t);
    result += 1.13e-09 * std::cos(4.8332070787 + 16496.3613962024 * t);
    result += 1.21e-09 * std::cos(6.19860353182 + 9388.0059094152 * t);
    result += 8.9e-10 * std::cos(4.08082274765 + 22805.7355659936 * t);
    result += 9.8e-10 * std::cos(1.0918183283 + 12043.574281889 * t);
    result += 8.6e-10 * std::cos(1.13655027605 + 143571.324284816 * t);
    result += 8.8e-10 * std::cos(5.96980472191 + 107.6635239386 * t);
    result += 8.2e-10 * std::cos(5.01340404594 + 22003.9146348698 * t);
    result += 9.4e-10 * std::cos(1.69615700473 + 23006.4259925864 * t);
    result += 8.1e-10 * std::cos(3.00657814365 + 2118.7638603784 * t);
    result += 9.8e-10 * std::cos(1.39215287161 + 8662.240323563 * t);
    result += 7.7e-10 * std::cos(3.3355519084 + 15720.8387848784 * t);
    result += 8.2e-10 * std::cos(5.86880116464 + 2787.0430238574 * t);
    result += 7.6e-10 * std::cos(5.67183650604 + 14.2270940016 * t);
    result += 8.1e-10 * std::cos(6.16619455699 + 1039.0266107904 * t);
    result += 7.6e-10 * std::cos(3.21449884756 + 111.1866422876 * t);
    result += 7.8e-10 * std::cos(1.37531518377 + 21947.1113727 * t);
    result += 7.4e-10 * std::cos(3.58814195051 + 11609.8625440122 * t);
    result += 7.7e-10 * std::cos(4.84846488388 + 22743.4093795164 * t);
    result += 9e-10 * std::cos(1.48869013606 + 15671.0817594066 * t);
    result += 8.2e-10 * std::cos(3.48618399109 + 29088.811415985 * t);
    result += 6.9e-10 * std::cos(3.55746476593 + 4590.910180489 * t);
    result += 6.9e-10 * std::cos(1.93625656075 + 135.62532501 * t);
    result += 7e-10 * std::cos(2.66548322237 + 18875.525869774 * t);
    result += 6.9e-10 * std::cos(5.41478093731 + 26735.9452622132 * t);
    result += 7.9e-10 * std::cos(5.15154513662 + 12323.4230960088 * t);
    result += 9.4e-10 * std::cos(3.62899392448 + 77713.7714681205 * t);
    result += 7.8e-10 * std::cos(4.17011182047 + 1066.49547719 * t);
    result += 7.1e-10 * std::cos(3.89435637865 + 22779.4372461938 * t);
    result += 6.3e-10 * std::cos(4.53968787714 + 8982.810669309 * t);
    result += 6.9e-10 * std::cos(0.96028230548 + 14919.0178537546 * t);
    result += 7.6e-10 * std::cos(3.29092216589 + 2942.4634232916 * t);
    result += 6.3e-10 * std::cos(4.09167842893 + 16062.1845261168 * t);
    result += 6.5e-10 * std::cos(3.34580407184 + 51.28033786241 * t);
    result += 6.5e-10 * std::cos(5.75757544877 + 52670.0695933026 * t);
    result += 6.8e-10 * std::cos(5.75884067555 + 21424.4666443034 * t);
    result += 5.7e-10 * std::cos(5.4512239985 + 12592.4500197826 * t);
    result += 5.7e-10 * std::cos(5.25043362558 + 20995.3929664494 * t);
    result += 7.3e-10 * std::cos(0.53299090807 + 2301.58581590939 * t);
    result += 7e-10 * std::cos(4.31243357502 + 19402.7969528166 * t);
    result += 6.7e-10 * std::cos(2.53852336668 + 377.3736079158 * t);
    result += 5.6e-10 * std::cos(3.20816844695 + 24889.5747959916 * t);
    result += 5.3e-10 * std::cos(3.17816599142 + 18451.078546566 * t);
    result += 5.3e-10 * std::cos(3.61529270216 + 77.673770428 * t);
    result += 5.3e-10 * std::cos(0.45467549335 + 30666.1549584328 * t);
    result += 6.1e-10 * std::cos(0.14807288453 + 23013.5395395872 * t);
    result += 5.1e-10 * std::cos(3.32803972907 + 56.8983749356 * t);
    result += 5.2e-10 * std::cos(3.41177624177 + 23141.5583829246 * t);
    result += 5.8e-10 * std::cos(3.13638677202 + 309.2783226558 * t);
    result += 7e-10 * std::cos(2.50592323465 + 31415.379249957 * t);
    result += 5.2e-10 * std::cos(5.10673376738 + 17796.9591667858 * t);
    result += 6.7e-10 * std::cos(6.27917920454 + 22345.2603761082 * t);
    result += 5e-10 * std::cos(0.42577644151 + 25685.872802808 * t);
    result += 4.8e-10 * std::cos(0.70204553333 + 1162.4747044078 * t);
    result += 6.6e-10 * std::cos(3.64350022359 + 15265.8865193004 * t);
    result += 5e-10 * std::cos(5.7438291744 + 19.66976089979 * t);
    result += 5e-10 * std::cos(4.69825387775 + 28237.2334593894 * t);
    result += 4.7e-10 * std::cos(5.74015846442 + 12139.5535091068 * t);
    result += 5.4e-10 * std::cos(1.97301333704 + 23581.2581773176 * t);
    result += 4.9e-10 * std::cos(4.98223579027 + 10021.8372800994 * t);
    result += 4.6e-10 * std::cos(5.41431705539 + 33019.0211122046 * t);
    result += 5.1e-10 * std::cos(1.23882053879 + 12539.853380183 * t);
    result += 4.6e-10 * std::cos(2.41369976086 + 98068.5367163054 * t);
    result += 4.4e-10 * std::cos(0.80750593746 + 167283.761587666 * t);
    result += 4.5e-10 * std::cos(4.39613584445 + 433.7117378768 * t);
    result += 4.4e-10 * std::cos(2.57358208785 + 12964.300703391 * t);
    result += 4.6e-10 * std::cos(0.26142733448 + 11.0457002639 * t);
    result += 4.5e-10 * std::cos(2.46230645202 + 51868.2486621788 * t);
    result += 4.8e-10 * std::cos(0.89551707131 + 56600.2792895222 * t);
    result += 5.7e-10 * std::cos(1.8641670701 + 25287.7237993998 * t);
    result += 4.2e-10 * std::cos(5.26377513431 + 26084.0218062162 * t);
    result += 4.9e-10 * std::cos(3.17757670611 + 6303.8512454838 * t);
    result += 5.2e-10 * std::cos(3.65266055509 + 7872.1487452752 * t);
    result += 4e-10 * std::cos(1.81891629936 + 34596.3646546524 * t);
    result += 4.3e-10 * std::cos(1.94164978061 + 1903.4368125012 * t);
    result += 4.1e-10 * std::cos(0.74461854136 + 23937.856389741 * t);
    result += 4.8e-10 * std::cos(6.26034008181 + 28286.9904848612 * t);
    result += 4.5e-10 * std::cos(5.4557501753 + 60530.4889857418 * t);
    result += 4e-10 * std::cos(2.92105728682 + 21548.9623692918 * t);
    result += 4e-10 * std::cos(0.04502010161 + 38526.574350872 * t);
    result += 5.3e-10 * std::cos(3.64791042082 + 11925.2740926006 * t);
    result += 4.1e-10 * std::cos(5.04048954693 + 27832.0382192832 * t);
    result += 4.2e-10 * std::cos(5.19292937193 + 19004.6479494084 * t);
    result += 4e-10 * std::cos(2.57120233428 + 24356.7807886416 * t);
    result += 3.8e-10 * std::cos(3.49190341464 + 226858.23855437 * t);
    result += 3.9e-10 * std::cos(4.61184303844 + 95.9792272178 * t);
    result += 4.3e-10 * std::cos(2.20648228147 + 13521.7514415914 * t);
    result += 4e-10 * std::cos(5.83461945819 + 16193.6591775004 * t);
    result += 4.5e-10 * std::cos(3.73714372195 + 7875.6718636242 * t);
    result += 4.3e-10 * std::cos(1.14078465002 + 49.7570254718 * t);
    result += 3.7e-10 * std::cos(1.29390383811 + 310.8407988684 * t);
    result += 3.8e-10 * std::cos(0.9597092595 + 664.75604513 * t);
    result += 3.7e-10 * std::cos(4.27532649462 + 6709.6740408674 * t);
    result += 3.8e-10 * std::cos(2.20108541046 + 28628.3362260996 * t);
    result += 3.9e-10 * std::cos(0.85957361635 + 16522.6597160022 * t);
    result += 4e-10 * std::cos(4.35214003837 + 48739.859897083 * t);
    result += 3.6e-10 * std::cos(1.68167662194 + 10344.2950653858 * t);
    result += 4e-10 * std::cos(5.13217319067 + 15664.0355227086 * t);
    result += 3.6e-10 * std::cos(3.72187132496 + 30774.5016425748 * t);
    result += 3.6e-10 * std::cos(3.32158458257 + 16207.886271502 * t);
    result += 4.5e-10 * std::cos(3.94202418608 + 10988.808157535 * t);
    result += 3.9e-10 * std::cos(1.51948786199 + 12029.3471878874 * t);
    result += 2.6e-10 * std::cos(3.8768588318 + 6262.7205305926 * t);
    result += 2.4e-10 * std::cos(4.91804163466 + 19651.048481098 * t);
    result += 2.3e-10 * std::cos(0.29300197709 + 13362.4497067992 * t);
    result += 2.1e-10 * std::cos(3.18605672363 + 6277.552925684 * t);
    result += 2.1e-10 * std::cos(6.07546891132 + 18139.2945014159 * t);
    result += 2.2e-10 * std::cos(2.31199937177 + 6303.4311693902 * t);
    result += 2.1e-10 * std::cos(3.58418394393 + 18209.3302636602 * t);
    result += 2.6e-10 * std::cos(2.068012969 + 12573.2652469836 * t);
    result += 2.1e-10 * std::cos(1.56857722317 + 13341.6743113068 * t);
    result += 2.4e-10 * std::cos(5.72605158675 + 29864.334027309 * t);
    result += 2.4e-10 * std::cos(1.40237993205 + 14712.317116458 * t);
    result += 2.5e-10 * std::cos(5.71466092822 + 25934.1243310894 * t);
    return result;
}

constexpr double getEarthL2(double t)
{
    double result = 0.0;
    result += 0.0005291887;
    result += 8.719837e-05 * std::cos(1.07209665242 + 6283.0758499914 * t);
    result += 3.09125e-06 * std::cos(0.86728818832 + 12566.1516999828 * t);
    result += 2.7339e-07 * std::cos(0.05297871691 + 3.523118349 * t);
    result += 1.6334e-07 * std::cos(5.18826691036 + 26.2983197998 * t);
    result += 1.5752e-07 * std::cos(3.6845788943 + 155.4203994342 * t);
    result += 9.541e-08 * std::cos(0.75742297675 + 18849.2275499742 * t);
    result += 8.937e-08 * std::cos(2.05705419118 + 77713.7714681205 * t);
    result += 6.952e-08 * std::cos(0.8267330541 + 775.522611324 * t);
    result += 5.064e-08 * std::cos(4.66284525271 + 1577.3435424478 * t);
    result += 4.061e-08 * std::cos(1.03057162962 + 7.1135470008 * t);
    result += 3.463e-08 * std::cos(5.14074632811 + 796.2980068164 * t);
    result += 3.169e-08 * std::cos(6.05291851171 + 5507.5532386674 * t);
    result += 3.02e-08 * std::cos(1.19246506441 + 242.728603974 * t);
    result += 2.886e-08 * std::cos(6.11652627155 + 529.6909650946 * t);
    result += 3.81e-08 * std::cos(3.4405080349 + 5573.1428014331 * t);
    result += 2.714e-08 * std::cos(0.30637881025 + 398.1490034082 * t);
    result += 2.371e-08 * std::cos(4.38118838167 + 5223.6939198022 * t);
    result += 2.538e-08 * std::cos(2.27992810679 + 553.5694028424 * t);
    result += 2.079e-08 * std::cos(3.75435330484 + 0.9803210682 * t);
    result += 1.675e-08 * std::cos(0.90216407959 + 951.7184062506 * t);
    result += 1.534e-08 * std::cos(5.75900462759 + 1349.8674096588 * t);
    result += 1.224e-08 * std::cos(2.97328088405 + 2146.1654164752 * t);
    result += 1.449e-08 * std::cos(4.3641591397 + 1748.016413067 * t);
    result += 1.341e-08 * std::cos(3.72061130861 + 1194.4470102246 * t);
    result += 1.254e-08 * std::cos(2.94846826628 + 6438.4962494256 * t);
    result += 9.99e-09 * std::cos(5.98640014468 + 6286.5989683404 * t);
    result += 9.17e-09 * std::cos(4.79788687522 + 5088.6288397668 * t);
    result += 8.28e-09 * std::cos(3.31321076572 + 213.299095438 * t);
    result += 1.103e-08 * std::cos(1.27104454479 + 161000.685737674 * t);
    result += 7.62e-09 * std::cos(3.41582762988 + 5486.777843175 * t);
    result += 1.044e-08 * std::cos(0.60409577691 + 3154.6870848956 * t);
    result += 8.87e-09 * std::cos(5.23465144638 + 7084.8967811152 * t);
    result += 6.45e-09 * std::cos(1.60096192515 + 2544.3144198834 * t);
    result += 6.81e-09 * std::cos(3.43155669169 + 4694.0029547076 * t);
    result += 6.05e-09 * std::cos(2.47806340546 + 10977.078804699 * t);
    result += 7.06e-09 * std::cos(6.19393222575 + 4690.4798363586 * t);
    result += 6.43e-09 * std::cos(1.98042503148 + 801.8209311238 * t);
    result += 5.02e-09 * std::cos(1.44394375363 + 6836.6452528338 * t);
    result += 4.9e-09 * std::cos(2.34129524194 + 1592.5960136328 * t);
    result += 4.58e-09 * std::cos(1.30876448575 + 4292.3308329504 * t);
    result += 4.31e-09 * std::cos(0.03526421494 + 7234.794256242 * t);
    result += 3.79e-09 * std::cos(3.17030522615 + 6309.3741697912 * t);
    result += 3.48e-09 * std::cos(0.99049550009 + 6040.3472460174 * t);
    result += 3.86e-09 * std::cos(1.57019797263 + 71430.6956181291 * t);
    result += 3.47e-09 * std::cos(0.67013291338 + 1059.3819301892 * t);
    result += 4.58e-09 * std::cos(3.81499443681 + 149854.400134808 * t);
    result += 3.02e-09 * std::cos(1.91760044838 + 10447.3878396044 * t);
    result += 3.07e-09 * std::cos(3.55343347416 + 8031.0922630584 * t);
    result += 3.95e-09 * std::cos(4.93701776616 + 7632.9432596502 * t);
    result += 3.14e-09 * std::cos(3.18093696547 + 2352.8661537718 * t);
    result += 2.82e-09 * std::cos(4.41936437052 + 9437.762934887 * t);
    result += 2.76e-09 * std::cos(2.71314254553 + 3894.1818295422 * t);
    result += 2.98e-09 * std::cos(2.5203747421 + 6127.6554505572 * t);
    result += 2.3e-09 * std::cos(1.37790215549 + 4705.7323075436 * t);
    result += 2.52e-09 * std::cos(0.55330133471 + 6279.5527316424 * t);
    result += 2.55e-09 * std::cos(5.26570187369 + 6812.766815086 * t);
    result += 2.75e-09 * std::cos(0.67264264272 + 25132.3033999656 * t);
    result += 1.78e-09 * std::cos(0.92820785174 + 1990.745017041 * t);
    result += 2.21e-09 * std::cos(0.63897368842 + 6256.7775301916 * t);
    result += 1.55e-09 * std::cos(0.77319790838 + 14143.4952424306 * t);
    result += 1.5e-09 * std::cos(2.40470465561 + 426.598190876 * t);
    result += 1.96e-09 * std::cos(6.06877865012 + 640.8776073822 * t);
    result += 1.37e-09 * std::cos(2.21679460145 + 8429.2412664666 * t);
    result += 1.27e-09 * std::cos(3.26094223174 + 17789.845619785 * t);
    result += 1.28e-09 * std::cos(5.47237279946 + 12036.4607348882 * t);
    result += 1.22e-09 * std::cos(2.16291082757 + 10213.285546211 * t);
    result += 1.18e-09 * std::cos(0.45789822268 + 7058.5984613154 * t);
    result += 1.41e-09 * std::cos(2.34932647403 + 11506.7697697936 * t);
    result += 1e-09 * std::cos(0.85621569847 + 6290.1893969922 * t);
    result += 9.2e-10 * std::cos(5.10587476002 + 7079.3738568078 * t);
    result += 1.26e-09 * std::cos(2.65428307012 + 88860.0570709867 * t);
    result += 1.06e-09 * std::cos(5.85646710022 + 7860.4193924392 * t);
    result += 8.4e-10 * std::cos(3.57457554262 + 16730.4636895958 * t);
    result += 8.9e-10 * std::cos(4.21433259618 + 83996.8473181119 * t);
    result += 9.7e-10 * std::cos(5.57938280855 + 13367.9726311066 * t);
    result += 1.02e-09 * std::cos(2.05853060226 + 87.30820453981 * t);
    result += 8e-10 * std::cos(4.73792651816 + 11926.2544136688 * t);
    result += 8e-10 * std::cos(5.41418965044 + 10973.55568635 * t);
    result += 1.06e-09 * std::cos(4.10978997399 + 3496.032826134 * t);
    result += 1.02e-09 * std::cos(3.62650006043 + 244287.600007228 * t);
    result += 7.5e-10 * std::cos(4.89483161769 + 5643.1785636774 * t);
    result += 8.7e-10 * std::cos(0.42863750683 + 11015.1064773348 * t);
    result += 6.9e-10 * std::cos(1.8890876072 + 10177.2576795336 * t);
    result += 8.9e-10 * std::cos(1.35567273119 + 6681.2248533996 * t);
    result += 6.6e-10 * std::cos(0.99455837265 + 6525.8044539654 * t);
    result += 6.7e-10 * std::cos(5.5124099707 + 3097.88382272579 * t);
    result += 7.6e-10 * std::cos(2.72016814799 + 4164.311989613 * t);
    result += 6.3e-10 * std::cos(1.4434990254 + 9917.6968745098 * t);
    result += 7.8e-10 * std::cos(3.51469733747 + 11856.2186514245 * t);
    result += 8.5e-10 * std::cos(0.50956043858 + 10575.4066829418 * t);
    result += 6.7e-10 * std::cos(3.62043033405 + 16496.3613962024 * t);
    result += 5.5e-10 * std::cos(5.24637517308 + 3340.6124266998 * t);
    result += 4.8e-10 * std::cos(5.43966777314 + 20426.571092422 * t);
    result += 6.4e-10 * std::cos(5.79535817813 + 2388.8940204492 * t);
    result += 4.6e-10 * std::cos(5.43499966519 + 6275.9623029906 * t);
    result += 5e-10 * std::cos(3.86263598617 + 5729.506447149 * t);
    result += 4.4e-10 * std::cos(1.52269529228 + 12168.0026965746 * t);
    result += 5.7e-10 * std::cos(4.96352373486 + 14945.3161735544 * t);
    result += 4.5e-10 * std::cos(1.0086123016 + 8635.9420037632 * t);
    result += 4.3e-10 * std::cos(3.30685683359 + 9779.1086761254 * t);
    result += 4.2e-10 * std::cos(0.6348125893 + 2699.7348193176 * t);
    result += 4.1e-10 * std::cos(5.67996766641 + 11712.9553182308 * t);
    result += 5.6e-10 * std::cos(4.34024451468 + 90955.5516944961 * t);
    result += 4.1e-10 * std::cos(5.81722212845 + 709.9330485583 * t);
    result += 5.3e-10 * std::cos(6.17052087143 + 233141.314404362 * t);
    result += 3.7e-10 * std::cos(3.12495025087 + 16200.7727245012 * t);
    result += 3.5e-10 * std::cos(5.76973458495 + 12569.6748183318 * t);
    result += 3.7e-10 * std::cos(0.31656444326 + 24356.7807886416 * t);
    result += 3.5e-10 * std::cos(0.96229051027 + 17298.1823273262 * t);
    result += 3.3e-10 * std::cos(5.23130355867 + 5331.3574437408 * t);
    result += 3.5e-10 * std::cos(0.62517020593 + 25158.6017197654 * t);
    result += 3.5e-10 * std::cos(0.80004512129 + 13916.0191096416 * t);
    result += 3.7e-10 * std::cos(2.89336088688 + 12721.572099417 * t);
    result += 3e-10 * std::cos(4.50198402401 + 23543.2305046818 * t);
    result += 3e-10 * std::cos(5.31355708693 + 18319.5365848796 * t);
    result += 2.9e-10 * std::cos(3.47275229977 + 13119.7211028252 * t);
    result += 2.9e-10 * std::cos(3.11002782516 + 4136.9104335162 * t);
    result += 3.2e-10 * std::cos(5.52273255667 + 5753.3848848968 * t);
    result += 3.5e-10 * std::cos(3.7969999668 + 143571.324284816 * t);
    result += 2.6e-10 * std::cos(1.50634201907 + 154717.609887683 * t);
    result += 3e-10 * std::cos(3.53519084118 + 6284.0561710596 * t);
    result += 2.3e-10 * std::cos(4.41808025967 + 5884.9268465832 * t);
    result += 2.5e-10 * std::cos(1.38477355808 + 65147.6197681377 * t);
    result += 2.3e-10 * std::cos(3.49782549797 + 7477.522860216 * t);
    result += 1.9e-10 * std::cos(3.14329413716 + 6496.3749454294 * t);
    result += 1.9e-10 * std::cos(2.20135125199 + 18073.7049386502 * t);
    result += 1.9e-10 * std::cos(4.95020255309 + 3930.2096962196 * t);
    result += 1.9e-10 * std::cos(0.57998702747 + 31415.379249957 * t);
    result += 2.1e-10 * std::cos(1.75474323399 + 12139.5535091068 * t);
    result += 1.9e-10 * std::cos(3.92233070499 + 19651.048481098 * t);
    result += 1.4e-10 * std::cos(0.98131213224 + 12559.038152982 * t);
    result += 1.9e-10 * std::cos(4.93309333729 + 2942.4634232916 * t);
    result += 1.6e-10 * std::cos(5.55997534558 + 8827.3902698748 * t);
    result += 1.3e-10 * std::cos(1.68808165516 + 4535.0594369244 * t);
    result += 1.3e-10 * std::cos(0.33982116161 + 4933.2084403326 * t);
    result += 1.2e-10 * std::cos(1.85426309994 + 5856.4776591154 * t);
    result += 1e-10 * std::cos(4.82763996845 + 13095.8426650774 * t);
    result += 1.1e-10 * std::cos(5.38005490571 + 11790.6290886588 * t);
    result += 1e-10 * std::cos(1.40815507226 + 10988.808157535 * t);
    result += 1.1e-10 * std::cos(3.05005267431 + 17260.1546546904 * t);
    result += 1e-10 * std::cos(4.93364992366 + 12352.8526045448 * t);
    return result;
}

constexpr double getEarthL3(double t)
{
    double result = 0.0;
    result += 2.89226e-06 * std::cos(5.84384198723 + 6283.0758499914 * t);
    result += 3.4955e-07;
    result += 1.6819e-07 * std::cos(5.48766912348 + 12566.1516999828 * t);
    result += 2.962e-08 * std::cos(5.19577265202 + 155.4203994342 * t);
    result += 1.288e-08 * std::cos(4.72200252235 + 3.523118349 * t);
    result += 6.35e-09 * std::cos(5.96925937141 + 242.728603974 * t);
    result += 7.14e-09 * std::cos(5.30045809128 + 18849.2275499742 * t);
    result += 4.02e-09 * std::cos(3.78682982419 + 553.5694028424 * t);
    result += 7.2e-10 * std::cos(4.2976812618 + 6286.5989683404 * t);
    result += 6.7e-10 * std::cos(0.90721687647 + 6127.6554505572 * t);
    result += 3.6e-10 * std::cos(5.24029648014 + 6438.4962494256 * t);
    result += 2.4e-10 * std::cos(5.16003960716 + 25132.3033999656 * t);
    result += 2.3e-10 * std::cos(3.01921570335 + 6309.3741697912 * t);
    result += 1.7e-10 * std::cos(5.82863573502 + 6525.8044539654 * t);
    result += 1.7e-10 * std::cos(3.6777286393 + 71430.6956181291 * t);
    result += 9e-11 * std::cos(4.58467294499 + 1577.3435424478 * t);
    result += 8e-11 * std::cos(1.40626662824 + 11856.2186514245 * t);
    result += 8e-11 * std::cos(5.07561257196 + 6256.7775301916 * t);
    result += 7e-11 * std::cos(2.82473374405 + 83996.8473181119 * t);
    result += 5e-11 * std::cos(2.71488713339 + 10977.078804699 * t);
    result += 5e-11 * std::cos(3.76879847273 + 12036.4607348882 * t);
    result += 5e-11 * std::cos(4.28412873331 + 6275.9623029906 * t);
    return result;
}

constexpr double getEarthL4(double t)
{
    double result = 0.0;
    result -= -1.14084e-06;
    result += 7.717e-08 * std::cos(4.13446589358 + 6283.0758499914 * t);
    result += 7.65e-09 * std::cos(3.83803776214 + 12566.1516999828 * t);
    result += 4.2e-09 * std::cos(0.41925861858 + 155.4203994342 * t);
    result += 4e-10 * std::cos(3.5984758584 + 18849.2275499742 * t);
    result += 4.1e-10 * std::cos(3.14398414077 + 3.523118349 * t);
    result += 3.5e-10 * std::cos(5.00298940826 + 5573.1428014331 * t);
    result += 1.3e-10 * std::cos(0.48794833701 + 77713.7714681205 * t);
    result += 1e-10 * std::cos(5.6480176635 + 6127.6554505572 * t);
    result += 8e-11 * std::cos(2.84160570605 + 161000.685737674 * t);
    result += 2e-11 * std::cos(0.54912904658 + 6438.4962494256 * t);
    return result;
}

constexpr double getEarthL5(double t)
{
    double result = 0.0;
    result -= -8.78e-09;
    result += 1.72e-09 * std::cos(2.7657906951 + 6283.0758499914 * t);
    result += 5e-10 * std::cos(2.01353298182 + 155.4203994342 * t);
    result += 2.8e-10 * std::cos(2.21496423926 + 12566.1516999828 * t);
    result += 5e-11 * std::cos(1.75600058765 + 18849.2275499742 * t);
    return result;
}

constexpr double getEarthB0(double t)
{
    double result = 0.0;
    result += 2.7962e-06 * std::cos(3.19870156017 + 84334.6615813083 * t);
    result += 1.01643e-06 * std::cos(5.42248619256 + 5507.5532386674 * t);
    result += 8.0445e-07 * std::cos(3.88013204458 + 5223.6939198022 * t);
    result += 4.3806e-07 * std::cos(3.70444689758 + 2352.8661537718 * t);
    result += 3.1933e-07 * std::cos(4.00026369781 + 1577.3435424478 * t);
    result += 2.2724e-07 * std::cos(3.9847383156 + 1047.7473117547 * t);
    result += 1.6392e-07 * std::cos(3.56456119782 + 5856.4776591154 * t);
    result += 1.8141e-07 * std::cos(4.98367470263 + 6283.0758499914 * t);
    result += 1.4443e-07 * std::cos(3.70275614914 + 9437.762934887 * t);
    result += 1.4304e-07 * std::cos(3.41117857525 + 10213.285546211 * t);
    result += 1.1246e-07 * std::cos(4.8282069053 + 14143.4952424306 * t);
    result += 1.09e-07 * std::cos(2.08574562327 + 6812.766815086 * t);
    result += 9.714e-08 * std::cos(3.47303947752 + 4694.0029547076 * t);
    result += 1.0367e-07 * std::cos(4.05663927946 + 71092.8813549327 * t);
    result += 8.775e-08 * std::cos(4.44016515669 + 5753.3848848968 * t);
    result += 8.366e-08 * std::cos(4.9925151218 + 7084.8967811152 * t);
    result += 6.921e-08 * std::cos(4.32559054073 + 6275.9623029906 * t);
    result += 9.145e-08 * std::cos(1.14182646613 + 6620.8901131878 * t);
    result += 7.194e-08 * std::cos(3.60193205752 + 529.6909650946 * t);
    result += 7.698e-08 * std::cos(5.55425745881 + 167621.575850862 * t);
    result += 5.285e-08 * std::cos(2.48446991566 + 4705.7323075436 * t);
    result += 5.208e-08 * std::cos(6.24992674537 + 18073.7049386502 * t);
    result += 4.529e-08 * std::cos(2.33827747356 + 6309.3741697912 * t);
    result += 5.579e-08 * std::cos(4.41023653738 + 7860.4193924392 * t);
    result += 4.743e-08 * std::cos(0.70995680136 + 5884.9268465832 * t);
    result += 4.301e-08 * std::cos(1.10255777773 + 6681.2248533996 * t);
    result += 3.849e-08 * std::cos(1.82229412531 + 5486.777843175 * t);
    result += 4.093e-08 * std::cos(5.11700141207 + 13367.9726311066 * t);
    result += 3.681e-08 * std::cos(0.43793170356 + 3154.6870848956 * t);
    result += 3.42e-08 * std::cos(5.42034800952 + 6069.7767545534 * t);
    result += 3.617e-08 * std::cos(6.04641937526 + 3930.2096962196 * t);
    result += 3.67e-08 * std::cos(4.58210192227 + 12194.0329146209 * t);
    result += 2.918e-08 * std::cos(1.95463881126 + 10977.078804699 * t);
    result += 2.797e-08 * std::cos(5.61259275048 + 11790.6290886588 * t);
    result += 2.502e-08 * std::cos(0.60499729367 + 6496.3749454294 * t);
    result += 2.319e-08 * std::cos(5.01648216014 + 1059.3819301892 * t);
    result += 2.684e-08 * std::cos(1.39470396488 + 22003.9146348698 * t);
    result += 2.428e-08 * std::cos(3.24183056052 + 78051.5857313169 * t);
    result += 2.12e-08 * std::cos(4.30691000285 + 5643.1785636774 * t);
    result += 2.257e-08 * std::cos(3.15557225618 + 90617.7374312997 * t);
    result += 1.813e-08 * std::cos(3.75574218285 + 3340.6124266998 * t);
    result += 2.226e-08 * std::cos(2.79699346659 + 12036.4607348882 * t);
    result += 1.888e-08 * std::cos(0.86991545823 + 8635.9420037632 * t);
    result += 1.517e-08 * std::cos(1.95852055701 + 398.1490034082 * t);
    result += 1.581e-08 * std::cos(3.19976230948 + 5088.6288397668 * t);
    result += 1.421e-08 * std::cos(6.25530883827 + 2544.3144198834 * t);
    result += 1.595e-08 * std::cos(0.25619915135 + 17298.1823273262 * t);
    result += 1.391e-08 * std::cos(4.69964175561 + 7058.5984613154 * t);
    result += 1.478e-08 * std::cos(2.81808207569 + 25934.1243310894 * t);
    result += 1.481e-08 * std::cos(3.65823554806 + 11506.7697697936 * t);
    result += 1.693e-08 * std::cos(4.95689385293 + 156475.290247996 * t);
    result += 1.183e-08 * std::cos(1.29343061246 + 775.522611324 * t);
    result += 1.114e-08 * std::cos(2.37889311846 + 3738.761430108 * t);
    result += 9.94e-09 * std::cos(4.30088900425 + 9225.539273283 * t);
    result += 9.24e-09 * std::cos(3.06451026812 + 4164.311989613 * t);
    result += 8.67e-09 * std::cos(0.55606931068 + 8429.2412664666 * t);
    result += 9.88e-09 * std::cos(5.97286104208 + 7079.3738568078 * t);
    result += 8.24e-09 * std::cos(1.50984806173 + 10447.3878396044 * t);
    result += 9.15e-09 * std::cos(0.12635654592 + 11015.1064773348 * t);
    result += 7.42e-09 * std::cos(1.99159139281 + 26087.9031415742 * t);
    result -= -1.039e-08;
    result += 8.5e-09 * std::cos(4.24120016095 + 29864.334027309 * t);
    result += 7.55e-09 * std::cos(2.8963187332 + 4732.0306273434 * t);
    result += 7.14e-09 * std::cos(1.37548118603 + 2146.1654164752 * t);
    result += 7.08e-09 * std::cos(1.91406542362 + 8031.0922630584 * t);
    result += 7.46e-09 * std::cos(0.57893808616 + 796.2980068164 * t);
    result += 8.02e-09 * std::cos(5.1233913723 + 2942.4634232916 * t);
    result += 7.51e-09 * std::cos(1.67479850166 + 21228.3920235458 * t);
    result += 6.02e-09 * std::cos(4.09976538826 + 64809.8055049413 * t);
    result += 5.94e-09 * std::cos(3.49580704962 + 16496.3613962024 * t);
    result += 5.92e-09 * std::cos(4.59481504319 + 4690.4798363586 * t);
    result += 5.3e-09 * std::cos(5.739792952 + 8827.3902698748 * t);
    result += 5.03e-09 * std::cos(5.66433137112 + 33794.5437235286 * t);
    result += 4.83e-09 * std::cos(1.57106522411 + 801.8209311238 * t);
    result += 4.38e-09 * std::cos(0.06707733767 + 3128.3887650958 * t);
    result += 4.23e-09 * std::cos(2.86944595927 + 12566.1516999828 * t);
    result += 5.04e-09 * std::cos(3.2620766916 + 7632.9432596502 * t);
    result += 5.52e-09 * std::cos(1.02926440457 + 239762.204517549 * t);
    result += 4.27e-09 * std::cos(3.6743437821 + 213.299095438 * t);
    result += 4.04e-09 * std::cos(1.46193297142 + 15720.8387848784 * t);
    result += 5.03e-09 * std::cos(4.85802444134 + 6290.1893969922 * t);
    result += 4.17e-09 * std::cos(0.81920713533 + 5216.5803728014 * t);
    result += 3.65e-09 * std::cos(0.01002966162 + 12168.0026965746 * t);
    result += 3.63e-09 * std::cos(1.28376436579 + 6206.8097787158 * t);
    result += 3.53e-09 * std::cos(4.7005913311 + 7234.794256242 * t);
    result += 4.15e-09 * std::cos(0.96862624175 + 4136.9104335162 * t);
    result += 3.87e-09 * std::cos(3.09145061418 + 25158.6017197654 * t);
    result += 3.73e-09 * std::cos(2.65119262792 + 7342.4577801806 * t);
    result += 3.61e-09 * std::cos(2.97762937739 + 9623.6882766912 * t);
    result += 4.18e-09 * std::cos(3.75759994446 + 5230.807466803 * t);
    result += 3.96e-09 * std::cos(1.22507712354 + 6438.4962494256 * t);
    result += 3.22e-09 * std::cos(1.21162178805 + 8662.240323563 * t);
    result += 2.84e-09 * std::cos(5.64170320068 + 1589.0728952838 * t);
    result += 3.79e-09 * std::cos(1.72248432748 + 14945.3161735544 * t);
    result += 3.2e-09 * std::cos(3.94161159962 + 7330.8231617461 * t);
    result += 3.13e-09 * std::cos(5.47602376446 + 1194.4470102246 * t);
    result += 2.92e-09 * std::cos(1.38971327603 + 11769.8536931664 * t);
    result += 3.05e-09 * std::cos(0.80429352049 + 37724.7534197482 * t);
    result += 2.57e-09 * std::cos(5.81382809757 + 426.598190876 * t);
    result += 2.65e-09 * std::cos(6.10358507671 + 6836.6452528338 * t);
    result += 2.5e-09 * std::cos(4.56452895547 + 7477.522860216 * t);
    result += 2.66e-09 * std::cos(2.62926282354 + 7238.6755916 * t);
    result += 2.63e-09 * std::cos(6.22089501237 + 6133.5126528568 * t);
    result += 3.06e-09 * std::cos(2.79682380531 + 1748.016413067 * t);
    result += 2.36e-09 * std::cos(2.46093023714 + 11371.7046897582 * t);
    result += 3.16e-09 * std::cos(1.62662805006 + 250908.490120415 * t);
    result += 2.16e-09 * std::cos(3.68721275185 + 5849.3641121146 * t);
    result += 2.3e-09 * std::cos(0.36165162947 + 5863.5912061162 * t);
    result += 2.33e-09 * std::cos(5.03509933858 + 20426.571092422 * t);
    result += 2e-09 * std::cos(5.86073159059 + 4535.0594369244 * t);
    result += 2.77e-09 * std::cos(4.65400292395 + 82239.1669577989 * t);
    result += 2.09e-09 * std::cos(3.72323200804 + 10973.55568635 * t);
    result += 1.99e-09 * std::cos(5.05186622555 + 5429.8794682394 * t);
    result += 2.56e-09 * std::cos(2.4092327977 + 19651.048481098 * t);
    result += 2.1e-09 * std::cos(4.50691909144 + 29088.811415985 * t);
    result += 1.81e-09 * std::cos(6.00294783127 + 4292.3308329504 * t);
    result += 2.49e-09 * std::cos(0.12900984422 + 154379.795624486 * t);
    result += 2.09e-09 * std::cos(3.87759458598 + 17789.845619785 * t);
    result += 2.25e-09 * std::cos(3.18339652605 + 18875.525869774 * t);
    result += 1.91e-09 * std::cos(4.53897489299 + 18477.1087646123 * t);
    result += 1.72e-09 * std::cos(2.09694183014 + 13095.8426650774 * t);
    result += 1.82e-09 * std::cos(3.161079435 + 16730.4636895958 * t);
    result += 1.88e-09 * std::cos(2.22746128596 + 41654.9631159678 * t);
    result += 1.64e-09 * std::cos(5.18686275017 + 5481.2549188676 * t);
    result += 1.6e-09 * std::cos(2.49298855159 + 12592.4500197826 * t);
    result += 1.55e-09 * std::cos(1.5959543823 + 10021.8372800994 * t);
    result += 1.35e-09 * std::cos(0.21349051064 + 10988.808157535 * t);
    result += 1.78e-09 * std::cos(3.8037517797 + 23581.2581773176 * t);
    result += 1.23e-09 * std::cos(1.66800739151 + 15110.4661198662 * t);
    result += 1.22e-09 * std::cos(2.72678272244 + 18849.2275499742 * t);
    result += 1.26e-09 * std::cos(1.1767551291 + 14919.0178537546 * t);
    result += 1.42e-09 * std::cos(3.95053441332 + 337.8142631964 * t);
    result += 1.16e-09 * std::cos(6.06340906229 + 6709.6740408674 * t);
    result += 1.37e-09 * std::cos(3.52143246757 + 12139.5535091068 * t);
    result += 1.36e-09 * std::cos(2.92179113542 + 32217.2001810808 * t);
    result += 1.1e-09 * std::cos(3.51203379263 + 18052.9295431578 * t);
    result += 1.47e-09 * std::cos(4.63371971408 + 22805.7355659936 * t);
    result += 1.08e-09 * std::cos(5.45280814878 + 7.1135470008 * t);
    result += 1.48e-09 * std::cos(0.65447253687 + 95480.9471841745 * t);
    result += 1.19e-09 * std::cos(5.92110458985 + 33019.0211122046 * t);
    result += 1.1e-09 * std::cos(5.34824206306 + 639.897286314 * t);
    result += 1.06e-09 * std::cos(3.71081682629 + 14314.1681130498 * t);
    result += 1.39e-09 * std::cos(6.17607198418 + 24356.7807886416 * t);
    result += 1.18e-09 * std::cos(5.5973871267 + 161338.50000087 * t);
    result += 1.17e-09 * std::cos(3.6506527164 + 45585.1728121874 * t);
    result += 1.27e-09 * std::cos(4.74596574209 + 49515.382508407 * t);
    result += 1.2e-09 * std::cos(1.04211499785 + 6915.8595893046 * t);
    result += 1.2e-09 * std::cos(5.60638811846 + 5650.2921106782 * t);
    result += 1.15e-09 * std::cos(3.10668213289 + 14712.317116458 * t);
    result += 9.9e-10 * std::cos(0.69018940049 + 12779.4507954208 * t);
    result += 9.7e-10 * std::cos(1.07908724794 + 9917.6968745098 * t);
    result += 9.3e-10 * std::cos(2.62295197319 + 17260.1546546904 * t);
    result += 9.9e-10 * std::cos(4.45774681732 + 4933.2084403326 * t);
    result += 1.23e-09 * std::cos(1.37488922089 + 28286.9904848612 * t);
    result += 1.21e-09 * std::cos(5.19767249813 + 27511.4678735372 * t);
    result += 1.05e-09 * std::cos(0.87192267806 + 77375.9572049241 * t);
    result += 8.7e-10 * std::cos(3.9363781295 + 17654.7805397496 * t);
    result += 1.22e-09 * std::cos(2.2395606868 + 83997.0911355954 * t);
    result += 8.7e-10 * std::cos(4.18201600952 + 22779.4372461938 * t);
    result += 1.04e-09 * std::cos(4.59580877295 + 1349.8674096588 * t);
    result += 1.02e-09 * std::cos(2.83545248411 + 12352.8526045448 * t);
    result += 1.02e-09 * std::cos(3.97386522171 + 10818.1352869158 * t);
    result += 1.01e-09 * std::cos(4.32892825857 + 36147.4098773004 * t);
    result += 9.4e-10 * std::cos(5.00001709261 + 150192.214398004 * t);
    result += 7.7e-10 * std::cos(3.97199369296 + 1592.5960136328 * t);
    result += 1e-09 * std::cos(6.07733097102 + 26735.9452622132 * t);
    result += 8.6e-10 * std::cos(5.2602963825 + 28313.288804661 * t);
    result += 9.3e-10 * std::cos(4.31900620254 + 44809.6502008634 * t);
    result += 7.6e-10 * std::cos(6.22743405935 + 13521.7514415914 * t);
    result += 7.2e-10 * std::cos(1.55820597747 + 6256.7775301916 * t);
    result += 8.2e-10 * std::cos(4.95202664555 + 10575.4066829418 * t);
    result += 8.2e-10 * std::cos(1.69647647075 + 1990.745017041 * t);
    result += 7.5e-10 * std::cos(2.29836095644 + 3634.6210245184 * t);
    result += 7.5e-10 * std::cos(2.66367876557 + 16200.7727245012 * t);
    result += 8.7e-10 * std::cos(0.26630214764 + 31441.6775697568 * t);
    result += 7.7e-10 * std::cos(2.25530954137 + 5235.3285382367 * t);
    result += 7.6e-10 * std::cos(1.09869730846 + 12903.9659631792 * t);
    result += 5.8e-10 * std::cos(4.28246138307 + 12559.038152982 * t);
    result += 6.4e-10 * std::cos(5.51112830114 + 173904.651700853 * t);
    result += 5.6e-10 * std::cos(2.60133794851 + 73188.3759784421 * t);
    result += 5.5e-10 * std::cos(5.81483150022 + 143233.51002162 * t);
    result += 5.4e-10 * std::cos(3.38482031504 + 323049.118787103 * t);
    result += 3.9e-10 * std::cos(3.28500401343 + 71768.5098813255 * t);
    result += 3.9e-10 * std::cos(3.1123991069 + 96900.8132812911 * t);
    return result;
}

constexpr double getEarthB1(double t)
{
    double result = 0.0;
    result += 9.03e-08 * std::cos(3.8972906189 + 5507.5532386674 * t);
    result += 6.177e-08 * std::cos(1.73038850355 + 5223.6939198022 * t);
    result += 3.8e-08 * std::cos(5.24404145734 + 2352.8661537718 * t);
    result += 2.834e-08 * std::cos(2.4734503745 + 1577.3435424478 * t);
    result += 1.817e-08 * std::cos(0.41874743765 + 6283.0758499914 * t);
    result += 1.499e-08 * std::cos(1.83320979291 + 5856.4776591154 * t);
    result += 1.466e-08 * std::cos(5.69401926017 + 5753.3848848968 * t);
    result += 1.301e-08 * std::cos(2.18890066314 + 9437.762934887 * t);
    result += 1.233e-08 * std::cos(4.95222451476 + 10213.285546211 * t);
    result += 1.021e-08 * std::cos(0.12866660208 + 7860.4193924392 * t);
    result += 9.82e-09 * std::cos(0.09005453285 + 14143.4952424306 * t);
    result += 8.65e-09 * std::cos(1.73949953555 + 3930.2096962196 * t);
    result += 5.81e-09 * std::cos(2.26949174067 + 5884.9268465832 * t);
    result += 5.24e-09 * std::cos(5.65662503159 + 529.6909650946 * t);
    result += 4.73e-09 * std::cos(6.22750969242 + 6309.3741697912 * t);
    result += 4.51e-09 * std::cos(1.53288619213 + 18073.7049386502 * t);
    result += 3.64e-09 * std::cos(3.61614477374 + 13367.9726311066 * t);
    result += 3.72e-09 * std::cos(3.2247072132 + 6275.9623029906 * t);
    result += 2.68e-09 * std::cos(2.34341267879 + 11790.6290886588 * t);
    result += 3.22e-09 * std::cos(0.94084045832 + 6069.7767545534 * t);
    result += 2.32e-09 * std::cos(0.26781182579 + 7058.5984613154 * t);
    result += 2.16e-09 * std::cos(6.05952221329 + 10977.078804699 * t);
    result += 2.32e-09 * std::cos(2.93325646109 + 22003.9146348698 * t);
    result += 2.04e-09 * std::cos(3.86264841382 + 6496.3749454294 * t);
    result += 2.02e-09 * std::cos(2.81892511133 + 15720.8387848784 * t);
    result += 1.85e-09 * std::cos(4.93512381859 + 12036.4607348882 * t);
    result += 2.2e-09 * std::cos(3.99305643742 + 6812.766815086 * t);
    result += 1.66e-09 * std::cos(1.74970002999 + 11506.7697697936 * t);
    result += 2.12e-09 * std::cos(1.57166285369 + 4694.0029547076 * t);
    result += 1.57e-09 * std::cos(1.08259734788 + 5643.1785636774 * t);
    result += 1.54e-09 * std::cos(5.99434678412 + 5486.777843175 * t);
    result += 1.44e-09 * std::cos(5.23285656085 + 78051.5857313169 * t);
    result += 1.44e-09 * std::cos(1.16454655948 + 90617.7374312997 * t);
    result += 1.37e-09 * std::cos(2.67760436027 + 6290.1893969922 * t);
    result += 1.8e-09 * std::cos(2.06509026215 + 7084.8967811152 * t);
    result += 1.21e-09 * std::cos(5.90212574947 + 9225.539273283 * t);
    result += 1.5e-09 * std::cos(2.00175038718 + 5230.807466803 * t);
    result += 1.49e-09 * std::cos(5.06157254516 + 17298.1823273262 * t);
    result += 1.18e-09 * std::cos(5.39979058038 + 3340.6124266998 * t);
    result += 1.61e-09 * std::cos(3.32421999691 + 6283.3196674749 * t);
    result += 1.21e-09 * std::cos(4.36722193162 + 19651.048481098 * t);
    result += 1.16e-09 * std::cos(5.83462858507 + 4705.7323075436 * t);
    result += 1.28e-09 * std::cos(4.35489873365 + 25934.1243310894 * t);
    result += 1.43e-09;
    result += 1.09e-09 * std::cos(2.52157834166 + 6438.4962494256 * t);
    result += 9.9e-10 * std::cos(2.70727488041 + 5216.5803728014 * t);
    result += 1.03e-09 * std::cos(0.93782340879 + 8827.3902698748 * t);
    result += 8.2e-10 * std::cos(4.2921468039 + 8635.9420037632 * t);
    result += 7.9e-10 * std::cos(2.24085737326 + 1059.3819301892 * t);
    result += 9.7e-10 * std::cos(5.50959692365 + 29864.334027309 * t);
    result += 7.2e-10 * std::cos(0.21891639822 + 21228.3920235458 * t);
    result += 7.1e-10 * std::cos(2.86755026812 + 6681.2248533996 * t);
    result += 7.4e-10 * std::cos(2.20184828895 + 37724.7534197482 * t);
    result += 6.3e-10 * std::cos(4.45586625948 + 7079.3738568078 * t);
    result += 6.1e-10 * std::cos(0.63918772258 + 33794.5437235286 * t);
    result += 4.7e-10 * std::cos(2.09070235724 + 3128.3887650958 * t);
    result += 4.7e-10 * std::cos(3.325438433 + 26087.9031415742 * t);
    result += 4.9e-10 * std::cos(1.60680905005 + 6702.5604938666 * t);
    result += 5.7e-10 * std::cos(0.11215813438 + 29088.811415985 * t);
    result += 5.6e-10 * std::cos(5.47982934911 + 775.522611324 * t);
    result += 5e-10 * std::cos(1.89396788463 + 12139.5535091068 * t);
    result += 4.7e-10 * std::cos(2.9721490724 + 20426.571092422 * t);
    result += 4.1e-10 * std::cos(5.5532939489 + 11015.1064773348 * t);
    result += 4.1e-10 * std::cos(5.91861144924 + 23581.2581773176 * t);
    result += 4.5e-10 * std::cos(4.95273290181 + 5863.5912061162 * t);
    result += 5e-10 * std::cos(3.62740835096 + 41654.9631159678 * t);
    result += 3.7e-10 * std::cos(6.09033460601 + 64809.8055049413 * t);
    result += 3.7e-10 * std::cos(5.86153655431 + 12566.1516999828 * t);
    result += 4.6e-10 * std::cos(1.65798680284 + 25158.6017197654 * t);
    result += 3.8e-10 * std::cos(2.00673650251 + 426.598190876 * t);
    result += 3.6e-10 * std::cos(6.24373396652 + 6283.14316029419 * t);
    result += 3.6e-10 * std::cos(0.40465162918 + 6283.0085396886 * t);
    result += 3.2e-10 * std::cos(6.03707103538 + 2942.4634232916 * t);
    result += 4.1e-10 * std::cos(4.86809570283 + 1592.5960136328 * t);
    result += 2.8e-10 * std::cos(4.38359423735 + 7632.9432596502 * t);
    result += 2.8e-10 * std::cos(6.03334294232 + 17789.845619785 * t);
    result += 2.6e-10 * std::cos(3.88971333608 + 5331.3574437408 * t);
    result += 2.6e-10 * std::cos(5.94932724051 + 16496.3613962024 * t);
    result += 3.1e-10 * std::cos(1.44666331503 + 16730.4636895958 * t);
    result += 2.6e-10 * std::cos(6.26376705837 + 23543.2305046818 * t);
    result += 3.3e-10 * std::cos(0.93797239147 + 213.299095438 * t);
    result += 2.6e-10 * std::cos(3.71858432944 + 13095.8426650774 * t);
    result += 2.7e-10 * std::cos(0.60565274405 + 10988.808157535 * t);
    result += 2.3e-10 * std::cos(4.4438898555 + 18849.2275499742 * t);
    result += 2.8e-10 * std::cos(1.53862289477 + 6279.4854213396 * t);
    result += 2.8e-10 * std::cos(1.96831814872 + 6286.6662786432 * t);
    result += 2.8e-10 * std::cos(5.78094918529 + 15110.4661198662 * t);
    result += 2.6e-10 * std::cos(2.48165809843 + 5729.506447149 * t);
    result += 2e-10 * std::cos(3.85655029499 + 9623.6882766912 * t);
    result += 2.1e-10 * std::cos(5.83006047147 + 7234.794256242 * t);
    result += 2.1e-10 * std::cos(0.69628570421 + 398.1490034082 * t);
    result += 2.2e-10 * std::cos(5.02222806555 + 6127.6554505572 * t);
    result += 2e-10 * std::cos(3.4761126529 + 6148.010769956 * t);
    result += 2e-10 * std::cos(0.90769829044 + 5481.2549188676 * t);
    result += 2e-10 * std::cos(0.03081589303 + 6418.1409300268 * t);
    result += 2e-10 * std::cos(3.74220084927 + 1589.0728952838 * t);
    result += 2.1e-10 * std::cos(4.00149269576 + 3154.6870848956 * t);
    result += 1.8e-10 * std::cos(1.58348238359 + 2118.7638603784 * t);
    result += 1.9e-10 * std::cos(0.85407021371 + 14712.317116458 * t);
    return result;
}

constexpr double getEarthB2(double t)
{
    double result = 0.0;
    result += 1.662e-08 * std::cos(1.62703209173 + 84334.6615813083 * t);
    result += 4.92e-09 * std::cos(2.41382223971 + 1047.7473117547 * t);
    result += 3.44e-09 * std::cos(2.24353004539 + 5507.5532386674 * t);
    result += 2.58e-09 * std::cos(6.00906896311 + 5223.6939198022 * t);
    result += 1.31e-09 * std::cos(0.9544734524 + 6283.0758499914 * t);
    result += 8.6e-10 * std::cos(1.67530247303 + 7860.4193924392 * t);
    result += 9e-10 * std::cos(0.97606804452 + 1577.3435424478 * t);
    result += 9e-10 * std::cos(0.37899871725 + 2352.8661537718 * t);
    result += 8.9e-10 * std::cos(6.25807507963 + 10213.285546211 * t);
    result += 7.5e-10 * std::cos(0.84213523741 + 167621.575850862 * t);
    result += 5.2e-10 * std::cos(1.70501566089 + 14143.4952424306 * t);
    result += 5.7e-10 * std::cos(6.15295833679 + 12194.0329146209 * t);
    result += 5.1e-10 * std::cos(1.2761601674 + 5753.3848848968 * t);
    result += 5.1e-10 * std::cos(5.37229738682 + 6812.766815086 * t);
    result += 3.4e-10 * std::cos(1.73672994279 + 7058.5984613154 * t);
    result += 3.8e-10 * std::cos(2.77761031485 + 10988.808157535 * t);
    result += 4.6e-10 * std::cos(3.38617099014 + 156475.290247996 * t);
    result += 2.1e-10 * std::cos(1.95248349228 + 8827.3902698748 * t);
    result += 1.8e-10 * std::cos(3.33419222028 + 8429.2412664666 * t);
    result += 1.9e-10 * std::cos(4.32945160287 + 17789.845619785 * t);
    result += 1.7e-10 * std::cos(0.66191210656 + 6283.0085396886 * t);
    result += 1.8e-10 * std::cos(3.74885333072 + 11769.8536931664 * t);
    result += 1.7e-10 * std::cos(4.23058370776 + 10977.078804699 * t);
    result += 1.7e-10 * std::cos(1.78116162721 + 5486.777843175 * t);
    result += 2.1e-10 * std::cos(1.36972913918 + 12036.4607348882 * t);
    result += 1.7e-10 * std::cos(2.79601092529 + 796.2980068164 * t);
    result += 1.5e-10 * std::cos(0.4308784885 + 11790.6290886588 * t);
    result += 1.7e-10 * std::cos(1.35132152761 + 78051.5857313169 * t);
    result += 1.5e-10 * std::cos(1.17032155085 + 213.299095438 * t);
    result += 1.8e-10 * std::cos(2.85221514199 + 5088.6288397668 * t);
    result += 1.7e-10 * std::cos(0.21780913672 + 6283.14316029419 * t);
    result += 1.3e-10 * std::cos(1.21201504386 + 25132.3033999656 * t);
    result += 1.2e-10 * std::cos(1.12953712197 + 90617.7374312997 * t);
    result += 1.2e-10 * std::cos(5.13714452592 + 7079.3738568078 * t);
    result += 1.3e-10 * std::cos(3.79842135217 + 4933.2084403326 * t);
    result += 1.2e-10 * std::cos(4.89407978213 + 3738.761430108 * t);
    result += 1.5e-10 * std::cos(6.05682328852 + 398.1490034082 * t);
    result += 1.4e-10 * std::cos(4.81029291856 + 4694.0029547076 * t);
    result += 1.1e-10 * std::cos(0.61684523405 + 3128.3887650958 * t);
    result += 1.1e-10 * std::cos(5.328765385 + 6040.3472460174 * t);
    result += 1.4e-10 * std::cos(5.27227350286 + 4535.0594369244 * t);
    result += 1.1e-10 * std::cos(2.39292099451 + 5331.3574437408 * t);
    result += 1e-10 * std::cos(4.4529653271 + 6525.8044539654 * t);
    result += 1.4e-10 * std::cos(4.66400985037 + 8031.0922630584 * t);
    result += 1e-10 * std::cos(3.22472385926 + 9437.762934887 * t);
    result += 1.1e-10 * std::cos(3.80913404437 + 801.8209311238 * t);
    result += 1e-10 * std::cos(5.15032130575 + 11371.7046897582 * t);
    result += 1.3e-10 * std::cos(0.98720797401 + 5729.506447149 * t);
    result += 9e-11 * std::cos(5.94191743597 + 7632.9432596502 * t);
    return result;
}

constexpr double getEarthB3(double t)
{
    double result = 0.0;
    result += 1.1e-10 * std::cos(0.23877262399 + 7860.4193924392 * t);
    result += 9e-11 * std::cos(1.16069982609 + 5507.5532386674 * t);
    result += 8e-11 * std::cos(1.65357552925 + 5884.9268465832 * t);
    result += 8e-11 * std::cos(2.86720038197 + 7058.5984613154 * t);
    result += 7e-11 * std::cos(3.04818741666 + 5486.777843175 * t);
    result += 7e-11 * std::cos(2.59437103785 + 529.6909650946 * t);
    result += 8e-11 * std::cos(4.02863090524 + 6256.7775301916 * t);
    result += 8e-11 * std::cos(2.42003508927 + 5753.3848848968 * t);
    result += 6e-11 * std::cos(0.84181087594 + 6275.9623029906 * t);
    result += 6e-11 * std::cos(5.40160929468 + 1577.3435424478 * t);
    result += 7e-11 * std::cos(2.73399865247 + 6309.3741697912 * t);
    return result;
}

constexpr double getEarthB4(double t)
{
    double result = 0.0;
    result += 4e-11 * std::cos(0.79662198849 + 6438.4962494256 * t);
    result += 5e-11 * std::cos(0.84308705203 + 1047.7473117547 * t);
    result += 5e-11 * std::cos(0.05711572303 + 84334.6615813083 * t);
    result += 3e-11 * std::cos(3.46779895686 + 6279.5527316424 * t);
    result += 3e-11 * std::cos(2.89822201212 + 6127.6554505572 * t);
    return result;
}

constexpr double getEarthR0(double t)
{
    double result = 0.0;
    result += 1.00013988799;
    result += 0.01670699626 * std::cos(3.09846350771 + 6283.0758499914 * t);
    result += 0.00013956023 * std::cos(3.0552460962 + 12566.1516999828 * t);
    result += 3.08372e-05 * std::cos(5.19846674381 + 77713.7714681205 * t);
    result += 1.628461e-05 * std::cos(1.17387749012 + 5753.3848848968 * t);
    result += 1.575568e-05 * std::cos(2.84685245825 + 7860.4193924392 * t);
    result += 9.24799e-06 * std::cos(5.45292234084 + 11506.7697697936 * t);
    result += 5.42444e-06 * std::cos(4.56409149777 + 3930.2096962196 * t);
    result += 4.7211e-06 * std::cos(3.66100022149 + 5884.9268465832 * t);
    result += 3.2878e-06 * std::cos(5.89983646482 + 5223.6939198022 * t);
    result += 3.45983e-06 * std::cos(0.96368617687 + 5507.5532386674 * t);
    result += 3.06784e-06 * std::cos(0.29867139512 + 5573.1428014331 * t);
    result += 1.74844e-06 * std::cos(3.01193636534 + 18849.2275499742 * t);
    result += 2.43189e-06 * std::cos(4.27349536153 + 11790.6290886588 * t);
    result += 2.11829e-06 * std::cos(5.84714540314 + 1577.3435424478 * t);
    result += 1.85752e-06 * std::cos(5.02194447178 + 10977.078804699 * t);
    result += 1.09835e-06 * std::cos(5.05510636285 + 5486.777843175 * t);
    result += 9.8316e-07 * std::cos(0.88681311277 + 6069.7767545534 * t);
    result += 8.6499e-07 * std::cos(5.68959778254 + 15720.8387848784 * t);
    result += 8.5825e-07 * std::cos(1.27083733351 + 161000.685737674 * t);
    result += 6.2916e-07 * std::cos(0.92177108832 + 529.6909650946 * t);
    result += 5.7056e-07 * std::cos(2.01374292014 + 83996.8473181119 * t);
    result += 6.4903e-07 * std::cos(0.27250613787 + 17260.1546546904 * t);
    result += 4.9384e-07 * std::cos(3.24501240359 + 2544.3144198834 * t);
    result += 5.5736e-07 * std::cos(5.24159798933 + 71430.6956181291 * t);
    result += 4.2515e-07 * std::cos(6.01110242003 + 6275.9623029906 * t);
    result += 4.6963e-07 * std::cos(2.57805070386 + 775.522611324 * t);
    result += 3.8968e-07 * std::cos(5.36071738169 + 4694.0029547076 * t);
    result += 4.4661e-07 * std::cos(5.53715807302 + 9437.762934887 * t);
    result += 3.566e-07 * std::cos(1.67468058995 + 12036.4607348882 * t);
    result += 3.1921e-07 * std::cos(0.18368229781 + 5088.6288397668 * t);
    result += 3.1846e-07 * std::cos(1.77775642085 + 398.1490034082 * t);
    result += 3.3193e-07 * std::cos(0.24370300098 + 7084.8967811152 * t);
    result += 3.8245e-07 * std::cos(2.39255343974 + 8827.3902698748 * t);
    result += 2.8464e-07 * std::cos(1.21344868176 + 6286.5989683404 * t);
    result += 3.749e-07 * std::cos(0.82952922332 + 19651.048481098 * t);
    result += 3.6957e-07 * std::cos(4.90107591914 + 12139.5535091068 * t);
    result += 3.4537e-07 * std::cos(1.84270693282 + 2942.4634232916 * t);
    result += 2.6275e-07 * std::cos(4.58896850401 + 10447.3878396044 * t);
    result += 2.4596e-07 * std::cos(3.78660875483 + 8429.2412664666 * t);
    result += 2.3587e-07 * std::cos(0.26866117066 + 796.2980068164 * t);
    result += 2.7793e-07 * std::cos(1.89934330904 + 6279.5527316424 * t);
    result += 2.3927e-07 * std::cos(4.99598548138 + 5856.4776591154 * t);
    result += 2.0349e-07 * std::cos(4.65267995431 + 2146.1654164752 * t);
    result += 2.3287e-07 * std::cos(2.80783650928 + 14143.4952424306 * t);
    result += 2.2103e-07 * std::cos(1.95004702988 + 3154.6870848956 * t);
    result += 1.9506e-07 * std::cos(5.38227371393 + 2352.8661537718 * t);
    result += 1.7958e-07 * std::cos(0.19871379385 + 6812.766815086 * t);
    result += 1.7174e-07 * std::cos(4.43315560735 + 10213.285546211 * t);
    result += 1.619e-07 * std::cos(5.23160507859 + 17789.845619785 * t);
    result += 1.7314e-07 * std::cos(6.15200787916 + 16730.4636895958 * t);
    result += 1.3814e-07 * std::cos(5.18962074032 + 8031.0922630584 * t);
    result += 1.8833e-07 * std::cos(0.67306674027 + 149854.400134808 * t);
    result += 1.8331e-07 * std::cos(2.25348733734 + 23581.2581773176 * t);
    result += 1.3641e-07 * std::cos(3.68516118804 + 4705.7323075436 * t);
    result += 1.3139e-07 * std::cos(0.65289581324 + 13367.9726311066 * t);
    result += 1.0414e-07 * std::cos(4.33285688538 + 11769.8536931664 * t);
    result += 9.978e-08 * std::cos(4.20126336355 + 6309.3741697912 * t);
    result += 1.0169e-07 * std::cos(1.59390681369 + 4690.4798363586 * t);
    result += 7.564e-08 * std::cos(2.6256059739 + 6256.7775301916 * t);
    result += 9.661e-08 * std::cos(3.6758679122 + 27511.4678735372 * t);
    result += 6.743e-08 * std::cos(0.56270332741 + 3340.6124266998 * t);
    result += 8.743e-08 * std::cos(6.06359123461 + 1748.016413067 * t);
    result += 7.786e-08 * std::cos(3.67371235637 + 12168.0026965746 * t);
    result += 6.633e-08 * std::cos(5.66149277792 + 11371.7046897582 * t);
    result += 7.712e-08 * std::cos(0.31242577789 + 7632.9432596502 * t);
    result += 6.592e-08 * std::cos(3.13576266188 + 801.8209311238 * t);
    result += 7.46e-08 * std::cos(5.64757188143 + 11926.2544136688 * t);
    result += 6.933e-08 * std::cos(2.923845864 + 6681.2248533996 * t);
    result += 6.802e-08 * std::cos(1.4232980642 + 23013.5395395872 * t);
    result += 6.115e-08 * std::cos(5.13393615454 + 1194.4470102246 * t);
    result += 6.477e-08 * std::cos(2.64986648492 + 19804.8272915828 * t);
    result += 5.233e-08 * std::cos(4.62434053374 + 6438.4962494256 * t);
    result += 6.147e-08 * std::cos(3.02863936662 + 233141.314404362 * t);
    result += 4.608e-08 * std::cos(1.72194702724 + 7234.794256242 * t);
    result += 4.221e-08 * std::cos(1.55697533729 + 7238.6755916 * t);
    result += 5.314e-08 * std::cos(2.40716580847 + 11499.6562227928 * t);
    result += 5.128e-08 * std::cos(5.3239896569 + 11513.8833167944 * t);
    result += 4.77e-08 * std::cos(0.25554312006 + 11856.2186514245 * t);
    result += 5.519e-08 * std::cos(2.09089154502 + 17298.1823273262 * t);
    result += 5.625e-08 * std::cos(4.34052903053 + 90955.5516944961 * t);
    result += 4.578e-08 * std::cos(4.4656964157 + 5746.271337896 * t);
    result += 3.788e-08 * std::cos(4.9072938351 + 4164.311989613 * t);
    result += 5.337e-08 * std::cos(5.09957905104 + 31441.6775697568 * t);
    result += 3.967e-08 * std::cos(1.20054555174 + 1349.8674096588 * t);
    result += 4.008e-08 * std::cos(3.03007204392 + 1059.3819301892 * t);
    result += 3.476e-08 * std::cos(0.7608027703 + 10973.55568635 * t);
    result += 4.232e-08 * std::cos(1.05485713117 + 5760.4984318976 * t);
    result += 4.582e-08 * std::cos(3.76570026763 + 6386.16862421 * t);
    result += 3.335e-08 * std::cos(3.13829943354 + 6836.6452528338 * t);
    result += 3.418e-08 * std::cos(3.00072390334 + 4292.3308329504 * t);
    result += 3.598e-08 * std::cos(5.70718084323 + 5643.1785636774 * t);
    result += 3.237e-08 * std::cos(4.16448773994 + 9917.6968745098 * t);
    result += 4.154e-08 * std::cos(2.59941292162 + 7058.5984613154 * t);
    result += 3.362e-08 * std::cos(4.54577697964 + 4732.0306273434 * t);
    result += 2.978e-08 * std::cos(1.3056126882 + 6283.14316029419 * t);
    result += 2.765e-08 * std::cos(0.51311975679 + 26.2983197998 * t);
    result += 2.802e-08 * std::cos(5.66263240521 + 8635.9420037632 * t);
    result += 2.927e-08 * std::cos(5.73787481548 + 16200.7727245012 * t);
    result += 3.164e-08 * std::cos(1.69140262657 + 11015.1064773348 * t);
    result += 2.598e-08 * std::cos(2.96244118586 + 25132.3033999656 * t);
    result += 3.519e-08 * std::cos(3.62639325753 + 244287.600007228 * t);
    result += 2.676e-08 * std::cos(4.2072570085 + 18073.7049386502 * t);
    result += 2.978e-08 * std::cos(1.74971565805 + 6283.0085396886 * t);
    result += 2.287e-08 * std::cos(1.06975704977 + 14314.1681130498 * t);
    result += 2.863e-08 * std::cos(5.92838131397 + 14712.317116458 * t);
    result += 3.071e-08 * std::cos(0.23793217002 + 35371.8872659764 * t);
    result += 2.656e-08 * std::cos(0.8995930178 + 12352.8526045448 * t);
    result += 2.415e-08 * std::cos(2.79975176257 + 709.9330485583 * t);
    result += 2.814e-08 * std::cos(3.51488206882 + 21228.3920235458 * t);
    result += 1.977e-08 * std::cos(2.6135829755 + 951.7184062506 * t);
    result += 2.548e-08 * std::cos(2.47684686575 + 6208.2942514241 * t);
    result += 1.999e-08 * std::cos(0.5609038816 + 7079.3738568078 * t);
    result += 2.305e-08 * std::cos(1.05376461628 + 22483.8485744926 * t);
    result += 1.855e-08 * std::cos(2.86090681163 + 5216.5803728014 * t);
    result += 2.157e-08 * std::cos(1.31396741861 + 154717.609887683 * t);
    result += 1.97e-08 * std::cos(4.36929875289 + 167283.761587666 * t);
    result += 1.635e-08 * std::cos(5.85571606764 + 10984.1923516998 * t);
    result += 1.754e-08 * std::cos(2.14452408833 + 6290.1893969922 * t);
    result += 2.154e-08 * std::cos(6.03828341543 + 10873.9860304804 * t);
    result += 1.714e-08 * std::cos(3.70157691113 + 1592.5960136328 * t);
    result += 1.541e-08 * std::cos(6.21598380732 + 23543.2305046818 * t);
    result += 1.611e-08 * std::cos(1.99824499377 + 10969.9652576982 * t);
    result += 1.712e-08 * std::cos(1.34295663542 + 3128.3887650958 * t);
    result += 1.642e-08 * std::cos(5.55026665339 + 6496.3749454294 * t);
    result += 1.502e-08 * std::cos(5.43948825854 + 155.4203994342 * t);
    result += 1.827e-08 * std::cos(5.91227480261 + 3738.761430108 * t);
    result += 1.726e-08 * std::cos(2.16764983583 + 10575.4066829418 * t);
    result += 1.532e-08 * std::cos(5.3568310707 + 13521.7514415914 * t);
    result += 1.829e-08 * std::cos(1.66006148731 + 39302.096962196 * t);
    result += 1.605e-08 * std::cos(1.90928637633 + 6133.5126528568 * t);
    result += 1.282e-08 * std::cos(2.46014880418 + 13916.0191096416 * t);
    result += 1.211e-08 * std::cos(4.4136063155 + 3894.1818295422 * t);
    result += 1.394e-08 * std::cos(1.77801929354 + 9225.539273283 * t);
    result += 1.571e-08 * std::cos(4.95512957592 + 25158.6017197654 * t);
    result += 1.205e-08 * std::cos(1.19212540615 + 3.523118349 * t);
    result += 1.132e-08 * std::cos(2.69830084955 + 6040.3472460174 * t);
    result += 1.504e-08 * std::cos(5.77002730341 + 18209.3302636602 * t);
    result += 1.393e-08 * std::cos(1.62621805428 + 5120.6011455836 * t);
    result += 1.077e-08 * std::cos(2.93931554233 + 17256.6315363414 * t);
    result += 1.232e-08 * std::cos(0.71655165307 + 143571.324284816 * t);
    result += 1.087e-08 * std::cos(0.99769687939 + 955.5997416086 * t);
    result += 1.068e-08 * std::cos(5.28472576231 + 65147.6197681377 * t);
    result += 9.8e-09 * std::cos(5.10949204607 + 6172.869528772 * t);
    result += 1.169e-08 * std::cos(3.11664290862 + 14945.3161735544 * t);
    result += 1.202e-08 * std::cos(4.02992510402 + 553.5694028424 * t);
    result += 9.79e-09 * std::cos(2.00000879212 + 15110.4661198662 * t);
    result += 9.62e-09 * std::cos(4.023807714 + 6282.0955289232 * t);
    result += 9.99e-09 * std::cos(3.6264300279 + 6262.300454499 * t);
    result += 1.03e-08 * std::cos(5.84989900289 + 213.299095438 * t);
    result += 1.014e-08 * std::cos(2.84221578218 + 8662.240323563 * t);
    result += 1.185e-08 * std::cos(1.51330541132 + 17654.7805397496 * t);
    result += 9.67e-09 * std::cos(2.67081017562 + 5650.2921106782 * t);
    result += 1.222e-08 * std::cos(2.65423784904 + 88860.0570709867 * t);
    result += 9.81e-09 * std::cos(2.36370360283 + 6206.8097787158 * t);
    result += 1.033e-08 * std::cos(0.13874927606 + 11712.9553182308 * t);
    result += 1.103e-08 * std::cos(3.08477302937 + 43232.3066584156 * t);
    result += 7.81e-09 * std::cos(2.53372735932 + 16496.3613962024 * t);
    result += 1.019e-08 * std::cos(3.04569392376 + 6037.244203762 * t);
    result += 7.95e-09 * std::cos(5.80662989111 + 5230.807466803 * t);
    result += 8.13e-09 * std::cos(3.57710279439 + 10177.2576795336 * t);
    result += 9.62e-09 * std::cos(5.31470594766 + 6284.0561710596 * t);
    result += 7.21e-09 * std::cos(5.96264301567 + 12559.038152982 * t);
    result += 9.66e-09 * std::cos(2.74714939953 + 6244.9428143536 * t);
    result += 9.21e-09 * std::cos(0.10155275926 + 29088.811415985 * t);
    result += 6.92e-09 * std::cos(3.89764447548 + 1589.0728952838 * t);
    result += 7.19e-09 * std::cos(5.91791450402 + 4136.9104335162 * t);
    result += 7.72e-09 * std::cos(4.05505682353 + 6127.6554505572 * t);
    result += 7.12e-09 * std::cos(5.49291532439 + 22003.9146348698 * t);
    result += 6.72e-09 * std::cos(1.60700490811 + 11087.2851259184 * t);
    result += 6.9e-09 * std::cos(4.50539825563 + 426.598190876 * t);
    result += 8.54e-09 * std::cos(3.26104981596 + 20426.571092422 * t);
    result += 6.56e-09 * std::cos(4.3241018294 + 16858.4825329332 * t);
    result += 8.4e-09 * std::cos(2.59572585222 + 28766.924424484 * t);
    result += 6.92e-09 * std::cos(0.61650089011 + 11403.676995575 * t);
    result += 7e-09 * std::cos(3.40901167143 + 7.1135470008 * t);
    result += 7.26e-09 * std::cos(0.04243053594 + 5481.2549188676 * t);
    result += 5.57e-09 * std::cos(4.78317696534 + 20199.094959633 * t);
    result += 6.49e-09 * std::cos(1.04027912958 + 6062.6632075526 * t);
    result += 6.33e-09 * std::cos(5.70229959167 + 45892.730433157 * t);
    result += 5.92e-09 * std::cos(6.11836729658 + 9623.6882766912 * t);
    result += 5.23e-09 * std::cos(3.62840021266 + 5333.9002410216 * t);
    result += 6.04e-09 * std::cos(5.57734696185 + 10344.2950653858 * t);
    result += 4.96e-09 * std::cos(2.21023499449 + 1990.745017041 * t);
    result += 6.91e-09 * std::cos(1.96071732602 + 12416.5885028482 * t);
    result += 6.4e-09 * std::cos(1.59074172032 + 18319.5365848796 * t);
    result += 6.25e-09 * std::cos(3.82362791378 + 13517.8701062334 * t);
    result += 6.63e-09 * std::cos(5.08444996779 + 283.8593188652 * t);
    result += 4.75e-09 * std::cos(1.17025894287 + 12569.6748183318 * t);
    result += 6.64e-09 * std::cos(4.50029469969 + 47162.5163546352 * t);
    result += 5.69e-09 * std::cos(0.16310365162 + 17267.2682016912 * t);
    result += 5.68e-09 * std::cos(3.86100969474 + 6076.8903015542 * t);
    result += 5.39e-09 * std::cos(4.83282276086 + 18422.6293590982 * t);
    result += 4.66e-09 * std::cos(0.75872342878 + 7342.4577801806 * t);
    result += 5.41e-09 * std::cos(3.07212190507 + 226858.23855437 * t);
    result += 4.58e-09 * std::cos(0.26774483096 + 4590.910180489 * t);
    result += 6.1e-09 * std::cos(1.53597051291 + 33019.0211122046 * t);
    result += 6.17e-09 * std::cos(2.62356328726 + 11190.377900137 * t);
    result += 5.48e-09 * std::cos(4.55798855791 + 18875.525869774 * t);
    result += 6.33e-09 * std::cos(4.60110281228 + 66567.4858652543 * t);
    result += 5.96e-09 * std::cos(5.78202396722 + 632.7837393132 * t);
    result += 5.33e-09 * std::cos(5.01786882904 + 12132.439962106 * t);
    result += 6.03e-09 * std::cos(5.38458554802 + 316428.228673915 * t);
    result += 4.69e-09 * std::cos(0.59168241917 + 21954.157609398 * t);
    result += 5.48e-09 * std::cos(3.50613163558 + 17253.0411076896 * t);
    result += 5.02e-09 * std::cos(0.98804327589 + 11609.8625440122 * t);
    result += 5.68e-09 * std::cos(1.98497313089 + 7668.6374249425 * t);
    result += 4.82e-09 * std::cos(1.62141803864 + 12146.6670561076 * t);
    result += 3.91e-09 * std::cos(3.68718382989 + 18052.9295431578 * t);
    result += 4.57e-09 * std::cos(3.7720573734 + 156137.475984799 * t);
    result += 4.01e-09 * std::cos(5.28260651958 + 15671.0817594066 * t);
    result += 4.69e-09 * std::cos(1.80963184268 + 12562.6285816338 * t);
    result += 5.08e-09 * std::cos(3.36399024699 + 20597.2439630412 * t);
    result += 4.5e-09 * std::cos(5.6605429925 + 10454.5013866052 * t);
    result += 3.75e-09 * std::cos(4.98534633105 + 9779.1086761254 * t);
    result += 5.23e-09 * std::cos(0.97215560834 + 155427.542936241 * t);
    result += 4.03e-09 * std::cos(5.13939866506 + 1551.045222648 * t);
    result += 3.72e-09 * std::cos(3.69883738807 + 9388.0059094152 * t);
    result += 3.67e-09 * std::cos(4.43875659716 + 4535.0594369244 * t);
    result += 4.06e-09 * std::cos(4.208631566 + 12592.4500197826 * t);
    result += 3.6e-09 * std::cos(2.53924644657 + 242.728603974 * t);
    result += 4.71e-09 * std::cos(4.61907324819 + 5436.9930152402 * t);
    result += 4.41e-09 * std::cos(5.83872966262 + 3496.032826134 * t);
    result += 3.85e-09 * std::cos(4.94496680973 + 24356.7807886416 * t);
    result += 3.49e-09 * std::cos(6.15018231784 + 19800.9459562248 * t);
    result += 3.55e-09 * std::cos(0.21895678106 + 5429.8794682394 * t);
    result += 3.44e-09 * std::cos(5.62993724928 + 2379.1644735716 * t);
    result += 3.8e-09 * std::cos(2.72105213143 + 11933.3679606696 * t);
    result += 4.32e-09 * std::cos(0.24221790536 + 17996.0311682222 * t);
    result += 3.78e-09 * std::cos(5.22517556974 + 7477.522860216 * t);
    result += 3.37e-09 * std::cos(5.10888041439 + 5849.3641121146 * t);
    result += 3.15e-09 * std::cos(0.57827745123 + 10557.5941608238 * t);
    result += 3.18e-09 * std::cos(4.49953141399 + 3634.6210245184 * t);
    result += 3.23e-09 * std::cos(1.54274281393 + 10440.2742926036 * t);
    result += 3.09e-09 * std::cos(5.76839284397 + 20.7753954924 * t);
    result += 3.01e-09 * std::cos(2.34727604008 + 4686.8894077068 * t);
    result += 4.14e-09 * std::cos(5.9323760231 + 51092.7260508548 * t);
    result += 3.61e-09 * std::cos(2.1639860955 + 28237.2334593894 * t);
    result += 2.88e-09 * std::cos(0.18376252189 + 13095.8426650774 * t);
    result += 2.77e-09 * std::cos(5.12952205045 + 13119.7211028252 * t);
    result += 3.27e-09 * std::cos(6.19222146204 + 6268.8487559898 * t);
    result += 2.73e-09 * std::cos(0.30522428863 + 23141.5583829246 * t);
    result += 2.67e-09 * std::cos(5.76152585786 + 5966.6839803348 * t);
    result += 3.08e-09 * std::cos(5.99280509979 + 22805.7355659936 * t);
    result += 3.45e-09 * std::cos(2.92489919444 + 36949.2308084242 * t);
    result += 2.53e-09 * std::cos(5.20995219509 + 24072.9214697764 * t);
    result += 3.42e-09 * std::cos(5.72702586209 + 16460.333529525 * t);
    result += 2.61e-09 * std::cos(2.00304796059 + 6148.010769956 * t);
    result += 2.38e-09 * std::cos(5.08264392839 + 6915.8595893046 * t);
    result += 2.49e-09 * std::cos(2.94762789744 + 135.0650800354 * t);
    result += 3.06e-09 * std::cos(3.89764686987 + 10988.808157535 * t);
    result += 3.05e-09 * std::cos(0.05827812117 + 4701.1165017084 * t);
    result += 3.19e-09 * std::cos(2.95712862064 + 163096.180361183 * t);
    result += 2.09e-09 * std::cos(4.43768461442 + 6546.1597733642 * t);
    result += 2.7e-09 * std::cos(2.06643178717 + 4804.209275927 * t);
    result += 2.17e-09 * std::cos(0.73691592312 + 6303.8512454838 * t);
    result += 2.06e-09 * std::cos(0.32075959415 + 25934.1243310894 * t);
    result += 2.18e-09 * std::cos(0.18428135264 + 28286.9904848612 * t);
    result += 2.05e-09 * std::cos(5.21312087405 + 20995.3929664494 * t);
    result += 1.99e-09 * std::cos(0.44384292491 + 16737.5772365966 * t);
    result += 2.3e-09 * std::cos(6.06567392849 + 6287.0080032545 * t);
    result += 2.19e-09 * std::cos(1.291942163 + 5326.7866940208 * t);
    result += 2.01e-09 * std::cos(1.74700937253 + 22743.4093795164 * t);
    result += 2.07e-09 * std::cos(4.45440927276 + 6279.4854213396 * t);
    result += 2.69e-09 * std::cos(6.0564044503 + 64471.9912417449 * t);
    result += 1.9e-09 * std::cos(0.99256176518 + 29296.6153895786 * t);
    result += 2.38e-09 * std::cos(5.42471431221 + 39609.6545831656 * t);
    result += 2.62e-09 * std::cos(5.26961924198 + 522.5774180938 * t);
    result += 2.1e-09 * std::cos(4.68618183158 + 6254.6266625236 * t);
    result += 1.97e-09 * std::cos(2.8062455408 + 4933.2084403326 * t);
    result += 2.52e-09 * std::cos(4.36220154608 + 40879.4405046438 * t);
    result += 2.61e-09 * std::cos(1.07241516738 + 55022.9357470744 * t);
    result += 1.89e-09 * std::cos(3.82966734476 + 419.4846438752 * t);
    result += 1.85e-09 * std::cos(4.14324541379 + 5642.1982426092 * t);
    result += 2.47e-09 * std::cos(3.44855612987 + 6702.5604938666 * t);
    result += 2.05e-09 * std::cos(4.04424043223 + 536.8045120954 * t);
    result += 1.91e-09 * std::cos(3.14082686083 + 16723.350142595 * t);
    result += 2.22e-09 * std::cos(5.16263907319 + 23539.7073863328 * t);
    result += 1.8e-09 * std::cos(4.56214752149 + 6489.2613984286 * t);
    result += 2.19e-09 * std::cos(0.80382553358 + 16627.3709153772 * t);
    result += 2.27e-09 * std::cos(0.60156339452 + 5905.7022420756 * t);
    result += 1.68e-09 * std::cos(0.88753528161 + 16062.1845261168 * t);
    result += 1.58e-09 * std::cos(0.92127725775 + 23937.856389741 * t);
    result += 1.57e-09 * std::cos(4.69607868164 + 6805.6532680852 * t);
    result += 2.07e-09 * std::cos(4.88410451334 + 6286.6662786432 * t);
    result += 1.6e-09 * std::cos(4.95943826846 + 10021.8372800994 * t);
    result += 1.66e-09 * std::cos(0.97126433565 + 3097.88382272579 * t);
    result += 2.09e-09 * std::cos(5.75663411805 + 3646.3503773544 * t);
    result += 1.75e-09 * std::cos(6.12762824412 + 239424.390254353 * t);
    result += 1.73e-09 * std::cos(3.13887234973 + 6179.9830757728 * t);
    result += 1.57e-09 * std::cos(3.62822058179 + 18451.078546566 * t);
    result += 1.57e-09 * std::cos(4.67695912235 + 6709.6740408674 * t);
    result += 1.46e-09 * std::cos(3.09506069735 + 4907.3020501456 * t);
    result += 1.65e-09 * std::cos(2.2713912876 + 10660.6869350424 * t);
    result += 2.01e-09 * std::cos(1.67701267433 + 2107.0345075424 * t);
    result += 1.44e-09 * std::cos(3.96947747592 + 6019.9919266186 * t);
    result += 1.71e-09 * std::cos(5.91302216729 + 6058.7310542895 * t);
    result += 1.44e-09 * std::cos(2.1315565512 + 26084.0218062162 * t);
    result += 1.51e-09 * std::cos(0.67417383554 + 2388.8940204492 * t);
    result += 1.89e-09 * std::cos(5.07122281033 + 263.0839233728 * t);
    result += 1.46e-09 * std::cos(5.10373877968 + 10770.8932562618 * t);
    result += 1.87e-09 * std::cos(1.23915444627 + 19402.7969528166 * t);
    result += 1.74e-09 * std::cos(0.08407293391 + 9380.9596727172 * t);
    result += 1.37e-09 * std::cos(1.26247412309 + 12566.2190102856 * t);
    result += 1.37e-09 * std::cos(3.52826010842 + 639.897286314 * t);
    result += 1.48e-09 * std::cos(1.76124372592 + 5888.4499649322 * t);
    result += 1.64e-09 * std::cos(2.39195095081 + 6357.8574485587 * t);
    result += 1.46e-09 * std::cos(2.43675816553 + 5881.4037282342 * t);
    result += 1.61e-09 * std::cos(1.15721259372 + 26735.9452622132 * t);
    result += 1.31e-09 * std::cos(2.51859277344 + 6599.467719648 * t);
    result += 1.53e-09 * std::cos(5.85203687779 + 6281.5913772831 * t);
    result += 1.51e-09 * std::cos(3.72338532649 + 12669.2444742014 * t);
    result += 1.32e-09 * std::cos(2.38417741883 + 6525.8044539654 * t);
    result += 1.29e-09 * std::cos(0.75556744143 + 5017.508371365 * t);
    result += 1.27e-09 * std::cos(0.00254936441 + 10027.9031957292 * t);
    result += 1.48e-09 * std::cos(2.85102145528 + 6418.1409300268 * t);
    result += 1.43e-09 * std::cos(5.74460279367 + 26087.9031415742 * t);
    result += 1.72e-09 * std::cos(0.4128996224 + 174242.46596405 * t);
    result += 1.36e-09 * std::cos(4.15497742275 + 6311.5250374592 * t);
    result += 1.7e-09 * std::cos(5.98194913129 + 327574.514276781 * t);
    result += 1.24e-09 * std::cos(1.65497607604 + 32217.2001810808 * t);
    result += 1.36e-09 * std::cos(2.48430783417 + 13341.6743113068 * t);
    result += 1.65e-09 * std::cos(2.496679246 + 58953.145443294 * t);
    result += 1.23e-09 * std::cos(3.45660563754 + 6277.552925684 * t);
    result += 1.17e-09 * std::cos(0.86065134175 + 6245.0481773556 * t);
    result += 1.49e-09 * std::cos(5.61358280963 + 5729.506447149 * t);
    result += 1.53e-09 * std::cos(0.2686002995 + 245.8316462294 * t);
    result += 1.28e-09 * std::cos(0.71204006588 + 103.0927742186 * t);
    result += 1.59e-09 * std::cos(2.43166592149 + 221995.028801495 * t);
    result += 1.3e-09 * std::cos(2.80707316718 + 6016.4688082696 * t);
    result += 1.37e-09 * std::cos(1.70657709294 + 12566.08438968 * t);
    result += 1.11e-09 * std::cos(1.56305648432 + 17782.7320727842 * t);
    result += 1.13e-09 * std::cos(3.58302904101 + 25685.872802808 * t);
    result += 1.09e-09 * std::cos(3.26403795962 + 6819.8803620868 * t);
    result += 1.22e-09 * std::cos(0.34120688217 + 1162.4747044078 * t);
    result += 1.19e-09 * std::cos(5.84644718278 + 12721.572099417 * t);
    result += 1.44e-09 * std::cos(2.28899679126 + 12489.8856287072 * t);
    result += 1.37e-09 * std::cos(5.82029768354 + 44809.6502008634 * t);
    result += 1.07e-09 * std::cos(2.4281854414 + 5547.1993364596 * t);
    result += 1.34e-09 * std::cos(1.26539982939 + 5331.3574437408 * t);
    result += 1.03e-09 * std::cos(5.96518130595 + 6321.1035226272 * t);
    result += 1.09e-09 * std::cos(0.33808549034 + 11300.5842213564 * t);
    result += 1.29e-09 * std::cos(5.89187277327 + 12029.3471878874 * t);
    result += 1.22e-09 * std::cos(5.77325634636 + 11919.140866668 * t);
    result += 1.07e-09 * std::cos(6.2499898935 + 77690.7595057385 * t);
    result += 1.07e-09 * std::cos(1.00535580713 + 77736.7834305025 * t);
    result += 1.43e-09 * std::cos(0.24122178432 + 4214.0690150848 * t);
    result += 1.43e-09 * std::cos(0.88529649733 + 7576.560073574 * t);
    result += 1.07e-09 * std::cos(2.92124030496 + 31415.379249957 * t);
    result += 9.9e-10 * std::cos(5.70862227072 + 5540.0857894588 * t);
    result += 1.1e-09 * std::cos(0.37528037383 + 5863.5912061162 * t);
    result += 1.04e-09 * std::cos(4.44107178366 + 2118.7638603784 * t);
    result += 9.8e-10 * std::cos(5.95877916706 + 4061.2192153944 * t);
    result += 1.13e-09 * std::cos(1.24206857385 + 84672.4758445047 * t);
    result += 1.24e-09 * std::cos(2.55619029867 + 12539.853380183 * t);
    result += 1.1e-09 * std::cos(3.66952094329 + 238004.524157236 * t);
    result += 1.12e-09 * std::cos(4.32512422943 + 97238.6275444875 * t);
    result += 9.7e-10 * std::cos(3.70151541181 + 11720.0688652316 * t);
    result += 1.2e-09 * std::cos(1.26895630252 + 12043.574281889 * t);
    result += 9.4e-10 * std::cos(2.56461130309 + 19004.6479494084 * t);
    result += 1.17e-09 * std::cos(3.65425622684 + 34520.3093093808 * t);
    result += 9.8e-10 * std::cos(0.13589994287 + 11080.1715789176 * t);
    result += 9.7e-10 * std::cos(5.38330115253 + 7834.1210726394 * t);
    result += 9.7e-10 * std::cos(2.46722096722 + 71980.6335747312 * t);
    result += 9.5e-10 * std::cos(5.36958330451 + 6288.5987742988 * t);
    result += 1.11e-09 * std::cos(5.01961920313 + 11823.1616394502 * t);
    result += 9e-10 * std::cos(2.72299804525 + 26880.3198130326 * t);
    result += 9.9e-10 * std::cos(0.90164266377 + 18635.9284545362 * t);
    result += 1.26e-09 * std::cos(4.78722177847 + 305281.943071049 * t);
    result += 9.3e-10 * std::cos(0.21240380046 + 18139.2945014159 * t);
    result += 1.24e-09 * std::cos(5.00979495566 + 172146.97134054 * t);
    result += 9.9e-10 * std::cos(5.67090026475 + 16522.6597160022 * t);
    result += 9.2e-10 * std::cos(2.28180963676 + 12491.3701014155 * t);
    result += 9e-10 * std::cos(4.50544881196 + 40077.61957352 * t);
    result += 1e-09 * std::cos(2.00639461612 + 12323.4230960088 * t);
    result += 9.5e-10 * std::cos(5.68801979087 + 14919.0178537546 * t);
    result += 8.7e-10 * std::cos(1.86043406047 + 27707.5424942948 * t);
    result += 1.05e-09 * std::cos(3.02903468417 + 22345.2603761082 * t);
    result += 8.7e-10 * std::cos(5.43970168638 + 6272.0301497275 * t);
    result += 8.9e-10 * std::cos(1.63389387182 + 33326.5787331742 * t);
    result += 8.2e-10 * std::cos(5.58298993353 + 10241.2022911672 * t);
    result += 9.4e-10 * std::cos(5.47749711149 + 9924.8104215106 * t);
    result += 8.2e-10 * std::cos(4.71988314145 + 15141.390794312 * t);
    result += 9.7e-10 * std::cos(5.61458778738 + 2787.0430238574 * t);
    result += 9.6e-10 * std::cos(3.89073946348 + 6379.0550772092 * t);
    result += 8.1e-10 * std::cos(3.13038482444 + 36147.4098773004 * t);
    result += 1.1e-09 * std::cos(4.89978492291 + 72140.6286666874 * t);
    result += 9.7e-10 * std::cos(5.20764563059 + 6303.4311693902 * t);
    result += 8.2e-10 * std::cos(5.26342716139 + 9814.6041002912 * t);
    result += 1.09e-09 * std::cos(2.3555558977 + 83286.9142695536 * t);
    result += 9.7e-10 * std::cos(2.58492958057 + 30666.1549584328 * t);
    result += 9.3e-10 * std::cos(1.32651591333 + 23020.653086588 * t);
    result += 7.8e-10 * std::cos(3.99588630754 + 11293.4706743556 * t);
    result += 9e-10 * std::cos(0.57771932738 + 26482.1708096244 * t);
    result += 1.06e-09 * std::cos(3.92012705073 + 62883.3551395136 * t);
    result += 9.8e-10 * std::cos(2.94397773524 + 316.3918696566 * t);
    result += 7.6e-10 * std::cos(3.96310417608 + 29026.4852295078 * t);
    result += 7.8e-10 * std::cos(1.97068529306 + 90279.9231681033 * t);
    result += 7.6e-10 * std::cos(0.23027966596 + 21424.4666443034 * t);
    result += 8e-10 * std::cos(2.23099742212 + 266.6070417218 * t);
    result += 7.9e-10 * std::cos(1.46227790922 + 8982.810669309 * t);
    result += 1.02e-09 * std::cos(4.92129953565 + 5621.8429232104 * t);
    result += 1e-09 * std::cos(0.39243148321 + 24279.1070182136 * t);
    result += 7.1e-10 * std::cos(1.52014858474 + 33794.5437235286 * t);
    result += 7.6e-10 * std::cos(0.22880641443 + 57375.8019008462 * t);
    result += 9.1e-10 * std::cos(0.96515913904 + 48739.859897083 * t);
    result += 7.5e-10 * std::cos(2.77638585157 + 12964.300703391 * t);
    result += 7.7e-10 * std::cos(5.18846946344 + 11520.9968637952 * t);
    result += 6.8e-10 * std::cos(0.50006599129 + 4274.5183108324 * t);
    result += 7.5e-10 * std::cos(2.07323762803 + 15664.0355227086 * t);
    result += 7.4e-10 * std::cos(1.01884134928 + 6393.2821712108 * t);
    result += 7.7e-10 * std::cos(0.4666517878 + 16207.886271502 * t);
    result += 8.1e-10 * std::cos(4.10452219483 + 161710.618786232 * t);
    result += 6.7e-10 * std::cos(3.83840630887 + 6262.7205305926 * t);
    result += 7.1e-10 * std::cos(3.91415523291 + 7875.6718636242 * t);
    result += 8.1e-10 * std::cos(0.91938383237 + 74.7815985673 * t);
    result += 8.3e-10 * std::cos(4.69916218791 + 23006.4259925864 * t);
    result += 6.3e-10 * std::cos(2.32556465878 + 6279.1945146334 * t);
    result += 6.5e-10 * std::cos(5.41938745446 + 28628.3362260996 * t);
    result += 6.5e-10 * std::cos(3.02336771694 + 5959.570433334 * t);
    result += 6.4e-10 * std::cos(3.3103319837 + 2636.725472637 * t);
    result += 6.4e-10 * std::cos(0.18375587519 + 1066.49547719 * t);
    result += 8e-10 * std::cos(5.81239171612 + 12341.8069042809 * t);
    result += 6.6e-10 * std::cos(2.15105504851 + 38.0276726358 * t);
    result += 6.2e-10 * std::cos(2.43313614978 + 10138.1095169486 * t);
    result += 6e-10 * std::cos(3.1615390647 + 5490.300961524 * t);
    result += 6.9e-10 * std::cos(0.30764736334 + 7018.9523635232 * t);
    result += 6.8e-10 * std::cos(2.24442548639 + 24383.0791084414 * t);
    result += 7.8e-10 * std::cos(1.39649386463 + 9411.4646150872 * t);
    result += 6.3e-10 * std::cos(0.72976362625 + 6286.9571853494 * t);
    result += 7.3e-10 * std::cos(4.95125917731 + 6453.7487206106 * t);
    result += 7.8e-10 * std::cos(0.32736023459 + 6528.9074962208 * t);
    result += 5.9e-10 * std::cos(4.95362151577 + 35707.7100829074 * t);
    result += 7e-10 * std::cos(2.37962727525 + 15508.6151232744 * t);
    result += 7.3e-10 * std::cos(1.35229143111 + 5327.4761083828 * t);
    result += 7.2e-10 * std::cos(5.91833527334 + 10881.0995774812 * t);
    result += 5.9e-10 * std::cos(5.36231868425 + 10239.5838660108 * t);
    result += 5.9e-10 * std::cos(1.63156134967 + 61306.0115970658 * t);
    result += 5.4e-10 * std::cos(4.29491690425 + 21947.1113727 * t);
    result += 5.7e-10 * std::cos(5.89190132575 + 34513.2630726828 * t);
    result += 7.4e-10 * std::cos(1.38235845304 + 9967.4538999816 * t);
    result += 5.3e-10 * std::cos(3.86543309344 + 32370.9789915656 * t);
    result += 5.5e-10 * std::cos(4.51794544854 + 34911.412076091 * t);
    result += 6.3e-10 * std::cos(5.41479412056 + 11502.8376165305 * t);
    result += 6.3e-10 * std::cos(2.34416220742 + 11510.7019230567 * t);
    result += 6.8e-10 * std::cos(0.77493931112 + 29864.334027309 * t);
    result += 6e-10 * std::cos(5.57024703495 + 5756.9080032458 * t);
    result += 7.2e-10 * std::cos(2.80863088166 + 10866.8724834796 * t);
    result += 6.1e-10 * std::cos(2.69736991384 + 82576.9812209953 * t);
    result += 6.3e-10 * std::cos(5.32068807257 + 3116.6594122598 * t);
    result += 5.2e-10 * std::cos(1.02278758099 + 6272.4391846416 * t);
    result += 6.9e-10 * std::cos(5.00698550308 + 25287.7237993998 * t);
    result += 6.6e-10 * std::cos(6.12047940728 + 12074.488407524 * t);
    result += 5.1e-10 * std::cos(2.59519527563 + 11396.5634485742 * t);
    result += 5.6e-10 * std::cos(2.57995973521 + 17892.9383940036 * t);
    result += 5.9e-10 * std::cos(0.4416723762 + 250570.675857219 * t);
    result += 5.9e-10 * std::cos(3.84070143543 + 5483.254724826 * t);
    result += 4.9e-10 * std::cos(0.54704693048 + 22594.054895712 * t);
    result += 6.5e-10 * std::cos(2.38423614501 + 52670.0695933026 * t);
    result += 6.9e-10 * std::cos(5.34363738671 + 66813.5648357332 * t);
    result += 5.7e-10 * std::cos(5.42770501007 + 310145.152823924 * t);
    result += 5.3e-10 * std::cos(1.17760296075 + 149.5631971346 * t);
    result += 6.1e-10 * std::cos(4.02090887211 + 34596.3646546524 * t);
    result += 4.9e-10 * std::cos(4.18361320516 + 18606.4989460002 * t);
    result += 5.5e-10 * std::cos(0.83886167974 + 20452.8694122218 * t);
    result += 5e-10 * std::cos(1.46327331958 + 37455.7264959744 * t);
    result += 4.8e-10 * std::cos(4.53854727167 + 29822.7832363242 * t);
    result += 5.8e-10 * std::cos(3.34847975377 + 33990.6183442862 * t);
    result += 6.5e-10 * std::cos(1.45522693982 + 76251.3277706202 * t);
    result += 5.6e-10 * std::cos(2.35650663692 + 37724.7534197482 * t);
    result += 5.2e-10 * std::cos(2.61551081496 + 5999.2165311262 * t);
    result += 5.3e-10 * std::cos(0.17334326094 + 77717.2945864695 * t);
    result += 5.3e-10 * std::cos(0.79879700631 + 77710.2483497715 * t);
    result += 4.7e-10 * std::cos(0.43240779709 + 735.8765135318 * t);
    result += 5.3e-10 * std::cos(4.58763261686 + 11616.976091013 * t);
    result += 4.8e-10 * std::cos(6.20230111054 + 4171.4255366138 * t);
    result += 5.2e-10 * std::cos(1.09723616404 + 640.8776073822 * t);
    result += 5.7e-10 * std::cos(3.42008310383 + 50317.2034395308 * t);
    result += 5.3e-10 * std::cos(1.01528448581 + 149144.46708625 * t);
    result += 4.7e-10 * std::cos(3.00924906195 + 52175.8062831484 * t);
    result += 5.2e-10 * std::cos(2.03254070404 + 6293.7125153412 * t);
    result += 4.8e-10 * std::cos(0.12356889734 + 13362.4497067992 * t);
    result += 4.5e-10 * std::cos(3.37963782356 + 10763.779709261 * t);
    result += 4.7e-10 * std::cos(5.50981287869 + 12779.4507954208 * t);
    result += 6.2e-10 * std::cos(5.45209070099 + 949.1756089698 * t);
    result += 6.1e-10 * std::cos(2.93237974631 + 5791.4125575326 * t);
    result += 4.4e-10 * std::cos(2.87440620802 + 8584.6616659008 * t);
    result += 4.6e-10 * std::cos(4.0314179656 + 10667.8004820432 * t);
    result += 4.7e-10 * std::cos(3.89902931422 + 3903.9113764198 * t);
    result += 4.6e-10 * std::cos(2.75700467329 + 6993.0088985497 * t);
    result += 4.5e-10 * std::cos(1.933862933 + 206.1855484372 * t);
    result += 4.7e-10 * std::cos(2.57670800912 + 11492.542675792 * t);
    result += 4.4e-10 * std::cos(3.62570223167 + 63658.8777508376 * t);
    result += 5.1e-10 * std::cos(0.84536826273 + 12345.739057544 * t);
    result += 4.3e-10 * std::cos(0.01524970172 + 37853.8754993826 * t);
    result += 4.1e-10 * std::cos(3.27146326065 + 8858.3149443206 * t);
    result += 4.5e-10 * std::cos(3.03765521215 + 65236.2212932854 * t);
    result += 4.7e-10 * std::cos(1.44447548944 + 21393.5419698576 * t);
    result += 5.8e-10 * std::cos(5.45843180927 + 1975.492545856 * t);
    result += 5e-10 * std::cos(2.13285524146 + 12573.2652469836 * t);
    result += 4.1e-10 * std::cos(1.32190847146 + 2547.8375382324 * t);
    result += 4.7e-10 * std::cos(3.67579608544 + 28313.288804661 * t);
    result += 4.1e-10 * std::cos(2.24013475126 + 8273.8208670324 * t);
    result += 4.7e-10 * std::cos(6.21438985953 + 10991.3058987006 * t);
    result += 4.2e-10 * std::cos(3.0163181735 + 853.196381752 * t);
    result += 5.6e-10 * std::cos(1.09773690181 + 77376.2010224076 * t);
    result += 4e-10 * std::cos(2.35698541041 + 2699.7348193176 * t);
    result += 4.3e-10 * std::cos(5.28030898459 + 17796.9591667858 * t);
    result += 5.4e-10 * std::cos(2.59175932091 + 22910.4467653686 * t);
    result += 5.4e-10 * std::cos(0.88027764102 + 71960.3865832237 * t);
    result += 5.5e-10 * std::cos(0.07988899477 + 83467.1563530173 * t);
    result += 3.9e-10 * std::cos(1.12867321442 + 9910.583327509 * t);
    result += 4e-10 * std::cos(1.35670430524 + 27177.8515292002 * t);
    result += 3.9e-10 * std::cos(4.39624220245 + 5618.3198048614 * t);
    result += 4.2e-10 * std::cos(4.78798367468 + 7856.89627409019 * t);
    result += 4.7e-10 * std::cos(2.75482175292 + 18202.2167166594 * t);
    result += 3.9e-10 * std::cos(1.97008298629 + 24491.4257925834 * t);
    result += 4.2e-10 * std::cos(4.04346599946 + 7863.9425107882 * t);
    result += 3.8e-10 * std::cos(0.49178679251 + 38650.173506199 * t);
    result += 3.6e-10 * std::cos(4.86047906533 + 4157.1984426122 * t);
    result += 4.3e-10 * std::cos(5.64354880978 + 1062.9050485382 * t);
    result += 3.6e-10 * std::cos(3.98066313627 + 12565.1713789146 * t);
    result += 4.2e-10 * std::cos(2.30753932657 + 6549.6828917132 * t);
    result += 4e-10 * std::cos(5.3969491832 + 9498.2122306346 * t);
    result += 4e-10 * std::cos(3.30603243754 + 23536.116957681 * t);
    result += 5e-10 * std::cos(6.15760345261 + 78051.3419138334 * t);
    return result;
}

constexpr double getEarthR1(double t)
{
    double result = 0.0;
    result += 0.00103018608 * std::cos(1.10748969588 + 6283.0758499914 * t);
    result += 1.721238e-05 * std::cos(1.06442301418 + 12566.1516999828 * t);
    result -= -7.02215e-06;
    result += 3.2346e-07 * std::cos(1.02169059149 + 18849.2275499742 * t);
    result += 3.0799e-07 * std::cos(2.84353804832 + 5507.5532386674 * t);
    result += 2.4971e-07 * std::cos(1.31906709482 + 5223.6939198022 * t);
    result += 1.8485e-07 * std::cos(1.42429748614 + 1577.3435424478 * t);
    result += 1.0078e-07 * std::cos(5.91378194648 + 10977.078804699 * t);
    result += 8.634e-08 * std::cos(0.27146150602 + 5486.777843175 * t);
    result += 8.654e-08 * std::cos(1.42046854427 + 6275.9623029906 * t);
    result += 5.069e-08 * std::cos(1.68613426734 + 5088.6288397668 * t);
    result += 4.985e-08 * std::cos(6.01401770704 + 6286.5989683404 * t);
    result += 4.669e-08 * std::cos(5.98724494073 + 529.6909650946 * t);
    result += 4.395e-08 * std::cos(0.51800238019 + 4694.0029547076 * t);
    result += 3.872e-08 * std::cos(4.74969833437 + 2544.3144198834 * t);
    result += 3.75e-08 * std::cos(5.07097685568 + 796.2980068164 * t);
    result += 4.1e-08 * std::cos(1.08424786092 + 9437.762934887 * t);
    result += 3.518e-08 * std::cos(0.02290216272 + 83996.8473181119 * t);
    result += 3.436e-08 * std::cos(0.94937019624 + 71430.6956181291 * t);
    result += 3.221e-08 * std::cos(6.15628775313 + 2146.1654164752 * t);
    result += 3.414e-08 * std::cos(5.41218322538 + 775.522611324 * t);
    result += 2.863e-08 * std::cos(5.48432847146 + 10447.3878396044 * t);
    result += 2.52e-08 * std::cos(0.24276941146 + 398.1490034082 * t);
    result += 2.201e-08 * std::cos(4.95216196651 + 6812.766815086 * t);
    result += 2.186e-08 * std::cos(0.41991743105 + 8031.0922630584 * t);
    result += 2.838e-08 * std::cos(3.42034351366 + 2352.8661537718 * t);
    result += 2.554e-08 * std::cos(6.13241878525 + 6438.4962494256 * t);
    result += 1.932e-08 * std::cos(5.31374608366 + 8429.2412664666 * t);
    result += 2.429e-08 * std::cos(3.09164528262 + 4690.4798363586 * t);
    result += 1.73e-08 * std::cos(1.5368620855 + 4705.7323075436 * t);
    result += 2.25e-08 * std::cos(3.68863633842 + 7084.8967811152 * t);
    result += 2.093e-08 * std::cos(1.28191783032 + 1748.016413067 * t);
    result += 1.441e-08 * std::cos(0.81656250862 + 14143.4952424306 * t);
    result += 1.483e-08 * std::cos(3.22225357771 + 7234.794256242 * t);
    result += 1.754e-08 * std::cos(3.22883705112 + 6279.5527316424 * t);
    result += 1.583e-08 * std::cos(4.09702349428 + 11499.6562227928 * t);
    result += 1.575e-08 * std::cos(5.53890170575 + 3154.6870848956 * t);
    result += 1.847e-08 * std::cos(1.82040335363 + 7632.9432596502 * t);
    result += 1.504e-08 * std::cos(3.63293385726 + 11513.8833167944 * t);
    result += 1.337e-08 * std::cos(4.64440864339 + 6836.6452528338 * t);
    result += 1.275e-08 * std::cos(2.69341415363 + 1349.8674096588 * t);
    result += 1.352e-08 * std::cos(6.15101580257 + 5746.271337896 * t);
    result += 1.125e-08 * std::cos(3.35673439497 + 17789.845619785 * t);
    result += 1.47e-08 * std::cos(3.65282991755 + 1194.4470102246 * t);
    result += 1.177e-08 * std::cos(2.57676109092 + 13367.9726311066 * t);
    result += 1.101e-08 * std::cos(4.49748696552 + 4292.3308329504 * t);
    result += 1.234e-08 * std::cos(5.65036509521 + 5760.4984318976 * t);
    result += 9.84e-09 * std::cos(0.65517395136 + 5856.4776591154 * t);
    result += 9.28e-09 * std::cos(2.32420318751 + 10213.285546211 * t);
    result += 1.077e-08 * std::cos(5.82812169132 + 12036.4607348882 * t);
    result += 9.16e-09 * std::cos(0.76613009583 + 16730.4636895958 * t);
    result += 8.77e-09 * std::cos(1.50137505051 + 11926.2544136688 * t);
    result += 1.023e-08 * std::cos(5.62076589825 + 6256.7775301916 * t);
    result += 8.51e-09 * std::cos(0.65709335533 + 155.4203994342 * t);
    result += 8.02e-09 * std::cos(4.10519132088 + 951.7184062506 * t);
    result += 8.57e-09 * std::cos(1.41661697538 + 5753.3848848968 * t);
    result += 9.94e-09 * std::cos(1.14418521187 + 1059.3819301892 * t);
    result += 8.13e-09 * std::cos(1.63948433322 + 6681.2248533996 * t);
    result += 6.62e-09 * std::cos(4.5520045226 + 5216.5803728014 * t);
    result += 6.44e-09 * std::cos(4.19478168733 + 6040.3472460174 * t);
    result += 6.26e-09 * std::cos(1.50767713598 + 5643.1785636774 * t);
    result += 5.9e-09 * std::cos(6.18277145205 + 4164.311989613 * t);
    result += 6.35e-09 * std::cos(0.52413263542 + 6290.1893969922 * t);
    result += 6.5e-09 * std::cos(0.9793569035 + 25132.3033999656 * t);
    result += 5.68e-09 * std::cos(2.30125315873 + 10973.55568635 * t);
    result += 5.47e-09 * std::cos(5.27256412213 + 3340.6124266998 * t);
    result += 5.47e-09 * std::cos(2.20144422886 + 1592.5960136328 * t);
    result += 5.26e-09 * std::cos(0.92464258226 + 11371.7046897582 * t);
    result += 4.9e-09 * std::cos(5.90951388655 + 3894.1818295422 * t);
    result += 4.78e-09 * std::cos(1.66857963179 + 12168.0026965746 * t);
    result += 5.16e-09 * std::cos(3.59803483887 + 10969.9652576982 * t);
    result += 5.18e-09 * std::cos(3.97914412373 + 17298.1823273262 * t);
    result += 5.34e-09 * std::cos(5.03740926442 + 9917.6968745098 * t);
    result += 4.87e-09 * std::cos(2.50545369269 + 6127.6554505572 * t);
    result += 4.16e-09 * std::cos(4.04828175503 + 10984.1923516998 * t);
    result += 5.38e-09 * std::cos(5.54081539805 + 553.5694028424 * t);
    result += 4.02e-09 * std::cos(2.16544019233 + 7860.4193924392 * t);
    result += 5.53e-09 * std::cos(2.32177369366 + 11506.7697697936 * t);
    result += 3.67e-09 * std::cos(3.3915253225 + 6496.3749454294 * t);
    result += 3.6e-09 * std::cos(5.34379853282 + 7079.3738568078 * t);
    result += 3.37e-09 * std::cos(3.61563704045 + 11790.6290886588 * t);
    result += 4.56e-09 * std::cos(0.30754294809 + 801.8209311238 * t);
    result += 4.17e-09 * std::cos(3.70009308674 + 10575.4066829418 * t);
    result += 3.81e-09 * std::cos(5.82033971802 + 7058.5984613154 * t);
    result += 3.21e-09 * std::cos(0.31988767355 + 16200.7727245012 * t);
    result += 3.64e-09 * std::cos(1.08414306177 + 6309.3741697912 * t);
    result += 2.94e-09 * std::cos(4.54798604957 + 11856.2186514245 * t);
    result += 2.9e-09 * std::cos(1.26473978562 + 8635.9420037632 * t);
    result += 3.99e-09 * std::cos(4.16998866302 + 26.2983197998 * t);
    result += 2.62e-09 * std::cos(5.08316906342 + 10177.2576795336 * t);
    result += 2.43e-09 * std::cos(2.2574609119 + 11712.9553182308 * t);
    result += 2.37e-09 * std::cos(1.05070575346 + 242.728603974 * t);
    result += 2.75e-09 * std::cos(3.45319481756 + 5884.9268465832 * t);
    result += 2.55e-09 * std::cos(5.38496831087 + 21228.3920235458 * t);
    result += 3.07e-09 * std::cos(4.24313526604 + 3738.761430108 * t);
    result += 2.16e-09 * std::cos(3.46037894728 + 213.299095438 * t);
    result += 1.96e-09 * std::cos(0.69029243914 + 1990.745017041 * t);
    result += 1.98e-09 * std::cos(5.16301829964 + 12352.8526045448 * t);
    result += 2.14e-09 * std::cos(3.91876200279 + 13916.0191096416 * t);
    result += 2.12e-09 * std::cos(4.00861198517 + 5230.807466803 * t);
    result += 1.84e-09 * std::cos(5.59805976614 + 6283.14316029419 * t);
    result += 1.84e-09 * std::cos(2.85275392124 + 7238.6755916 * t);
    result += 1.79e-09 * std::cos(2.54259058334 + 14314.1681130498 * t);
    result += 2.25e-09 * std::cos(1.64458698399 + 4732.0306273434 * t);
    result += 2.36e-09 * std::cos(5.58826125715 + 6069.7767545534 * t);
    result += 1.87e-09 * std::cos(2.72805985443 + 6062.6632075526 * t);
    result += 1.84e-09 * std::cos(6.04216273598 + 6283.0085396886 * t);
    result += 2.3e-09 * std::cos(3.62591335086 + 6284.0561710596 * t);
    result += 1.63e-09 * std::cos(2.19117396803 + 18073.7049386502 * t);
    result += 1.72e-09 * std::cos(0.9761295074 + 3930.2096962196 * t);
    result += 2.15e-09 * std::cos(1.04672844028 + 3496.032826134 * t);
    result += 1.69e-09 * std::cos(4.75084479006 + 17267.2682016912 * t);
    result += 1.52e-09 * std::cos(0.19390712179 + 9779.1086761254 * t);
    result += 1.82e-09 * std::cos(5.16288118255 + 17253.0411076896 * t);
    result += 1.49e-09 * std::cos(0.8094418426 + 709.9330485583 * t);
    result += 1.63e-09 * std::cos(2.1920957039 + 6076.8903015542 * t);
    result += 1.86e-09 * std::cos(5.01159497089 + 11015.1064773348 * t);
    result += 1.34e-09 * std::cos(0.97765485759 + 65147.6197681377 * t);
    result += 1.41e-09 * std::cos(4.38421981312 + 4136.9104335162 * t);
    result += 1.58e-09 * std::cos(4.60974280627 + 9623.6882766912 * t);
    result += 1.33e-09 * std::cos(3.30508592837 + 154717.609887683 * t);
    result += 1.63e-09 * std::cos(6.11782626245 + 3.523118349 * t);
    result += 1.74e-09 * std::cos(1.58078542187 + 7.1135470008 * t);
    result += 1.41e-09 * std::cos(0.49976927274 + 25158.6017197654 * t);
    result += 1.24e-09 * std::cos(6.03440460031 + 9225.539273283 * t);
    result += 1.5e-09 * std::cos(5.30166336812 + 13517.8701062334 * t);
    result += 1.27e-09 * std::cos(1.92389511438 + 22483.8485744926 * t);
    result += 1.21e-09 * std::cos(2.37813129011 + 167283.761587666 * t);
    result += 1.2e-09 * std::cos(3.98423684853 + 4686.8894077068 * t);
    result += 1.17e-09 * std::cos(5.81072642211 + 12569.6748183318 * t);
    result += 1.22e-09 * std::cos(5.60973054224 + 5642.1982426092 * t);
    result += 1.57e-09 * std::cos(3.40236426002 + 16496.3613962024 * t);
    result += 1.29e-09 * std::cos(2.10705116371 + 1589.0728952838 * t);
    result += 1.16e-09 * std::cos(0.55839966736 + 5849.3641121146 * t);
    result += 1.23e-09 * std::cos(1.52961392771 + 12559.038152982 * t);
    result += 1.11e-09 * std::cos(0.44848279675 + 6172.869528772 * t);
    result += 1.23e-09 * std::cos(5.81645568991 + 6282.0955289232 * t);
    result += 1.5e-09 * std::cos(4.26278409223 + 3128.3887650958 * t);
    result += 1.06e-09 * std::cos(2.27437761356 + 5429.8794682394 * t);
    result += 1.04e-09 * std::cos(4.42743707728 + 23543.2305046818 * t);
    result += 1.21e-09 * std::cos(0.39459045915 + 12132.439962106 * t);
    result += 1.04e-09 * std::cos(2.41842602527 + 426.598190876 * t);
    result += 1.1e-09 * std::cos(5.80381480447 + 16858.4825329332 * t);
    result += 1e-09 * std::cos(2.93805577485 + 4535.0594369244 * t);
    result += 9.7e-10 * std::cos(3.97935904984 + 6133.5126528568 * t);
    result += 1.1e-09 * std::cos(6.22339014386 + 12146.6670561076 * t);
    result += 9.8e-10 * std::cos(0.87576563709 + 6525.8044539654 * t);
    result += 9.8e-10 * std::cos(3.15248421301 + 10440.2742926036 * t);
    result += 9.5e-10 * std::cos(2.461684111 + 3097.88382272579 * t);
    result += 8.8e-10 * std::cos(0.23371480284 + 13119.7211028252 * t);
    result += 9.8e-10 * std::cos(5.77016493489 + 7342.4577801806 * t);
    result += 9.2e-10 * std::cos(6.03915555063 + 20426.571092422 * t);
    result += 9.6e-10 * std::cos(5.56909292561 + 2388.8940204492 * t);
    result += 8.1e-10 * std::cos(1.32131147691 + 5650.2921106782 * t);
    result += 8.6e-10 * std::cos(3.94529200528 + 10454.5013866052 * t);
    result += 7.6e-10 * std::cos(2.70729716925 + 143571.324284816 * t);
    result += 9.1e-10 * std::cos(5.64100034152 + 8827.3902698748 * t);
    result += 7.6e-10 * std::cos(1.80783856698 + 28286.9904848612 * t);
    result += 8.1e-10 * std::cos(1.90858992196 + 29088.811415985 * t);
    result += 7.5e-10 * std::cos(3.40955892978 + 5481.2549188676 * t);
    result += 6.9e-10 * std::cos(4.49936170873 + 17256.6315363414 * t);
    result += 8.8e-10 * std::cos(1.10098454357 + 11769.8536931664 * t);
    result += 6.6e-10 * std::cos(2.78285801977 + 536.8045120954 * t);
    result += 6.8e-10 * std::cos(3.88179770758 + 17260.1546546904 * t);
    result += 8.4e-10 * std::cos(1.59303306354 + 9380.9596727172 * t);
    result += 8.8e-10 * std::cos(3.88076636762 + 7477.522860216 * t);
    result += 6.1e-10 * std::cos(6.17558202197 + 11087.2851259184 * t);
    result += 6e-10 * std::cos(4.34824715818 + 6206.8097787158 * t);
    result += 8.2e-10 * std::cos(4.59843208943 + 9388.0059094152 * t);
    result += 7.9e-10 * std::cos(1.63131230601 + 4933.2084403326 * t);
    result += 7.8e-10 * std::cos(4.20905757484 + 5729.506447149 * t);
    result += 5.7e-10 * std::cos(5.48157926651 + 18319.5365848796 * t);
    result += 6e-10 * std::cos(1.01261781084 + 12721.572099417 * t);
    result += 5.6e-10 * std::cos(1.63031935692 + 15720.8387848784 * t);
    result += 5.5e-10 * std::cos(0.24926735018 + 15110.4661198662 * t);
    result += 6.1e-10 * std::cos(5.93059279661 + 12539.853380183 * t);
    result += 5.5e-10 * std::cos(4.84298966314 + 13095.8426650774 * t);
    result += 6.7e-10 * std::cos(6.11690589247 + 8662.240323563 * t);
    result += 5.4e-10 * std::cos(5.73750638571 + 3634.6210245184 * t);
    result += 7.4e-10 * std::cos(1.05466745829 + 16460.333529525 * t);
    result += 5.3e-10 * std::cos(2.29084335688 + 16062.1845261168 * t);
    result += 6.4e-10 * std::cos(2.13513767927 + 7875.6718636242 * t);
    result += 6.7e-10 * std::cos(0.07096807518 + 14945.3161735544 * t);
    result += 5.1e-10 * std::cos(2.31511194429 + 6262.7205305926 * t);
    result += 5.7e-10 * std::cos(5.77055471237 + 12043.574281889 * t);
    result += 5.6e-10 * std::cos(4.41980790431 + 4701.1165017084 * t);
    result += 5.9e-10 * std::cos(5.87963500073 + 5331.3574437408 * t);
    result += 5.8e-10 * std::cos(2.30546168628 + 955.5997416086 * t);
    result += 4.9e-10 * std::cos(1.93839278478 + 5333.9002410216 * t);
    result += 4.8e-10 * std::cos(2.69973662261 + 6709.6740408674 * t);
    result += 6.4e-10 * std::cos(1.64379897981 + 6262.300454499 * t);
    result += 4.6e-10 * std::cos(3.98449608961 + 98068.5367163054 * t);
    result += 5e-10 * std::cos(3.68875893005 + 12323.4230960088 * t);
    result += 4.5e-10 * std::cos(3.30068569697 + 22003.9146348698 * t);
    result += 4.7e-10 * std::cos(1.26317154881 + 11919.140866668 * t);
    result += 4.5e-10 * std::cos(0.89150445122 + 51868.2486621788 * t);
    result += 4.3e-10 * std::cos(1.61526242998 + 6277.552925684 * t);
    result += 4.3e-10 * std::cos(5.74295325645 + 11403.676995575 * t);
    result += 4.4e-10 * std::cos(3.43070646822 + 10021.8372800994 * t);
    result += 5.6e-10 * std::cos(0.02481833774 + 15671.0817594066 * t);
    result += 5.5e-10 * std::cos(3.14274403422 + 33019.0211122046 * t);
    result += 4.5e-10 * std::cos(3.00877289177 + 8982.810669309 * t);
    result += 4.6e-10 * std::cos(0.73303568429 + 6303.4311693902 * t);
    result += 4.9e-10 * std::cos(1.60455690285 + 6303.8512454838 * t);
    result += 4.5e-10 * std::cos(0.40210030323 + 6805.6532680852 * t);
    result += 5.3e-10 * std::cos(0.94869680175 + 10988.808157535 * t);
    result += 4.1e-10 * std::cos(1.61122384329 + 6819.8803620868 * t);
    result += 5.5e-10 * std::cos(0.89439119424 + 11933.3679606696 * t);
    result += 4.5e-10 * std::cos(3.88495384656 + 60530.4889857418 * t);
    result += 4e-10 * std::cos(4.75740908001 + 38526.574350872 * t);
    result += 4e-10 * std::cos(1.49921251887 + 18451.078546566 * t);
    result += 4e-10 * std::cos(3.77498297228 + 26087.9031415742 * t);
    result += 5.1e-10 * std::cos(1.70258603562 + 1551.045222648 * t);
    result += 3.9e-10 * std::cos(2.97100699926 + 2118.7638603784 * t);
    result += 5.3e-10 * std::cos(5.19854123078 + 77713.7714681205 * t);
    result += 4.7e-10 * std::cos(4.26356628717 + 21424.4666443034 * t);
    result += 3.7e-10 * std::cos(0.62902722802 + 24356.7807886416 * t);
    result += 3.6e-10 * std::cos(0.11087914947 + 10344.2950653858 * t);
    result += 3.6e-10 * std::cos(0.77037556319 + 12029.3471878874 * t);
    result += 3.5e-10 * std::cos(3.30933994515 + 24072.9214697764 * t);
    result += 3.5e-10 * std::cos(5.93650887012 + 31570.7996493912 * t);
    result += 3.6e-10 * std::cos(2.15108874765 + 30774.5016425748 * t);
    result += 3.6e-10 * std::cos(1.75078825382 + 16207.886271502 * t);
    result += 3.3e-10 * std::cos(5.06264177921 + 226858.23855437 * t);
    result += 3.4e-10 * std::cos(6.168913788 + 24491.4257925834 * t);
    result += 3.5e-10 * std::cos(3.19120695549 + 32217.2001810808 * t);
    result += 3.4e-10 * std::cos(2.31528650443 + 55798.4583583984 * t);
    result += 3.2e-10 * std::cos(4.21446357042 + 15664.0355227086 * t);
    result += 3.9e-10 * std::cos(1.24979117796 + 6418.1409300268 * t);
    result += 3.7e-10 * std::cos(4.1194365577 + 2787.0430238574 * t);
    result += 3.2e-10 * std::cos(1.6288771089 + 639.897286314 * t);
    result += 3.8e-10 * std::cos(5.89832942685 + 640.8776073822 * t);
    result += 3.2e-10 * std::cos(1.72442327688 + 27433.889215875 * t);
    result += 3.1e-10 * std::cos(2.78828943753 + 12139.5535091068 * t);
    result += 3.5e-10 * std::cos(4.44608896525 + 18202.2167166594 * t);
    result += 3.4e-10 * std::cos(3.96287980676 + 18216.443810661 * t);
    result += 3.3e-10 * std::cos(4.73611335874 + 16723.350142595 * t);
    result += 3.4e-10 * std::cos(1.43910280005 + 49515.382508407 * t);
    result += 3.1e-10 * std::cos(0.23302920161 + 23581.2581773176 * t);
    result += 2.9e-10 * std::cos(2.0263384022 + 11609.8625440122 * t);
    result += 3e-10 * std::cos(2.5492323024 + 9924.8104215106 * t);
    result += 3.2e-10 * std::cos(4.91793198558 + 11300.5842213564 * t);
    result += 2.8e-10 * std::cos(0.26187189577 + 13521.7514415914 * t);
    result += 2.8e-10 * std::cos(3.84568936822 + 2699.7348193176 * t);
    result += 2.9e-10 * std::cos(1.83149729794 + 29822.7832363242 * t);
    result += 3.3e-10 * std::cos(4.60320094415 + 19004.6479494084 * t);
    result += 2.7e-10 * std::cos(4.46183450287 + 6702.5604938666 * t);
    result += 3e-10 * std::cos(4.4649407224 + 36147.4098773004 * t);
    result += 2.7e-10 * std::cos(0.03211931363 + 6279.7894925736 * t);
    result += 2.6e-10 * std::cos(5.46497324333 + 6245.0481773556 * t);
    result += 3.5e-10 * std::cos(4.52695674113 + 36949.2308084242 * t);
    result += 2.7e-10 * std::cos(3.52528177609 + 10770.8932562618 * t);
    result += 2.6e-10 * std::cos(1.48499438453 + 11080.1715789176 * t);
    result += 3.5e-10 * std::cos(2.82154380962 + 19402.7969528166 * t);
    result += 2.5e-10 * std::cos(2.46339998836 + 6279.4854213396 * t);
    result += 2.6e-10 * std::cos(4.97688894643 + 16737.5772365966 * t);
    result += 2.6e-10 * std::cos(2.36136541526 + 17996.0311682222 * t);
    result += 2.9e-10 * std::cos(4.15148654061 + 45892.730433157 * t);
    result += 2.6e-10 * std::cos(4.50714272714 + 17796.9591667858 * t);
    result += 2.7e-10 * std::cos(4.72625223674 + 1066.49547719 * t);
    result += 2.5e-10 * std::cos(2.89309528854 + 6286.6662786432 * t);
    result += 2.7e-10 * std::cos(0.37462444357 + 12964.300703391 * t);
    result += 2.9e-10 * std::cos(4.94860010533 + 5863.5912061162 * t);
    result += 3.1e-10 * std::cos(3.93096113577 + 29864.334027309 * t);
    result += 2.4e-10 * std::cos(6.14987193584 + 18606.4989460002 * t);
    result += 2.4e-10 * std::cos(3.74225964547 + 29026.4852295078 * t);
    result += 2.5e-10 * std::cos(5.70460621565 + 27707.5424942948 * t);
    result += 2.5e-10 * std::cos(5.33928840652 + 15141.390794312 * t);
    result += 2.7e-10 * std::cos(3.0232089714 + 6286.3622074092 * t);
    result += 2.3e-10 * std::cos(0.28364955406 + 5327.4761083828 * t);
    result += 2.6e-10 * std::cos(1.34240461687 + 18875.525869774 * t);
    result += 2.4e-10 * std::cos(1.33998410121 + 19800.9459562248 * t);
    result += 2.5e-10 * std::cos(6.00172494004 + 6489.2613984286 * t);
    result += 2.2e-10 * std::cos(1.81777974484 + 6288.5987742988 * t);
    result += 2.2e-10 * std::cos(3.5860360664 + 6915.8595893046 * t);
    result += 2.9e-10 * std::cos(2.09564449439 + 15265.8865193004 * t);
    result += 2.2e-10 * std::cos(1.02173599251 + 11925.2740926006 * t);
    result += 2.2e-10 * std::cos(4.74660932338 + 28230.1872226914 * t);
    result += 2.1e-10 * std::cos(2.30688751432 + 5999.2165311262 * t);
    result += 2.1e-10 * std::cos(3.2265494443 + 25934.1243310894 * t);
    result += 2.1e-10 * std::cos(3.04956726238 + 6566.9351688566 * t);
    result += 2.7e-10 * std::cos(5.35653084499 + 33794.5437235286 * t);
    result += 2.8e-10 * std::cos(3.91168324815 + 18208.349942592 * t);
    result += 2e-10 * std::cos(1.52296293311 + 135.0650800354 * t);
    result += 2.2e-10 * std::cos(4.66462839521 + 13362.4497067992 * t);
    result += 1.9e-10 * std::cos(1.78121167862 + 156137.475984799 * t);
    result += 1.9e-10 * std::cos(2.99969102221 + 19651.048481098 * t);
    result += 1.9e-10 * std::cos(2.86664273362 + 18422.6293590982 * t);
    result += 2.5e-10 * std::cos(0.94995632141 + 31415.379249957 * t);
    result += 1.9e-10 * std::cos(4.71432851499 + 77690.7595057385 * t);
    result += 1.9e-10 * std::cos(2.54227398241 + 77736.7834305025 * t);
    result += 2e-10 * std::cos(5.91915117116 + 48739.859897083 * t);
    return result;
}

constexpr double getEarthR2(double t)
{
    double result = 0.0;
    result += 4.359385e-05 * std::cos(5.78455133738 + 6283.0758499914 * t);
    result += 1.23633e-06 * std::cos(5.57934722157 + 12566.1516999828 * t);
    result -= -1.2341e-07;
    result += 8.792e-08 * std::cos(3.62777733395 + 77713.7714681205 * t);
    result += 5.689e-08 * std::cos(1.86958905084 + 5573.1428014331 * t);
    result += 3.301e-08 * std::cos(5.47027913302 + 18849.2275499742 * t);
    result += 1.471e-08 * std::cos(4.48028885617 + 5507.5532386674 * t);
    result += 1.013e-08 * std::cos(2.81456417694 + 5223.6939198022 * t);
    result += 8.54e-09 * std::cos(3.10878241236 + 1577.3435424478 * t);
    result += 1.102e-08 * std::cos(2.84173992403 + 161000.685737674 * t);
    result += 6.48e-09 * std::cos(5.47349498544 + 775.522611324 * t);
    result += 6.09e-09 * std::cos(1.37969434104 + 6438.4962494256 * t);
    result += 4.99e-09 * std::cos(4.4164924225 + 6286.5989683404 * t);
    result += 4.17e-09 * std::cos(0.90242451175 + 10977.078804699 * t);
    result += 4.02e-09 * std::cos(3.2037658529 + 5088.6288397668 * t);
    result += 3.51e-09 * std::cos(1.8107922777 + 5486.777843175 * t);
    result += 4.67e-09 * std::cos(3.65753702738 + 7084.8967811152 * t);
    result += 4.58e-09 * std::cos(5.38585314743 + 149854.400134808 * t);
    result += 3.04e-09 * std::cos(3.51701098693 + 796.2980068164 * t);
    result += 2.66e-09 * std::cos(6.17413982699 + 6836.6452528338 * t);
    result += 2.79e-09 * std::cos(1.84120501086 + 4694.0029547076 * t);
    result += 2.6e-09 * std::cos(1.41629543251 + 2146.1654164752 * t);
    result += 2.66e-09 * std::cos(3.13832905677 + 71430.6956181291 * t);
    result += 3.21e-09 * std::cos(5.35313367048 + 3154.6870848956 * t);
    result += 2.38e-09 * std::cos(2.17720020018 + 155.4203994342 * t);
    result += 2.93e-09 * std::cos(4.61501268144 + 4690.4798363586 * t);
    result += 2.29e-09 * std::cos(4.7596958807 + 7234.794256242 * t);
    result += 2.11e-09 * std::cos(0.21868065485 + 4705.7323075436 * t);
    result += 2.01e-09 * std::cos(4.21905743357 + 1349.8674096588 * t);
    result += 1.95e-09 * std::cos(4.57808285364 + 529.6909650946 * t);
    result += 2.53e-09 * std::cos(2.81496293039 + 1748.016413067 * t);
    result += 1.82e-09 * std::cos(5.70454011389 + 6040.3472460174 * t);
    result += 1.79e-09 * std::cos(6.02897097053 + 4292.3308329504 * t);
    result += 1.86e-09 * std::cos(1.58690991244 + 6309.3741697912 * t);
    result += 1.7e-09 * std::cos(2.90220009715 + 9437.762934887 * t);
    result += 1.66e-09 * std::cos(1.99984925026 + 8031.0922630584 * t);
    result += 1.58e-09 * std::cos(0.04783713552 + 2544.3144198834 * t);
    result += 1.97e-09 * std::cos(2.01083639502 + 1194.4470102246 * t);
    result += 1.65e-09 * std::cos(5.78372596778 + 83996.8473181119 * t);
    result += 2.14e-09 * std::cos(3.38285934319 + 7632.9432596502 * t);
    result += 1.4e-09 * std::cos(0.36401486094 + 10447.3878396044 * t);
    result += 1.51e-09 * std::cos(0.95153163031 + 6127.6554505572 * t);
    result += 1.36e-09 * std::cos(1.48426306582 + 2352.8661537718 * t);
    result += 1.27e-09 * std::cos(5.48475435134 + 951.7184062506 * t);
    result += 1.26e-09 * std::cos(5.26866506592 + 6279.5527316424 * t);
    result += 1.25e-09 * std::cos(3.75754889288 + 6812.766815086 * t);
    result += 1.01e-09 * std::cos(4.95015746147 + 398.1490034082 * t);
    result += 1.02e-09 * std::cos(0.68468295277 + 1592.5960136328 * t);
    result += 1e-09 * std::cos(1.14568935785 + 3894.1818295422 * t);
    result += 1.29e-09 * std::cos(0.76540016965 + 553.5694028424 * t);
    result += 1.09e-09 * std::cos(5.41063597567 + 6256.7775301916 * t);
    result += 7.5e-10 * std::cos(5.84804322893 + 242.728603974 * t);
    result += 9.5e-10 * std::cos(1.94452244083 + 11856.2186514245 * t);
    result += 7.7e-10 * std::cos(0.69373708195 + 8429.2412664666 * t);
    result += 1e-09 * std::cos(5.19725292131 + 244287.600007228 * t);
    result += 8e-10 * std::cos(6.18440483705 + 1059.3819301892 * t);
    result += 6.9e-10 * std::cos(5.25699888595 + 14143.4952424306 * t);
    result += 8.5e-10 * std::cos(5.39484725499 + 25132.3033999656 * t);
    result += 6.6e-10 * std::cos(0.51779993906 + 801.8209311238 * t);
    result += 5.5e-10 * std::cos(5.16878202461 + 7058.5984613154 * t);
    result += 5.1e-10 * std::cos(3.88759155247 + 12036.4607348882 * t);
    result += 5e-10 * std::cos(5.57636570536 + 6290.1893969922 * t);
    result += 6.1e-10 * std::cos(2.24359003264 + 8635.9420037632 * t);
    result += 5e-10 * std::cos(5.54441900966 + 1990.745017041 * t);
    result += 5.6e-10 * std::cos(4.0030107804 + 13367.9726311066 * t);
    result += 5.2e-10 * std::cos(4.13138898038 + 7860.4193924392 * t);
    result += 5.2e-10 * std::cos(3.90943054011 + 26.2983197998 * t);
    result += 4.1e-10 * std::cos(3.5712848278 + 7079.3738568078 * t);
    result += 5.6e-10 * std::cos(2.76959005761 + 90955.5516944961 * t);
    result += 4.2e-10 * std::cos(1.91461189199 + 7477.522860216 * t);
    result += 4.2e-10 * std::cos(0.42728171713 + 10213.285546211 * t);
    result += 4.2e-10 * std::cos(1.09413724455 + 709.9330485583 * t);
    result += 3.9e-10 * std::cos(3.93298068961 + 10973.55568635 * t);
    result += 3.8e-10 * std::cos(6.17935925345 + 9917.6968745098 * t);
    result += 4.9e-10 * std::cos(0.83021145241 + 11506.7697697936 * t);
    result += 5.3e-10 * std::cos(1.45828359397 + 233141.314404362 * t);
    result += 4.7e-10 * std::cos(6.21568666789 + 6681.2248533996 * t);
    result += 3.7e-10 * std::cos(0.3635930998 + 10177.2576795336 * t);
    result += 3.5e-10 * std::cos(3.33024911524 + 5643.1785636774 * t);
    result += 3.4e-10 * std::cos(5.63446915337 + 6525.8044539654 * t);
    result += 3.5e-10 * std::cos(5.36033855038 + 25158.6017197654 * t);
    result += 3.4e-10 * std::cos(5.36319798321 + 4933.2084403326 * t);
    result += 3.3e-10 * std::cos(4.24722336872 + 12569.6748183318 * t);
    result += 4.3e-10 * std::cos(5.26370903404 + 10575.4066829418 * t);
    result += 4.2e-10 * std::cos(5.08837645072 + 11015.1064773348 * t);
    result += 4e-10 * std::cos(1.98334703186 + 6284.0561710596 * t);
    result += 4.2e-10 * std::cos(4.22496037505 + 88860.0570709867 * t);
    result += 2.9e-10 * std::cos(3.1908862817 + 11926.2544136688 * t);
    result += 2.9e-10 * std::cos(0.15217616684 + 12168.0026965746 * t);
    result += 3e-10 * std::cos(1.61904744136 + 9779.1086761254 * t);
    result += 2.7e-10 * std::cos(0.76388991416 + 1589.0728952838 * t);
    result += 3.6e-10 * std::cos(2.74712003443 + 3738.761430108 * t);
    result += 3.3e-10 * std::cos(3.08807829566 + 3930.2096962196 * t);
    result += 3.1e-10 * std::cos(5.34906619513 + 143571.324284816 * t);
    result += 2.5e-10 * std::cos(0.10240267494 + 22483.8485744926 * t);
    result += 3e-10 * std::cos(3.47110495524 + 14945.3161735544 * t);
    result += 2.4e-10 * std::cos(1.10425016019 + 4535.0594369244 * t);
    result += 2.4e-10 * std::cos(1.5803725978 + 6496.3749454294 * t);
    result += 2.3e-10 * std::cos(3.87710321433 + 6275.9623029906 * t);
    result += 2.5e-10 * std::cos(3.9452977897 + 3128.3887650958 * t);
    result += 2.3e-10 * std::cos(3.44685609601 + 4136.9104335162 * t);
    result += 2.3e-10 * std::cos(3.83156029849 + 5753.3848848968 * t);
    result += 2.2e-10 * std::cos(1.86956128067 + 16730.4636895958 * t);
    result += 2.5e-10 * std::cos(2.42188933855 + 5729.506447149 * t);
    result += 2e-10 * std::cos(1.78208352927 + 17789.845619785 * t);
    result += 2.1e-10 * std::cos(4.303630874 + 16858.4825329332 * t);
    result += 2.1e-10 * std::cos(0.49258939822 + 29088.811415985 * t);
    result += 2.5e-10 * std::cos(1.33030250444 + 6282.0955289232 * t);
    result += 2.7e-10 * std::cos(2.54785812264 + 3496.032826134 * t);
    result += 2.2e-10 * std::cos(1.1123252195 + 12721.572099417 * t);
    result += 2.1e-10 * std::cos(5.97759081637 + 7.1135470008 * t);
    result += 1.9e-10 * std::cos(0.80292033311 + 16062.1845261168 * t);
    result += 2.3e-10 * std::cos(4.12454848769 + 2388.8940204492 * t);
    result += 2.2e-10 * std::cos(4.92663152168 + 18875.525869774 * t);
    result += 2.3e-10 * std::cos(5.68902059771 + 16460.333529525 * t);
    result += 2.3e-10 * std::cos(4.97346265647 + 17260.1546546904 * t);
    result += 2.3e-10 * std::cos(3.03021283729 + 66567.4858652543 * t);
    result += 1.6e-10 * std::cos(3.89740925257 + 5331.3574437408 * t);
    result += 1.7e-10 * std::cos(3.08268671348 + 154717.609887683 * t);
    result += 1.6e-10 * std::cos(3.95085099736 + 3097.88382272579 * t);
    result += 1.6e-10 * std::cos(3.99041783945 + 6283.14316029419 * t);
    result += 2e-10 * std::cos(6.10644140189 + 167283.761587666 * t);
    result += 1.5e-10 * std::cos(4.09775914607 + 11712.9553182308 * t);
    result += 1.6e-10 * std::cos(5.717699407 + 17298.1823273262 * t);
    result += 1.6e-10 * std::cos(3.28894009404 + 5884.9268465832 * t);
    result += 1.5e-10 * std::cos(5.64785377164 + 12559.038152982 * t);
    result += 1.6e-10 * std::cos(4.4345208093 + 6283.0085396886 * t);
    result += 1.4e-10 * std::cos(2.31721603062 + 5481.2549188676 * t);
    result += 1.4e-10 * std::cos(4.43479032305 + 13517.8701062334 * t);
    result += 1.4e-10 * std::cos(4.73209312936 + 7342.4577801806 * t);
    result += 1.2e-10 * std::cos(0.64705975463 + 18073.7049386502 * t);
    result += 1.1e-10 * std::cos(1.514433322 + 16200.7727245012 * t);
    result += 1.1e-10 * std::cos(0.88708889185 + 21228.3920235458 * t);
    result += 1.4e-10 * std::cos(4.50116508534 + 640.8776073822 * t);
    result += 1.1e-10 * std::cos(4.64339996198 + 11790.6290886588 * t);
    result += 1.1e-10 * std::cos(1.31064298246 + 4164.311989613 * t);
    result += 9e-11 * std::cos(3.02238989305 + 23543.2305046818 * t);
    result += 9e-11 * std::cos(2.04999402381 + 22003.9146348698 * t);
    result += 9e-11 * std::cos(4.91488110218 + 213.299095438 * t);
    return result;
}

constexpr double getEarthR3(double t)
{
    double result = 0.0;
    result += 1.44595e-06 * std::cos(4.27319435148 + 6283.0758499914 * t);
    result += 6.729e-08 * std::cos(3.91697608662 + 12566.1516999828 * t);
    result += 7.74e-09;
    result += 2.47e-09 * std::cos(3.73019298781 + 18849.2275499742 * t);
    result += 3.6e-10 * std::cos(2.8008140905 + 6286.5989683404 * t);
    result += 3.3e-10 * std::cos(5.62216602775 + 6127.6554505572 * t);
    result += 1.9e-10 * std::cos(3.71292621802 + 6438.4962494256 * t);
    result += 1.6e-10 * std::cos(4.26011484232 + 6525.8044539654 * t);
    result += 1.6e-10 * std::cos(3.50416887054 + 6256.7775301916 * t);
    result += 1.4e-10 * std::cos(3.62127621114 + 25132.3033999656 * t);
    result += 1.1e-10 * std::cos(4.39200958819 + 4705.7323075436 * t);
    result += 1.1e-10 * std::cos(5.22327127059 + 6040.3472460174 * t);
    result += 1e-10 * std::cos(4.28045254647 + 83996.8473181119 * t);
    result += 9e-11 * std::cos(1.56864096494 + 5507.5532386674 * t);
    result += 1.1e-10 * std::cos(1.37795688024 + 6309.3741697912 * t);
    result += 1e-10 * std::cos(5.19937959068 + 71430.6956181291 * t);
    result += 9e-11 * std::cos(0.4727519993 + 6279.5527316424 * t);
    result += 9e-11 * std::cos(0.74642756529 + 5729.506447149 * t);
    result += 7e-11 * std::cos(2.9737489156 + 775.522611324 * t);
    result += 7e-11 * std::cos(3.28615691021 + 7058.5984613154 * t);
    result += 7e-11 * std::cos(2.19184402142 + 6812.766815086 * t);
    result += 5e-11 * std::cos(3.15419034438 + 529.6909650946 * t);
    result += 6e-11 * std::cos(4.54725567047 + 1059.3819301892 * t);
    result += 5e-11 * std::cos(1.51104406936 + 7079.3738568078 * t);
    result += 7e-11 * std::cos(2.98052059053 + 6681.2248533996 * t);
    result += 5e-11 * std::cos(2.30961231391 + 12036.4607348882 * t);
    result += 5e-11 * std::cos(3.71102966917 + 6290.1893969922 * t);
    return result;
}

constexpr double getEarthR4(double t)
{
    double result = 0.0;
    result += 3.858e-08 * std::cos(2.56384387339 + 6283.0758499914 * t);
    result += 3.06e-09 * std::cos(2.2676950123 + 12566.1516999828 * t);
    result += 5.3e-10 * std::cos(3.44031471924 + 5573.1428014331 * t);
    result += 1.5e-10 * std::cos(2.04794573436 + 18849.2275499742 * t);
    result += 1.3e-10 * std::cos(2.05688873673 + 77713.7714681205 * t);
    result += 7e-11 * std::cos(4.4121885448 + 161000.685737674 * t);
    result += 5e-11 * std::cos(5.26154653107 + 6438.4962494256 * t);
    result += 5e-11 * std::cos(4.07695126049 + 6127.6554505572 * t);
    result += 6e-11 * std::cos(3.81514213664 + 149854.400134808 * t);
    result += 3e-11 * std::cos(1.28175749811 + 6286.5989683404 * t);
    return result;
}

constexpr double getEarthR5(double t)
{
    double result = 0.0;
    result += 8.6e-10 * std::cos(1.21579741687 + 6283.0758499914 * t);
    result += 1.2e-10 * std::cos(0.65617264033 + 12566.1516999828 * t);
    result += 1e-11 * std::cos(0.38068797142 + 18849.2275499742 * t);
    return result;
}

constexpr double getJulianThousandYears(double jd)
{
    constexpr double DaysOf1000Years = 365.25 * 1000;
    return (jd - J2000) / DaysOf1000Years;
}

constexpr double mod2Pi(double r)
{
    while (r < 0) {
        r += std::numbers::pi * 2;
    }
    while (r > 2 * std::numbers::pi) {
        r -= std::numbers::pi * 2;
    }
    return r;
}

/**
 * Calculates the Heliocentric Yellow Meridian (rad) of the Earth by Julian day
 */
constexpr double getSunEclipticLongitudeForEarth(double julianDay)
{
    const double t = getJulianThousandYears(julianDay);
    const double L0 = getEarthL0(t);
    const double L1 = getEarthL1(t);
    const double L2 = getEarthL2(t);
    const double L3 = getEarthL3(t);
    const double L4 = getEarthL4(t);
    const double L5 = getEarthL5(t);
    const double L = ((((L5 * t + L4) * t + L3) * t + L2) * t + L1) * t + L0;
    return mod2Pi(L);
}

constexpr double getSunEclipticLatitudeForEarth(double jd)
{
    const double t = getJulianThousandYears(jd);
    const double B0 = getEarthB0(t);
    const double B1 = getEarthB1(t);
    const double B2 = getEarthB2(t);
    const double B3 = getEarthB3(t);
    const double B4 = getEarthB4(t);
    const double B = ((((B4 * t) + B3) * t + B2) * t + B1) * t + B0;
    return B;
}

constexpr double getJulianCentury(double julianDay)
{
    // days of 100 years
    constexpr double DaysOfCentury = 365.25 * 100;
    return (julianDay - J2000) / DaysOfCentury;
}

constexpr double secondsToDegrees(double seconds)
{
    return seconds / 3600;
}

constexpr double degreesToRadians(double degrees)
{
    return degrees * std::numbers::pi / 180;
}

constexpr double secondsToRadians(double seconds)
{
    return degreesToRadians(secondsToDegrees(seconds));
}

consteval double coefficient()
{
    return secondsToRadians(0.0001);
}

void getEarthNutationParameter(EarthNutationParameter &earthNutationParameter, double T)
{
    const double T2 = T * T;
    const double T3 = T2 * T;

    earthNutationParameter.D = degreesToRadians(297.85036 + 445267.111480 * T - 0.0019142 * T2 + T3 / 189474.0);
    earthNutationParameter.M = degreesToRadians(357.52772 + 35999.050340 * T - 0.0001603 * T2 - T3 / 300000.0);
    earthNutationParameter.Mp = degreesToRadians(134.96298 + 477198.867398 * T + 0.0086972 * T2 + T3 / 56250.0);
    earthNutationParameter.F = degreesToRadians(93.27191 + 483202.017538 * T - 0.0036825 * T2 + T3 / 327270.0);
    earthNutationParameter.Omega = degreesToRadians(125.04452 - 1934.136261 * T + 0.0020708 * T2 + T3 / 450000.0);
}

double calcEarthLongitudeNutation(double T)
{
    EarthNutationParameter radian;
    getEarthNutationParameter(radian, T);
    double result = 0.0;
    for (std::size_t i = 0; i < s_nuation.size(); i++) {
        double theta =
            s_nuation[i].D * radian.D + s_nuation[i].M * radian.M + s_nuation[i].Mp * radian.Mp + s_nuation[i].F * radian.F + s_nuation[i].Omega * radian.Omega;
        result += (s_nuation[i].Sine1 + s_nuation[i].Sine2 * T) * std::sin(theta);
    }
    return result * coefficient();
}

constexpr double Vsop2Fk5LongitudeCorrection(double latitude, double longitude, double julianDay)
{
    const double t = getJulianCentury(julianDay);
    const double lp = latitude - degreesToRadians(1.397) * t - degreesToRadians(0.00031) * t * t;
    return secondsToRadians(-0.09033 + 0.03916 * (std::cos(lp) + std::sin(lp)) * std::tan(longitude));
}

/**
 * Calculates ecliptical longitude of earth in heliocentric coordinates based on VSOP87D table
 * @see https://ftp.imcce.fr/pub/ephem/planets/vsop87/VSOP87D.ear
 * @return earth longitude in radians
 */
constexpr double getSunRadiusForEarth(double julianDay)
{
    const double t = getJulianThousandYears(julianDay);
    const double R0 = getEarthR0(t);
    const double R1 = getEarthR1(t);
    const double R2 = getEarthR2(t);
    const double R3 = getEarthR3(t);
    const double R4 = getEarthR4(t);
    const double R5 = getEarthR5(t);
    const double R = ((((R5 * t + R4) * t + R3) * t + R2) * t + R1) * t + R0;
    return R;
}

double getEarthEclipticLongitudeForSun(double jd)
{
    double l = getSunEclipticLongitudeForEarth(jd);
    const double b = getSunEclipticLatitudeForEarth(jd);
    l += calcEarthLongitudeNutation(getJulianCentury(jd));
    l += Vsop2Fk5LongitudeCorrection(l, b, jd);
    l = mod2Pi(l + std::numbers::pi);
    // Light-time correction depends upon the velocity and distance of the emitting object during the time it takes for its light to travel to Earth.
    // 20.49522″ = arctan(earth revolution linear velocity / view distance between earth and sun)
    // FIXME: why divided by r
    const double r = getSunRadiusForEarth(jd);
    l -= secondsToRadians(20.4898) / r;
    return l;
}

void getMoonEclipticParameter(MoonEclipticParameter &moonEclipticParameter, double T)
{
    double T2 = T * T;
    double T3 = T2 * T;
    double T4 = T3 * T;

    moonEclipticParameter.Lp = mod2Pi(degreesToRadians(218.3164591 + 481267.88134236 * T - 0.0013268 * T2 + T3 / 538841.0 - T4 / 65194000.0));
    moonEclipticParameter.D = mod2Pi(degreesToRadians(297.8502042 + 445267.1115168 * T - 0.0016300 * T2 + T3 / 545868.0 - T4 / 113065000.0));
    moonEclipticParameter.M = mod2Pi(degreesToRadians(357.5291092 + 35999.0502909 * T - 0.0001536 * T2 + T3 / 24490000.0));
    moonEclipticParameter.Mp = mod2Pi(degreesToRadians(134.9634114 + 477198.8676313 * T + 0.0089970 * T2 + T3 / 69699.0 - T4 / 14712000.0));
    moonEclipticParameter.F = mod2Pi(degreesToRadians(93.2720993 + 483202.0175273 * T - 0.0034029 * T2 - T3 / 3526000.0 + T4 / 863310000.0));
    moonEclipticParameter.E = 1 - 0.002516 * T - 0.0000074 * T2;
}

double calcMoonECLongitudePeriodic(MoonEclipticParameter &moonEclipticParameter)
{
    double EI = 0.0;
    for (std::size_t i = 0; i < s_moonLongitude.size(); i++) {
        double theta = s_moonLongitude[i].D * moonEclipticParameter.D + s_moonLongitude[i].M * moonEclipticParameter.M
            + s_moonLongitude[i].Mp * moonEclipticParameter.Mp + s_moonLongitude[i].F * moonEclipticParameter.F;
        EI += s_moonLongitude[i].EiA * std::sin(theta) * std::pow(moonEclipticParameter.E, std::abs(s_moonLongitude[i].M));
    }
    return EI;
}

double calcMoonLongitudePerturbation(double T, const MoonEclipticParameter &moonEclipticParameter)
{
    const double A1 = mod2Pi(degreesToRadians(119.75 + 131.849 * T));
    const double A2 = mod2Pi(degreesToRadians(53.09 + 479264.290 * T));

    return 3958.0 * std::sin(A1) + 1962.0 * std::sin(moonEclipticParameter.Lp - moonEclipticParameter.F) + 318.0 * std::sin(A2);
}

double getMoonEclipticLongitudeEC(double julianDay)
{
    MoonEclipticParameter radian;
    double T = getJulianCentury(julianDay);
    getMoonEclipticParameter(radian, T);
    double EI = calcMoonECLongitudePeriodic(radian);
    EI += calcMoonLongitudePerturbation(T, radian);
    double longitude = radian.Lp + degreesToRadians(EI / 1000000.0);
    longitude += calcEarthLongitudeNutation(T);
    return longitude;
}

double NewtonIteration(double angle, double x0)
{
    constexpr double EPSILON = 1e-7;
    constexpr double DELTA = 5e-6;
    // Make sure the rad value is in between -PI and PI
    auto func = [angle](double x) -> double {
        double r = getEarthEclipticLongitudeForSun(x) - angle;
        while (r < -std::numbers::pi) {
            r += std::numbers::pi * 2;
        }
        while (r > std::numbers::pi) {
            r -= std::numbers::pi * 2;
        }
        return r;
    };

    double x;
    unsigned count = 0;
    while (count++ < 100u) {
        x = x0;
        double fx, fpx;
        fx = func(x);
        // derivative
        fpx = (func(x + DELTA) - func(x - DELTA)) / DELTA / 2;
        x0 = x - fx / fpx;
        if (std::abs(x0 - x) <= EPSILON) {
            break;
        }
    }
    return x;
}

// From http://eclipse.gsfc.nasa.gov/SEhelp/deltatpoly2004.html
double getDeltaT(int year, int month)
{
    double y = double(year) + (double(month) - 0.5) / 12;

    if (year < -500) {
        double u = (double(year) - 1820) / 100;
        return -20 + 32 * u * u;
    } else if (year < 500) {
        double u = y / 100;
        double u2 = u * u;
        double u3 = u2 * u;
        double u4 = u3 * u;
        double u5 = u4 * u;
        double u6 = u5 * u;
        return 10583.6 - 1014.41 * u + 33.78311 * u2 - 5.952053 * u3 - 0.1798452 * u4 + 0.022174192 * u5 + 0.0090316521 * u6;
    } else if (year < 1600) {
        double u = (y - 1000) / 100;
        double u2 = u * u;
        double u3 = u2 * u;
        double u4 = u3 * u;
        double u5 = u4 * u;
        double u6 = u5 * u;
        return 1574.2 - 556.01 * u + 71.23472 * u2 + 0.319781 * u3 - 0.8503463 * u4 - 0.005050998 * u5 + 0.0083572073 * u6;
    } else if (year < 1700) {
        double t = y - 1600;
        double t2 = t * t;
        double t3 = t2 * t;
        return 120 - 0.9808 * t - 0.01532 * t2 + t3 / 7129;
    } else if (year < 1800) {
        double t = y - 1700;
        double t2 = t * t;
        double t3 = t2 * t;
        double t4 = t3 * t;
        return 8.83 + 0.1603 * t - 0.0059285 * t2 + 0.00013336 * t3 - t4 / 1174000;
    } else if (year < 1860) {
        double t = y - 1800;
        double t2 = t * t;
        double t3 = t2 * t;
        double t4 = t3 * t;
        double t5 = t4 * t;
        double t6 = t5 * t;
        double t7 = t6 * t;
        return 13.72 - 0.332447 * t + 0.0068612 * t2 + 0.0041116 * t3 - 0.00037436 * t4 + 0.0000121272 * t5 - 0.0000001699 * t6 + 0.000000000875 * t7;
    } else if (year < 1900) {
        double t = y - 1860;
        double t2 = t * t;
        double t3 = t2 * t;
        double t4 = t3 * t;
        double t5 = t4 * t;
        return 7.62 + 0.5737 * t - 0.251754 * t2 + 0.01680668 * t3 - 0.0004473624 * t4 + t5 / 233174;
    } else if (year < 1920) {
        double t = y - 1900;
        double t2 = t * t;
        double t3 = t2 * t;
        double t4 = t3 * t;
        return -2.79 + 1.494119 * t - 0.0598939 * t2 + 0.0061966 * t3 - 0.000197 * t4;
    } else if (year < 1941) {
        double t = y - 1920;
        double t2 = t * t;
        double t3 = t2 * t;
        return 21.20 + 0.84493 * t - 0.076100 * t2 + 0.0020936 * t3;
    } else if (year < 1961) {
        double t = y - 1950;
        double t2 = t * t;
        double t3 = t2 * t;
        return 29.07 + 0.407 * t - t2 / 233 + t3 / 2547;
    } else if (year < 1986) {
        double t = y - 1975;
        double t2 = t * t;
        double t3 = t2 * t;
        return 45.45 + 1.067 * t - t2 / 260 - t3 / 718;
    } else if (year < 2005) {
        double t = y - 2000;
        double t2 = t * t;
        double t3 = t2 * t;
        double t4 = t3 * t;
        double t5 = t4 * t;
        return 63.86 + 0.3345 * t - 0.060374 * t2 + 0.0017275 * t3 + 0.000651814 * t4 + 0.00002373599 * t5;
    } else if (year < 2050) {
        double t = y - 2000;
        double t2 = t * t;
        return 62.92 + 0.32217 * t + 0.005589 * t2;
    } else if (year < 2150) {
        double u = (y - 1820) / 100;
        double u2 = u * u;
        return -20 + 32 * u2 - 0.5628 * (2150 - y);
    } else {
        double u = (y - 1820) / 100;
        double u2 = u * u;
        return -20 + 32 * u2;
    }
}

int64_t toJulianDay(int year, int month, int day)
{
#if __has_cpp_attribute(assume)
    [[assume(year > 0 && day > 0)]];
    [[assume(month > 0 && month <= 12)]];
#endif
    int a = (14 - month) / 12;
    int y = year + 4800 - a;
    int m = month + 12 * a - 3;
    return day + (153 * m + 2) / 5 + 365 * y + y / 4 - y / 100 + y / 400 - 32045;
}

void getDateFromJulianDay(double julianDay, int &yy, int &mm, int &dd)
{
    /*
     * This algorithm is taken from
     * "Numerical Recipes in c, 2nd Ed." (1992), pp. 14-15
     * and converted to integer math.
     * The electronic version of the book is freely available
     * at http://www.nr.com/ , under "Obsolete Versions - Older
     * book and code versions.
     */
    constexpr int64_t JD_GREG_CAL = 2299161;
    constexpr int64_t JB_MAX_WITHOUT_OVERFLOW = 107374182;
    int64_t julian = int64_t(std::floor(julianDay + 0.5));

    int64_t ta, jalpha, tb, tc, td, te;

    if (julian >= JD_GREG_CAL) {
        jalpha = (4 * (julian - 1867216) - 1) / 146097;
        ta = julian + 1 + jalpha - jalpha / 4;
    } else if (julian < 0) {
        ta = julian + 36525 * (1 - julian / 36525);
    } else {
        ta = julian;
    }

    tb = ta + 1524;
    if (tb <= JB_MAX_WITHOUT_OVERFLOW) {
        tc = (tb * 20 - 2442) / 7305;
    } else {
        tc = int64_t((uint64_t(tb) * 20 - 2442) / 7305);
    }

    td = 365 * tc + tc / 4;
    te = ((tb - td) * 10000) / 306001;
    dd = int(tb - td - (306001 * te) / 10000);
    mm = int(te - 1);

    if (mm > 12) {
        mm -= 12;
    }
    yy = int(tc - 4715);
    if (mm > 2) {
        yy--;
    }
    if (julian < 0) {
        yy -= int(100 * (1 - julian / 36525));
    }
}
}