/*
 * // Copyright (c) Radzivon Bartoshyk 9/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::polyeval::f_polyeval4;
use crate::sin_cosf::ArgumentReducer;

// Generated in SageMath:
// print("[")
// for k in range(64):
//     k = RealField(150)(k) * RealField(150).pi() / RealField(150)(32)
//     print(double_to_hex(k.tan()) + ",")
// print("];")
pub(crate) static TAN_K_PI_OVER32: [u64; 64] = [
    0x0000000000000000,
    0x3fb936bb8c5b2da2,
    0x3fc975f5e0553158,
    0x3fd36a08355c63dc,
    0x3fda827999fcef32,
    0x3fe11ab7190834ec,
    0x3fe561b82ab7f990,
    0x3fea43002ae42850,
    0x3ff0000000000000,
    0x3ff37efd8d87607e,
    0x3ff7f218e25a7461,
    0x3ffdef13b73c1406,
    0x4003504f333f9de6,
    0x400a5f59e90600dd,
    0x40141bfee2424771,
    0x40244e6c595afdcc,
    0xc950457bf6be49c7,
    0xc0244e6c595afdcc,
    0xc0141bfee2424771,
    0xc00a5f59e90600dd,
    0xc003504f333f9de6,
    0xbffdef13b73c1406,
    0xbff7f218e25a7461,
    0xbff37efd8d87607e,
    0xbff0000000000000,
    0xbfea43002ae42850,
    0xbfe561b82ab7f990,
    0xbfe11ab7190834ec,
    0xbfda827999fcef32,
    0xbfd36a08355c63dc,
    0xbfc975f5e0553158,
    0xbfb936bb8c5b2da2,
    0x369f77598338bfdf,
    0x3fb936bb8c5b2da2,
    0x3fc975f5e0553158,
    0x3fd36a08355c63dc,
    0x3fda827999fcef32,
    0x3fe11ab7190834ec,
    0x3fe561b82ab7f990,
    0x3fea43002ae42850,
    0x3ff0000000000000,
    0x3ff37efd8d87607e,
    0x3ff7f218e25a7461,
    0x3ffdef13b73c1406,
    0x4003504f333f9de6,
    0x400a5f59e90600dd,
    0x40141bfee2424771,
    0x40244e6c595afdcc,
    0xc935b1fa9e530d0a,
    0xc0244e6c595afdcc,
    0xc0141bfee2424771,
    0xc00a5f59e90600dd,
    0xc003504f333f9de6,
    0xbffdef13b73c1406,
    0xbff7f218e25a7461,
    0xbff37efd8d87607e,
    0xbff0000000000000,
    0xbfea43002ae42850,
    0xbfe561b82ab7f990,
    0xbfe11ab7190834ec,
    0xbfda827999fcef32,
    0xbfd36a08355c63dc,
    0xbfc975f5e0553158,
    0xbfb936bb8c5b2da2,
];

pub(crate) struct TanfEval {
    pub(crate) tan_k: f64,
    pub(crate) tan_y: f64,
}

#[inline]
pub(crate) fn tanpif_eval(y: f64, k: i64) -> TanfEval {
    let y_sqr = y * y;

    // After range reduction, k = round(x * 32 / pi) and y = (x * 32 / pi) - k.
    // So k is an integer and -0.5 <= y <= 0.5.

    // picking minus sin and cos according to quadrant
    let tan_k = f64::from_bits(TAN_K_PI_OVER32[((k as u64) & 63) as usize]);

    // tan(x*pi/32) generated by Sollya:
    // d = [0, 0.5];
    // f_tan = tan(y*pi/32)/y;
    // Q = fpminimax(f_tan, [|0, 2, 4, 6|], [|D...|], d, relative, floating);
    // See ./notes/tanpif.sollya
    let tan_y = f_polyeval4(
        y_sqr,
        f64::from_bits(0x3fb921fb54442cef),
        f64::from_bits(0x3f34abbce63a363e),
        f64::from_bits(0x3eb466baced705e8),
        f64::from_bits(0x3e346a33cde88184),
    ) * y;
    TanfEval { tan_y, tan_k }
}

#[inline(always)]
#[allow(unused)]
pub(crate) fn tanpif_eval_fma(y: f64, k: i64) -> TanfEval {
    let y_sqr = y * y;

    // After range reduction, k = round(x * 32 / pi) and y = (x * 32 / pi) - k.
    // So k is an integer and -0.5 <= y <= 0.5.

    // picking minus sin and cos according to quadrant
    let tan_k = f64::from_bits(TAN_K_PI_OVER32[((k as u64) & 63) as usize]);

    // tan(x*pi/32) generated by Sollya:
    // d = [0, 0.5];
    // f_tan = tan(y*pi/32)/y;
    // Q = fpminimax(f_tan, [|0, 2, 4, 6|], [|D...|], d, relative, floating);
    // See ./notes/tanpif.sollya
    use crate::polyeval::d_polyeval4;
    let tan_y = d_polyeval4(
        y_sqr,
        f64::from_bits(0x3fb921fb54442cef),
        f64::from_bits(0x3f34abbce63a363e),
        f64::from_bits(0x3eb466baced705e8),
        f64::from_bits(0x3e346a33cde88184),
    ) * y;
    TanfEval { tan_y, tan_k }
}

#[inline(always)]
pub(crate) fn tanf_eval(x: f64, x_abs: u32) -> TanfEval {
    let (y, k) = ArgumentReducer { x, x_abs }.reduce();
    let y_sqr = y * y;

    // After range reduction, k = round(x * 32 / pi) and y = (x * 32 / pi) - k.
    // So k is an integer and -0.5 <= y <= 0.5.

    // picking minus sin and cos according to quadrant
    let tan_k = f64::from_bits(TAN_K_PI_OVER32[(k & 63) as usize]);

    // tan(x*pi/32) generated by Sollya:
    // d = [0, 0.5];
    // f_tan = tan(x*pi/32)/x;
    // Q = fpminimax(f_tan, [|0, 2, 4, 6|], [|D...|], d);
    // See ./notes/tanf.sollya
    let tan_y = f_polyeval4(
        y_sqr,
        f64::from_bits(0x3fb921fb54442cef),
        f64::from_bits(0x3f34abbce63a363e),
        f64::from_bits(0x3eb466baced705e8),
        f64::from_bits(0x3e346a33cde88184),
    ) * y;
    TanfEval { tan_y, tan_k }
}

#[inline(always)]
#[allow(unused)]
pub(crate) fn tanf_eval_fma(x: f64, x_abs: u32) -> TanfEval {
    let (y, k) = ArgumentReducer { x, x_abs }.reduce_fma();
    let y_sqr = y * y;

    // After range reduction, k = round(x * 32 / pi) and y = (x * 32 / pi) - k.
    // So k is an integer and -0.5 <= y <= 0.5.

    // picking minus sin and cos according to quadrant
    let tan_k = f64::from_bits(TAN_K_PI_OVER32[(k & 63) as usize]);

    // tan(x*pi/32) generated by Sollya:
    // d = [0, 0.5];
    // f_tan = tan(x*pi/32)/x;
    // Q = fpminimax(f_tan, [|0, 2, 4, 6|], [|D...|], d);
    // See ./notes/tanf.sollya
    use crate::polyeval::d_polyeval4;
    let tan_y = d_polyeval4(
        y_sqr,
        f64::from_bits(0x3fb921fb54442cef),
        f64::from_bits(0x3f34abbce63a363e),
        f64::from_bits(0x3eb466baced705e8),
        f64::from_bits(0x3e346a33cde88184),
    ) * y;
    TanfEval { tan_y, tan_k }
}
