% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mfx.R
\name{tidy.betamfx}
\alias{tidy.betamfx}
\title{Tidy a(n) betamfx object}
\usage{
\method{tidy}{betamfx}(x, conf.int = FALSE, conf.level = 0.95, ...)
}
\arguments{
\item{x}{A \code{betamfx} object.}

\item{conf.int}{Logical indicating whether or not to include a confidence
interval in the tidied output. Defaults to \code{FALSE}.}

\item{conf.level}{The confidence level to use for the confidence interval
if \code{conf.int = TRUE}. Must be strictly greater than 0 and less than 1.
Defaults to 0.95, which corresponds to a 95 percent confidence interval.}

\item{...}{Additional arguments. Not used. Needed to match generic
signature only. \strong{Cautionary note:} Misspelled arguments will be
absorbed in \code{...}, where they will be ignored. If the misspelled
argument has a default value, the default value will be used.
For example, if you pass \code{conf.lvel = 0.9}, all computation will
proceed using \code{conf.level = 0.95}. Two exceptions here are:
\itemize{
\item \code{tidy()} methods will warn when supplied an \code{exponentiate} argument if
it will be ignored.
\item \code{augment()} methods will warn when supplied a \code{newdata} argument if it
will be ignored.
}}
}
\description{
Tidy summarizes information about the components of a model.
A model component might be a single term in a regression, a single
hypothesis, a cluster, or a class. Exactly what tidy considers to be a
model component varies across models but is usually self-evident.
If a model has several distinct types of components, you will need to
specify which components to return.
}
\details{
The \code{mfx} package provides methods for calculating marginal effects
for various generalized linear models (GLMs). Unlike standard linear
models, estimated model coefficients in a GLM cannot be directly
interpreted as marginal effects (i.e., the change in the response variable
predicted after a one unit change in one of the regressors). This is
because the estimated coefficients are multiplicative, dependent on both
the link function that was used for the estimation and any other variables
that were included in the model. When calculating marginal effects, users
must typically choose whether they want to use i) the average observation
in the data, or ii) the average of the sample marginal effects. See
\code{vignette("mfxarticle")} from the \code{mfx} package for more details.
}
\examples{
\dontshow{if (rlang::is_installed("mfx")) withAutoprint(\{ # examplesIf}

library(mfx)

# Simulate some data
set.seed(12345)
n <- 1000
x <- rnorm(n)

# Beta outcome
y <- rbeta(n, shape1 = plogis(1 + 0.5 * x), shape2 = (abs(0.2 * x)))
# Use Smithson and Verkuilen correction
y <- (y * (n - 1) + 0.5) / n

d <- data.frame(y, x)
mod_betamfx <- betamfx(y ~ x | x, data = d)

tidy(mod_betamfx, conf.int = TRUE)

# Compare with the naive model coefficients of the equivalent betareg call (not run)
# tidy(betamfx(y ~ x | x, data = d), conf.int = TRUE)

augment(mod_betamfx)
glance(mod_betamfx)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=tidy.betareg]{tidy.betareg()}}, \code{\link[mfx:betamfx]{mfx::betamfx()}}

Other mfx tidiers: 
\code{\link{augment.betamfx}()},
\code{\link{augment.mfx}()},
\code{\link{glance.betamfx}()},
\code{\link{glance.mfx}()},
\code{\link{tidy.mfx}()}
}
\concept{mfx tidiers}
\value{
A \code{\link[tibble:tibble]{tibble::tibble()}} with columns:
  \item{conf.high}{Upper bound on the confidence interval for the estimate.}
  \item{conf.low}{Lower bound on the confidence interval for the estimate.}
  \item{estimate}{The estimated value of the regression term.}
  \item{p.value}{The two-sided p-value associated with the observed statistic.}
  \item{statistic}{The value of a T-statistic to use in a hypothesis that the regression term is non-zero.}
  \item{std.error}{The standard error of the regression term.}
  \item{term}{The name of the regression term.}
  \item{atmean}{TRUE if the marginal effects were originally calculated as the
    partial effects for the average observation. If FALSE, then these were
    instead calculated as average partial effects.}

}
