/*
 * Copyright (C) 2025 Michael Brown <mbrown@fensystems.co.uk>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * You can also choose to distribute this program under the terms of
 * the Unmodified Binary Distribution Licence (as given in the file
 * COPYING.UBDL), provided that you have satisfied its requirements.
 */

FILE_LICENCE ( GPL2_OR_LATER_OR_UBDL );

/** @file
 *
 * NIST P-256 elliptic curve self-tests
 *
 */

/* Forcibly enable assertions */
#undef NDEBUG

#include <ipxe/p256.h>
#include <ipxe/test.h>
#include "elliptic_test.h"

/* http://point-at-infinity.org/ecc/nisttv k=1 */
ELLIPTIC_TEST ( poi_1, &p256_curve, BASE_GENERATOR,
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 ),
		EXPECTED ( 0x6b, 0x17, 0xd1, 0xf2, 0xe1, 0x2c, 0x42, 0x47,
			   0xf8, 0xbc, 0xe6, 0xe5, 0x63, 0xa4, 0x40, 0xf2,
			   0x77, 0x03, 0x7d, 0x81, 0x2d, 0xeb, 0x33, 0xa0,
			   0xf4, 0xa1, 0x39, 0x45, 0xd8, 0x98, 0xc2, 0x96,
			   0x4f, 0xe3, 0x42, 0xe2, 0xfe, 0x1a, 0x7f, 0x9b,
			   0x8e, 0xe7, 0xeb, 0x4a, 0x7c, 0x0f, 0x9e, 0x16,
			   0x2b, 0xce, 0x33, 0x57, 0x6b, 0x31, 0x5e, 0xce,
			   0xcb, 0xb6, 0x40, 0x68, 0x37, 0xbf, 0x51, 0xf5 ) );

/* http://point-at-infinity.org/ecc/nisttv k=2 */
ELLIPTIC_TEST ( poi_2, &p256_curve, BASE_GENERATOR,
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02 ),
		EXPECTED ( 0x7c, 0xf2, 0x7b, 0x18, 0x8d, 0x03, 0x4f, 0x7e,
			   0x8a, 0x52, 0x38, 0x03, 0x04, 0xb5, 0x1a, 0xc3,
			   0xc0, 0x89, 0x69, 0xe2, 0x77, 0xf2, 0x1b, 0x35,
			   0xa6, 0x0b, 0x48, 0xfc, 0x47, 0x66, 0x99, 0x78,
			   0x07, 0x77, 0x55, 0x10, 0xdb, 0x8e, 0xd0, 0x40,
			   0x29, 0x3d, 0x9a, 0xc6, 0x9f, 0x74, 0x30, 0xdb,
			   0xba, 0x7d, 0xad, 0xe6, 0x3c, 0xe9, 0x82, 0x29,
			   0x9e, 0x04, 0xb7, 0x9d, 0x22, 0x78, 0x73, 0xd1 ) );

/* http://point-at-infinity.org/ecc/nisttv k=2 (as base) to k=20 */
ELLIPTIC_TEST ( poi_2_20, &p256_curve,
		BASE ( 0x7c, 0xf2, 0x7b, 0x18, 0x8d, 0x03, 0x4f, 0x7e,
		       0x8a, 0x52, 0x38, 0x03, 0x04, 0xb5, 0x1a, 0xc3,
		       0xc0, 0x89, 0x69, 0xe2, 0x77, 0xf2, 0x1b, 0x35,
		       0xa6, 0x0b, 0x48, 0xfc, 0x47, 0x66, 0x99, 0x78,
		       0x07, 0x77, 0x55, 0x10, 0xdb, 0x8e, 0xd0, 0x40,
		       0x29, 0x3d, 0x9a, 0xc6, 0x9f, 0x74, 0x30, 0xdb,
		       0xba, 0x7d, 0xad, 0xe6, 0x3c, 0xe9, 0x82, 0x29,
		       0x9e, 0x04, 0xb7, 0x9d, 0x22, 0x78, 0x73, 0xd1 ),
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a ),
		EXPECTED ( 0x83, 0xa0, 0x1a, 0x93, 0x78, 0x39, 0x5b, 0xab,
			   0x9b, 0xcd, 0x6a, 0x0a, 0xd0, 0x3c, 0xc5, 0x6d,
			   0x56, 0xe6, 0xb1, 0x92, 0x50, 0x46, 0x5a, 0x94,
			   0xa2, 0x34, 0xdc, 0x4c, 0x6b, 0x28, 0xda, 0x9a,
			   0x76, 0xe4, 0x9b, 0x6d, 0xe2, 0xf7, 0x32, 0x34,
			   0xae, 0x6a, 0x5e, 0xb9, 0xd6, 0x12, 0xb7, 0x5c,
			   0x9f, 0x22, 0x02, 0xbb, 0x69, 0x23, 0xf5, 0x4f,
			   0xf8, 0x24, 0x0a, 0xaa, 0x86, 0xf6, 0x40, 0xb8 ) );

/* http://point-at-infinity.org/ecc/nisttv k=112233445566778899 */
ELLIPTIC_TEST ( poi_mid, &p256_curve, BASE_GENERATOR,
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x01, 0x8e, 0xbb, 0xb9, 0x5e, 0xed, 0x0e, 0x13 ),
		EXPECTED ( 0x33, 0x91, 0x50, 0x84, 0x4e, 0xc1, 0x52, 0x34,
			   0x80, 0x7f, 0xe8, 0x62, 0xa8, 0x6b, 0xe7, 0x79,
			   0x77, 0xdb, 0xfb, 0x3a, 0xe3, 0xd9, 0x6f, 0x4c,
			   0x22, 0x79, 0x55, 0x13, 0xae, 0xaa, 0xb8, 0x2f,
			   0xb1, 0xc1, 0x4d, 0xdf, 0xdc, 0x8e, 0xc1, 0xb2,
			   0x58, 0x3f, 0x51, 0xe8, 0x5a, 0x5e, 0xb3, 0xa1,
			   0x55, 0x84, 0x0f, 0x20, 0x34, 0x73, 0x0e, 0x9b,
			   0x5a, 0xda, 0x38, 0xb6, 0x74, 0x33, 0x6a, 0x21 ) );

/* http://point-at-infinity.org/ecc/nisttv k=<largest> */
ELLIPTIC_TEST ( poi_large, &p256_curve, BASE_GENERATOR,
		SCALAR ( 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
			 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			 0xbc, 0xe6, 0xfa, 0xad, 0xa7, 0x17, 0x9e, 0x84,
			 0xf3, 0xb9, 0xca, 0xc2, 0xfc, 0x63, 0x25, 0x50 ),
		EXPECTED ( 0x6b, 0x17, 0xd1, 0xf2, 0xe1, 0x2c, 0x42, 0x47,
			   0xf8, 0xbc, 0xe6, 0xe5, 0x63, 0xa4, 0x40, 0xf2,
			   0x77, 0x03, 0x7d, 0x81, 0x2d, 0xeb, 0x33, 0xa0,
			   0xf4, 0xa1, 0x39, 0x45, 0xd8, 0x98, 0xc2, 0x96,
			   0xb0, 0x1c, 0xbd, 0x1c, 0x01, 0xe5, 0x80, 0x65,
			   0x71, 0x18, 0x14, 0xb5, 0x83, 0xf0, 0x61, 0xe9,
			   0xd4, 0x31, 0xcc, 0xa9, 0x94, 0xce, 0xa1, 0x31,
			   0x34, 0x49, 0xbf, 0x97, 0xc8, 0x40, 0xae, 0x0a ) );

/* Invalid curve point zero */
ELLIPTIC_TEST ( invalid_zero, &p256_curve,
		BASE ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		       0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ),
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 ),
		EXPECTED_FAIL );

/* Invalid curve point (base_x, base_y - 1) */
ELLIPTIC_TEST ( invalid_one, &p256_curve,
		BASE ( 0x6b, 0x17, 0xd1, 0xf2, 0xe1, 0x2c, 0x42, 0x47,
		       0xf8, 0xbc, 0xe6, 0xe5, 0x63, 0xa4, 0x40, 0xf2,
		       0x77, 0x03, 0x7d, 0x81, 0x2d, 0xeb, 0x33, 0xa0,
		       0xf4, 0xa1, 0x39, 0x45, 0xd8, 0x98, 0xc2, 0x96,
		       0x4f, 0xe3, 0x42, 0xe2, 0xfe, 0x1a, 0x7f, 0x9b,
		       0x8e, 0xe7, 0xeb, 0x4a, 0x7c, 0x0f, 0x9e, 0x16,
		       0x2b, 0xce, 0x33, 0x57, 0x6b, 0x31, 0x5e, 0xce,
		       0xcb, 0xb6, 0x40, 0x68, 0x37, 0xbf, 0x51, 0xf4 ),
		SCALAR ( 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 ),
		EXPECTED_FAIL );

/**
 * Perform P-256 self-test
 *
 */
static void p256_test_exec ( void ) {

	/* Tests from http://point-at-infinity.org/ecc/nisttv */
	elliptic_ok ( &poi_1 );
	elliptic_ok ( &poi_2 );
	elliptic_ok ( &poi_2_20 );
	elliptic_ok ( &poi_mid );
	elliptic_ok ( &poi_large );

	/* Invalid point tests */
	elliptic_ok ( &invalid_zero );
	elliptic_ok ( &invalid_one );
}

/** P-256 self-test */
struct self_test p256_test __self_test = {
	.name = "p256",
	.exec = p256_test_exec,
};
