/* SPDX-License-Identifier: GPL-2.0-or-later */

#include <device/device.h>
#include <device/pnp.h>
#include <pc80/keyboard.h>
#include <superio/conf_mode.h>
#include <superio/ite/common/env_ctrl.h>

#include "chip.h"
#include "it8613e.h"

static void it8613e_init(struct device *dev)
{
	const struct superio_ite_it8613e_config *conf = dev->chip_info;
	const struct resource *res;

	if (!dev->enabled)
		return;

	switch (dev->path.pnp.device) {
	case IT8613E_EC:
		res = probe_resource(dev, PNP_IDX_IO0);
		if (!conf || !res)
			break;
		ite_ec_init(res->base, &conf->ec);
		break;
	case IT8613E_KBCK:
		pc_keyboard_init(NO_AUX_DEVICE);
		break;
	case IT8613E_KBCM:
		break;
	}
}

static struct device_operations ops = {
	.read_resources		= pnp_read_resources,
	.set_resources		= pnp_set_resources,
	.enable_resources	= pnp_enable_resources,
	.enable				= pnp_alt_enable,
	.init				= it8613e_init,
	.ops_pnp_mode		= &pnp_conf_mode_870155_aa,
};

static struct pnp_info pnp_dev_info[] = {
	/* Serial Port 1 */
	{ NULL, IT8613E_SP1, PNP_IO0 | PNP_IRQ0 | PNP_MSC0, 0x0ff8, },
	/* Environmental Controller */
	{ NULL, IT8613E_EC, PNP_IO0 | PNP_IO1 | PNP_IRQ0, 0x0ff8, 0x0ff8, },
	/* KBC Keyboard */
	{ NULL, IT8613E_KBCK, PNP_IO0 | PNP_IO1 | PNP_IRQ0 | PNP_MSC0,
		0x0fff, 0x0fff, },
	/* KBC Mouse */
	{ NULL, IT8613E_KBCM, PNP_IRQ0 | PNP_MSC0, },
	/* GPIO */
	{ NULL, IT8613E_GPIO, PNP_IO0 | PNP_IO1 | PNP_IRQ0, 0x0ffc, 0x0ff8, },
	/* Consumer Infrared */
	{ NULL, IT8613E_CIR, PNP_IO0 | PNP_IRQ0 | PNP_MSC0, 0x0ff8, },
};

static void enable_dev(struct device *dev)
{
	pnp_enable_devices(dev, &ops, ARRAY_SIZE(pnp_dev_info), pnp_dev_info);
}

struct chip_operations superio_ite_it8613e_ops = {
	.name = "ITE IT8613E Super I/O",
	.enable_dev = enable_dev,
};
