#!/usr/bin/python

# ----------------------------------------------------------------------------
#
#  Copyright (C) 2008-2022 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


# Combine JackIECfilt with some PyQt6 code to create
# a 1/3 octave band spectrum analyser.


import sys
import signal
import globdef
from PyQt6 import QtGui, QtCore, QtWidgets
from zita_jacktools.jackiecfilt import *
from zita_jacktools.jackkmeter import *
sys.path.append ('..')
from utils.kmeters import *


# Window with 31 K-meters.
#
class Meterwin(QtWidgets.QWidget):
    
    def __init__(self, metrics):
        super(Meterwin, self).__init__()
        self.metrics = metrics
        self.disp = K20meter(self, metrics, (31,), 6, 4)
        self.disp.move(0, 0)
        self.disp.show()
        self.setFixedSize (self.disp.sx, self.disp.sy)
        self.setWindowTitle('1/3 octave band spectrum')
        self.show()        
        self.timer = QtCore.QBasicTimer ()
        self.timer.start (50, self)

    def timerEvent (self, ev):
        st, rms, dpk = K20.get_levels ()
        self.disp.rms = rms
        self.disp.pks = dpk
        self.disp.update ()

# Create JackIECfilt with 1 input and 31 outputs.        
IEC = JackIECfilt (1, 31, 'Filters')

# Create 31 K-meter processors. 
K20 = JackKmeter (31, 'Meters')

# Connect filter input.
IEC.connect_input (0, 'system:capture_1')

# Define filters, and connect to K-meter inputs.
for i in range (31):
    IEC.set_filter (0, i, JackIECfilt.OCT3, i)
    IEC.connect_output (i, 'Meters:in_%d' % (i,))


# Start PyQt6 app.
#
signal.signal (signal.SIGINT, signal.SIG_DFL)
app = QtWidgets.QApplication (sys.argv)
globdef.darkpal (app)

# Create meter window.
metrics = makek20metrics (globdef.COL0)
win = Meterwin (metrics)
sys.exit(app.exec())


