/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.smtpserver;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import jakarta.inject.Inject;
import jakarta.mail.Address;
import jakarta.mail.MessagingException;
import jakarta.mail.internet.AddressException;
import jakarta.mail.internet.InternetAddress;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.apache.commons.configuration2.Configuration;
import org.apache.james.core.Domain;
import org.apache.james.core.MailAddress;
import org.apache.james.core.MaybeSender;
import org.apache.james.jdkim.api.PublicKeyRecordRetriever;
import org.apache.james.jdkim.api.SignatureRecord;
import org.apache.james.jdkim.exceptions.FailException;
import org.apache.james.jdkim.mailets.DKIMVerifier;
import org.apache.james.protocols.smtp.SMTPSession;
import org.apache.james.protocols.smtp.hook.HookResult;
import org.apache.james.protocols.smtp.hook.HookReturnCode;
import org.apache.james.smtpserver.JamesMessageHook;
import org.apache.james.util.StreamUtils;
import org.apache.mailet.Mail;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DKIMHook
implements JamesMessageHook {
    private static final Logger LOGGER = LoggerFactory.getLogger(DKIMHook.class);
    @VisibleForTesting
    private final DKIMVerifier verifier;
    private Config config;
    private SignatureRecordValidation signatureRecordValidation;
    private DKIMCheckNeeded dkimCheckNeeded;

    @Inject
    public DKIMHook(PublicKeyRecordRetriever publicKeyRecordRetriever) {
        this.verifier = new DKIMVerifier(publicKeyRecordRetriever);
    }

    public void init(Configuration configuration) {
        this.config = Config.parse(configuration);
        this.dkimCheckNeeded = this.config.dkimCheckNeeded();
        this.signatureRecordValidation = this.config.signatureRecordValidation();
    }

    public HookResult onMessage(SMTPSession session, Mail mail) {
        if (session.isRelayingAllowed()) {
            return HookResult.DECLINED;
        }
        if (!this.dkimCheckNeeded.test(mail)) {
            return HookResult.DECLINED;
        }
        try {
            return this.signatureRecordValidation.validate(mail.getMaybeSender(), mail.getMessage().getMessageID(), this.verifier.verify(mail.getMessage(), this.config.forceCRLF));
        }
        catch (MessagingException e) {
            LOGGER.warn("Error while verifying DKIM signatures", (Throwable)e);
            return HookResult.builder().hookReturnCode(HookReturnCode.denySoft()).smtpReturnCode("451").smtpDescription("Failure computing DKIM signature.").build();
        }
        catch (FailException e) {
            LOGGER.warn("DKIM check failed. Invalid signature.", (Throwable)e);
            return HookResult.builder().hookReturnCode(HookReturnCode.deny()).smtpReturnCode("530").smtpDescription("DKIM check failed. Invalid signature.").build();
        }
    }

    public static class Config {
        public static final ImmutableList<ValidatedEntity> DEFAULT_VALIDATED_ENTITIES = ImmutableList.of((Object)((Object)ValidatedEntity.envelope), (Object)((Object)ValidatedEntity.headers));
        private final boolean forceCRLF;
        private final boolean signatureRequired;
        private final Optional<List<Domain>> onlyForSenderDomain;
        private final ImmutableList<ValidatedEntity> validatedEntities;
        private final Optional<List<String>> expectedDToken;

        public static Config parse(Configuration config) {
            return new Config(config.getBoolean("forceCRLF", true), config.getBoolean("signatureRequired", true), Optional.ofNullable(config.getString("onlyForSenderDomain", null)).map(s -> Splitter.on((char)',').splitToStream((CharSequence)s).map(Domain::of).toList()), Optional.ofNullable(config.getString("validatedEntities", null)).map(entities -> (ImmutableList)Stream.of(entities.split(",")).map(ValidatedEntity::from).collect(ImmutableList.toImmutableList())).orElse(DEFAULT_VALIDATED_ENTITIES), Optional.ofNullable(config.getString("expectedDToken", null)).map(s -> Splitter.on((char)',').splitToList((CharSequence)s)));
        }

        public Config(boolean forceCRLF, boolean signatureRequired, Optional<List<Domain>> onlyForSenderDomain, ImmutableList<ValidatedEntity> validatedEntities, Optional<List<String>> expectedDToken) {
            this.forceCRLF = forceCRLF;
            this.signatureRequired = signatureRequired;
            this.onlyForSenderDomain = onlyForSenderDomain;
            this.validatedEntities = validatedEntities;
            this.expectedDToken = expectedDToken;
        }

        DKIMCheckNeeded dkimCheckNeeded() {
            return this.onlyForSenderDomain.map(domains -> DKIMCheckNeeded.or(domains.stream().map(this::computeDKIMChecksNeeded).flatMap(Collection::stream).toList())).orElse(DKIMCheckNeeded.ALL);
        }

        SignatureRecordValidation signatureRecordValidation() {
            return SignatureRecordValidation.and(SignatureRecordValidation.signatureRequired(this.signatureRequired), this.expectedDToken.map(DTokenSignatureRecordValidation::new).orElse(SignatureRecordValidation.ALLOW_ALL));
        }

        private ImmutableList<DKIMCheckNeeded> computeDKIMChecksNeeded(Domain domain) {
            return (ImmutableList)this.validatedEntities.stream().map(entity -> this.toDKIMCheck((ValidatedEntity)((Object)entity), domain)).collect(ImmutableList.toImmutableList());
        }

        private DKIMCheckNeeded toDKIMCheck(ValidatedEntity entity, Domain domain) {
            return switch (entity.ordinal()) {
                default -> throw new MatchException(null, null);
                case 0 -> DKIMCheckNeeded.onlyForSenderDomain(domain);
                case 1 -> DKIMCheckNeeded.onlyForHeaderFromDomain(domain);
            };
        }

        public final boolean equals(Object o) {
            if (o instanceof Config) {
                Config config = (Config)o;
                return this.forceCRLF == config.forceCRLF && this.signatureRequired == config.signatureRequired && Objects.equals(this.onlyForSenderDomain, config.onlyForSenderDomain) && Objects.equals(this.validatedEntities, config.validatedEntities) && Objects.equals(this.expectedDToken, config.expectedDToken);
            }
            return false;
        }

        public final int hashCode() {
            return Objects.hash(this.forceCRLF, this.signatureRequired, this.onlyForSenderDomain, this.validatedEntities, this.expectedDToken);
        }

        public static enum ValidatedEntity {
            envelope,
            headers;


            static ValidatedEntity from(String rawValue) {
                Preconditions.checkNotNull((Object)rawValue);
                return Stream.of(ValidatedEntity.values()).filter(entity -> entity.name().equalsIgnoreCase(rawValue)).findAny().orElseThrow(() -> new IllegalArgumentException(String.format("invalid validated entity '%s'", rawValue)));
            }
        }
    }

    @FunctionalInterface
    static interface DKIMCheckNeeded
    extends Predicate<Mail> {
        public static final DKIMCheckNeeded ALL = any -> true;
        public static final DKIMCheckNeeded NONE = any -> false;

        public static DKIMCheckNeeded or(List<DKIMCheckNeeded> checkNeededs) {
            return mail -> checkNeededs.stream().anyMatch(predicate -> predicate.test(mail));
        }

        public static DKIMCheckNeeded onlyForSenderDomain(Domain domain) {
            return mail -> mail.getMaybeSender().asOptional().map(MailAddress::getDomain).map(arg_0 -> ((Domain)domain).equals(arg_0)).orElse(false);
        }

        public static DKIMCheckNeeded onlyForHeaderFromDomain(Domain domain) {
            return mail -> {
                try {
                    return StreamUtils.ofNullable((Object[])mail.getMessage().getFrom()).distinct().flatMap(DKIMCheckNeeded::parseMailAddress).findFirst().map(MailAddress::getDomain).map(arg_0 -> ((Domain)domain).equals(arg_0)).orElse(false);
                }
                catch (MessagingException me) {
                    try {
                        LOGGER.info("Unable to parse the \"FROM\" header {}; ignoring.", (Object[])mail.getMessage().getHeader("From"));
                    }
                    catch (MessagingException e) {
                        LOGGER.info("Unable to parse the \"FROM\" header; ignoring.");
                    }
                    return false;
                }
            };
        }

        private static Stream<MailAddress> parseMailAddress(Address from) {
            if (from instanceof InternetAddress) {
                InternetAddress internetAddress = (InternetAddress)from;
                try {
                    return Stream.of(new MailAddress(internetAddress.getAddress()));
                }
                catch (AddressException e) {
                    throw new RuntimeException(e);
                }
            }
            return Stream.empty();
        }
    }

    @FunctionalInterface
    static interface SignatureRecordValidation {
        public static final SignatureRecordValidation ALLOW_ALL = (sender, messageId, records) -> HookResult.DECLINED;

        public static SignatureRecordValidation and(SignatureRecordValidation a, SignatureRecordValidation b) {
            return (sender, messageId, records) -> {
                HookResult hookResult = a.validate(sender, messageId, records);
                if (hookResult.equals((Object)HookResult.DECLINED)) {
                    return b.validate(sender, messageId, records);
                }
                return hookResult;
            };
        }

        public static SignatureRecordValidation signatureRequired(boolean required) {
            return (sender, messageId, records) -> {
                if (required && (records == null || records.isEmpty())) {
                    LOGGER.warn("DKIM check failed for {}. Expecting DKIM signatures. Got none.", (Object)messageId);
                    return HookResult.builder().hookReturnCode(HookReturnCode.deny()).smtpReturnCode("530").smtpDescription("DKIM check failed. Expecting DKIM signatures. Got none.").build();
                }
                return HookResult.DECLINED;
            };
        }

        public HookResult validate(MaybeSender var1, String var2, List<SignatureRecord> var3);
    }

    static class DTokenSignatureRecordValidation
    implements SignatureRecordValidation {
        private final List<String> expectedDTokens;

        DTokenSignatureRecordValidation(List<String> expectedDTokens) {
            this.expectedDTokens = expectedDTokens;
        }

        DTokenSignatureRecordValidation(String expectedDToken) {
            this((List<String>)ImmutableList.of((Object)expectedDToken));
        }

        @Override
        public HookResult validate(MaybeSender maybeSender, String messageId, List<SignatureRecord> records) {
            if (records.stream().anyMatch(r -> this.expectedDTokens.stream().anyMatch(d -> r.getDToken().equals(d)))) {
                return HookResult.DECLINED;
            }
            ImmutableSet actualDToken = (ImmutableSet)records.stream().map(SignatureRecord::getDToken).collect(ImmutableSet.toImmutableSet());
            LOGGER.warn("DKIM check failed for {}. Wrong d token. Expecting {}. Got {}.", new Object[]{messageId, this.expectedDTokens, actualDToken});
            return HookResult.builder().hookReturnCode(HookReturnCode.deny()).smtpReturnCode("530").smtpDescription("DKIM check failed. Wrong d token. Expecting " + String.valueOf(this.expectedDTokens) + ". Got " + String.valueOf(actualDToken)).build();
        }
    }
}

